/**
 * Common Tables JavaScript
 * Handles table functionality for list views
 */
document.addEventListener('DOMContentLoaded', function() {
  
  // Jednostavniji selektori - sada tražimo samo jedan ID
  const dataTable = document.getElementById('dataTable');
  const dataSearch = document.getElementById('dataSearch') || document.getElementById('requestSearch') || document.getElementById('tableSearch');
  const exportToggle = document.getElementById('exportToggle');
  const exportPanel = document.getElementById('exportPanel');
  const closeExport = document.getElementById('closeExport');
  const deleteButtons = document.querySelectorAll('.action-btn.delete');
  const deleteModal = document.getElementById('deleteModal');
  const deleteForm = document.getElementById('deleteForm');
  const deleteId = document.getElementById('deleteId') || document.getElementById('deleteUserId');
  const deleteName = document.getElementById('deleteName');
  const modalClose = document.querySelector('.modal-close');
  const modalCancel = document.querySelector('.modal-cancel');
  
  // Initialize table functionality
  initTable();
  
  // Initialize export panel if exists
  if (exportToggle && exportPanel) {
    initExportPanel();
  }
  
  // Initialize delete confirmation
  initDeleteConfirmation();
  
  // Initialize filters
  initFilterSystem();
  
  /**
   * Initialize table functionality
   */
  function initTable() {
    if (!dataTable) {
      console.warn('Cannot initialize table: No table element found');
      return;
    }
    
    // Add data-label attributes for responsive display
    addResponsiveLabels();
    
    // Add search functionality
    if (dataSearch) {
      addSearchFunctionality();
    }
    
    // Add sorting functionality
    addSortingFunctionality();
    
    // Add row highlighting animation
    animateTableLoad();
  }
  
  /**
   * Add data-label attributes for responsive display
   */
  function addResponsiveLabels() {
    const headers = dataTable.querySelectorAll('thead th');
    const rows = dataTable.querySelectorAll('tbody tr');
    
    rows.forEach(row => {
      const cells = row.querySelectorAll('td');
      cells.forEach((cell, index) => {
        if (headers[index]) {
          cell.setAttribute('data-label', headers[index].textContent.trim());
        }
      });
    });
  }
  
  /**
   * Add search functionality to the table
   */
  function addSearchFunctionality() {
    dataSearch.addEventListener('input', function() {
      const searchTerm = this.value.toLowerCase();
      const rows = dataTable.querySelectorAll('tbody tr');
      
      rows.forEach(row => {
        const text = row.textContent.toLowerCase();
        if (text.includes(searchTerm)) {
          row.style.display = '';
        } else {
          row.style.display = 'none';
        }
      });
    });
  }
  
  /**
   * Add sorting functionality to the table
   */
  function addSortingFunctionality() {
    // Target all headers that are sortable OR all headers if no sortable class exists
    const allHeaders = dataTable.querySelectorAll('thead th');
    const specificSortableHeaders = dataTable.querySelectorAll('thead th.sortable, thead th[class*="sorting"]');
    
    // If there are specific sortable headers, use those. Otherwise, make all headers sortable
    const headers = specificSortableHeaders.length > 0 ? specificSortableHeaders : allHeaders;
    
    if (headers.length === 0) {
      console.warn('No headers found in table');
      return;
    }
    
    // Track current sort state
    let currentSortColumn = null;
    let currentSortDirection = true; // true = ascending, false = descending
    
    headers.forEach((header, index) => {
      header.style.cursor = 'pointer';
      header.setAttribute('data-sort-index', index);
      
      // Add initial indicator if it doesn't exist
      if (!header.querySelector('.sort-indicator')) {
        const indicator = document.createElement('span');
        indicator.className = 'sort-indicator';
        indicator.innerHTML = ' ↕';
        indicator.style.opacity = '0.5';
        indicator.style.fontSize = '0.8em';
        indicator.style.marginLeft = '5px';
        header.appendChild(indicator);
      }
      
      header.addEventListener('click', function() {
        const headerIndex = parseInt(this.getAttribute('data-sort-index'));
        console.log('Sort header clicked:', this.textContent.trim(), 'Index:', headerIndex);
        
        // Determine sort direction
        let isAscending;
        if (currentSortColumn === headerIndex) {
          // Same column clicked again, toggle direction
          isAscending = !currentSortDirection;
        } else {
          // New column, default to ascending
          isAscending = true;
        }
        
        // Update tracking variables
        currentSortColumn = headerIndex;
        currentSortDirection = isAscending;
        
        // Remove sorting classes from all headers
        headers.forEach(h => {
          h.classList.remove('sort-asc', 'sort-desc', 'sorting_asc', 'sorting_desc');
          
          // Reset indicator
          const indicator = h.querySelector('.sort-indicator');
          if (indicator) {
            indicator.innerHTML = ' ↕';
            indicator.style.opacity = '0.5';
          }
        });
        
        // Add appropriate sorting class to clicked header
        if (this.classList.contains('sorting')) {
          this.classList.add(isAscending ? 'sorting_asc' : 'sorting_desc');
        } else {
          this.classList.add(isAscending ? 'sort-asc' : 'sort-desc');
        }
        
        // Update indicator
        const indicator = this.querySelector('.sort-indicator');
        if (indicator) {
          indicator.innerHTML = isAscending ? ' ↑' : ' ↓';
          indicator.style.opacity = '1';
        }
        
        // Get the actual index in the table
        const actualColumnIndex = Array.from(dataTable.querySelectorAll('thead th')).indexOf(this);
        console.log('Sorting by column index:', actualColumnIndex, isAscending ? 'ascending' : 'descending');
        
        // Sort the table
        sortTable(actualColumnIndex, isAscending);
      });
    });
    
    // Initial sort on first sortable column if it exists
    if (headers.length > 0) {
      const firstSortable = headers[0];
      const firstSortableIndex = Array.from(dataTable.querySelectorAll('thead th')).indexOf(firstSortable);
      
      // Set initial sort state
      currentSortColumn = 0;
      currentSortDirection = true;
      
      if (firstSortable.classList.contains('sorting')) {
        firstSortable.classList.add('sorting_asc');
      } else {
        firstSortable.classList.add('sort-asc');
      }
      
      // Update indicator
      const indicator = firstSortable.querySelector('.sort-indicator');
      if (indicator) {
        indicator.innerHTML = ' ↑';
        indicator.style.opacity = '1';
      }
      
      sortTable(firstSortableIndex, true);
    }
  }
  
  /**
   * Check if a string represents a Croatian date format (dd.mm.yyyy or dd.mm.yyyy HH:MM)
   */
  function isHrDate(str) {
    // Check for dd.mm.yyyy with optional HH:MM
    return /^\d{1,2}\.\d{1,2}\.\d{4}(\s+\d{1,2}:\d{1,2})?$/.test(str);
  }

  /**
   * Sort table based on column index and direction
   */
  function sortTable(columnIndex, ascending) {
    const tbody = dataTable.querySelector('tbody');
    const rows = Array.from(tbody.querySelectorAll('tr'));

    // Sort the rows
    rows.sort((rowA, rowB) => {
      const cellsA = rowA.querySelectorAll('td');
      const cellsB = rowB.querySelectorAll('td');
      
      // Check if we have the necessary cells
      if (cellsA.length <= columnIndex || cellsB.length <= columnIndex) {
        console.warn('Column index out of bounds:', columnIndex, 'for row with', cellsA.length, 'cells');
        return 0;
      }
      
      const cellA = cellsA[columnIndex].textContent.trim();
      const cellB = cellsB[columnIndex].textContent.trim();
      
      // ======= DATE SPECIFIC HANDLING =======
      // If one is a date and the other isn't, handle special positioning
      const isDateA = isHrDate(cellA);
      const isDateB = isHrDate(cellB);
      
      if (isDateA && !isDateB) {
        // A is date, B is text
        // When ascending (oldest to newest), text comes before dates
        // When descending (newest to oldest), text comes after dates
        return ascending ? 1 : -1;
      }
      
      if (!isDateA && isDateB) {
        // A is text, B is date
        // When ascending (oldest to newest), text comes before dates
        // When descending (newest to oldest), text comes after dates
        return ascending ? -1 : 1;
      }
      
      // Both are dates, compare them normally
      if (isDateA && isDateB) {
        return compareCroatianDates(cellA, cellB, ascending);
      }
      
      // ======= NUMBER SPECIFIC HANDLING =======
      // Check if the values are numbers
      if (isNumeric(cellA) && isNumeric(cellB)) {
        const numA = parseFloat(cellA.replace(/[^\d.-]/g, ''));
        const numB = parseFloat(cellB.replace(/[^\d.-]/g, ''));
        return ascending ? numA - numB : numB - numA;
      }
      
      // ======= STATUS SPECIFIC HANDLING =======
      // Handle status text specially (active first, then inactive)
      if ((cellA.toLowerCase().includes('active') || cellA.toLowerCase().includes('inactive')) && 
          (cellB.toLowerCase().includes('active') || cellB.toLowerCase().includes('inactive'))) {
        const statusA = cellA.toLowerCase().includes('active') ? 1 : 0;
        const statusB = cellB.toLowerCase().includes('active') ? 1 : 0;
        return ascending ? statusB - statusA : statusA - statusB;
      }
      
      // ======= DEFAULT STRING HANDLING =======
      // Default string comparison
      return ascending 
        ? cellA.localeCompare(cellB, 'hr', { sensitivity: 'base' }) 
        : cellB.localeCompare(cellA, 'hr', { sensitivity: 'base' });
    });
    
    // Clear and re-append the sorted rows
    while (tbody.firstChild) {
      tbody.removeChild(tbody.firstChild);
    }
    
    rows.forEach(row => tbody.appendChild(row));
  }
  
  /**
   * Check if a string is numeric
   */
  function isNumeric(str) {
    // Remove formatting characters and check if it's a number
    return !isNaN(parseFloat(str.replace(/[^\d.-]/g, '')));
  }
  
  /**
   * Compare Croatian format dates (dd.mm.yyyy HH:MM) properly
   */
  function compareCroatianDates(dateStrA, dateStrB, ascending) {
    console.log('Comparing Croatian dates:', dateStrA, dateStrB);
    
    // Extract date parts
    const datePartsA = extractDateParts(dateStrA);
    const datePartsB = extractDateParts(dateStrB);
    
    // Compare year
    if (datePartsA.year !== datePartsB.year) {
      return ascending 
        ? datePartsA.year - datePartsB.year 
        : datePartsB.year - datePartsA.year;
    }
    
    // Compare month
    if (datePartsA.month !== datePartsB.month) {
      return ascending 
        ? datePartsA.month - datePartsB.month 
        : datePartsB.month - datePartsA.month;
    }
    
    // Compare day
    if (datePartsA.day !== datePartsB.day) {
      return ascending 
        ? datePartsA.day - datePartsB.day 
        : datePartsB.day - datePartsA.day;
    }
    
    // Compare hour
    if (datePartsA.hour !== datePartsB.hour) {
      return ascending 
        ? datePartsA.hour - datePartsB.hour 
        : datePartsB.hour - datePartsA.hour;
    }
    
    // Compare minute
    return ascending 
      ? datePartsA.minute - datePartsB.minute 
      : datePartsB.minute - datePartsA.minute;
  }
  
  /**
   * Extract date parts from Croatian date string
   */
  function extractDateParts(dateStr) {
    const parts = {};
    
    // Extract date components (dd.mm.yyyy)
    const dateMatch = dateStr.match(/^(\d{1,2})\.(\d{1,2})\.(\d{4})/);
    if (dateMatch) {
      parts.day = parseInt(dateMatch[1], 10);
      parts.month = parseInt(dateMatch[2], 10);
      parts.year = parseInt(dateMatch[3], 10);
    }
    
    // Extract time components if present (HH:MM)
    const timeMatch = dateStr.match(/(\d{1,2}):(\d{1,2})$/);
    if (timeMatch) {
      parts.hour = parseInt(timeMatch[1], 10);
      parts.minute = parseInt(timeMatch[2], 10);
    } else {
      // Default to 00:00 if no time specified
      parts.hour = 0;
      parts.minute = 0;
    }
    
    return parts;
  }
  
  /**
   * Initialize export panel functionality
   */
  function initExportPanel() {
    if (!exportToggle || !exportPanel || !closeExport) {
      console.warn('Export panel elements not found');
      return;
    }
    
    exportToggle.addEventListener('click', function() {
      exportPanel.classList.toggle('show');
    });
    
    closeExport.addEventListener('click', function() {
      exportPanel.classList.remove('show');
    });
    
    // Close panel when clicking outside
    document.addEventListener('click', function(event) {
      if (!exportPanel.contains(event.target) && 
          !exportToggle.contains(event.target) && 
          exportPanel.classList.contains('show')) {
        exportPanel.classList.remove('show');
      }
    });
  }
  
  /**
   * Initialize delete confirmation functionality
   */
  function initDeleteConfirmation() {
    if (!deleteModal) {
      return;
    }
    
    if (!deleteButtons.length) {
      console.warn('Delete buttons not found');
      return;
    }
    
    // Apply theme to the delete modal
    if (window.KrizanUI && typeof window.KrizanUI.applyModalTheme === 'function') {
      window.KrizanUI.applyModalTheme(deleteModal);
    }
    
    // Show modal when delete button is clicked
    deleteButtons.forEach(button => {
      button.addEventListener('click', function() {
        const id = this.getAttribute('data-id') || this.getAttribute('data-user-id');
        const name = this.getAttribute('data-name');
        
        console.log('Delete button clicked for ID:', id, 'Name:', name);
        
        if (deleteId) {
          deleteId.value = id;
        }
        
        if (deleteName) {
          deleteName.textContent = name || id;
        }
        
        deleteModal.classList.add('show');
      });
    });
    
    // Close modal when X is clicked
    if (modalClose) {
      modalClose.addEventListener('click', function() {
        deleteModal.classList.remove('show');
      });
    }
    
    // Close modal when Cancel is clicked
    if (modalCancel) {
      modalCancel.addEventListener('click', function() {
        deleteModal.classList.remove('show');
      });
    }
    
    // Close modal when clicking outside
    window.addEventListener('click', function(event) {
      if (event.target === deleteModal) {
        deleteModal.classList.remove('show');
      }
    });
  }
  
  /**
   * Initialize dynamic filtering system
   */
  function initFilterSystem() {
    
    // Get all filter checkboxes
    const filterCheckboxes = document.querySelectorAll('.filter-checkbox input');
    
    if (filterCheckboxes.length > 0) {
      // Add event listeners to all filter checkboxes
      filterCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', applyFilters);
      });
    }
  }
  
  /**
   * Apply filters to the table
   */
  function applyFilters() {
    if (!dataTable) return;
    
    const rows = dataTable.querySelectorAll('tbody tr');
    const activeFilters = {};
    const filterCheckboxes = document.querySelectorAll('.filter-checkbox input:checked');
    
    console.log('Applying filters, checked filters:', filterCheckboxes.length);
    
    // Build active filters map
    filterCheckboxes.forEach(checkbox => {
      const filterType = checkbox.getAttribute('data-filter');
      const filterValue = checkbox.getAttribute('data-value');
      
      if (!activeFilters[filterType]) {
        activeFilters[filterType] = [];
      }
      
      activeFilters[filterType].push(filterValue);
    });
    
    console.log('Active filters:', activeFilters);
    
    // Show all rows if no filters are active
    if (Object.keys(activeFilters).length === 0) {
      rows.forEach(row => {
        row.style.display = '';
      });
      return;
    }
    
    // Apply filtering
    let visibleCount = 0;
    
    rows.forEach(row => {
      let shouldShow = true;
      
      // Check each filter type
      for (const filterType in activeFilters) {
        let matches = false;
        
        if (filterType === 'status') {
          // Get the status badge from the row
          const statusBadge = row.querySelector('.status-badge');
          if (statusBadge) {
            const statusText = statusBadge.textContent.trim().toLowerCase();
            
            // Map Croatian status names if needed
            let mappedStatus = statusText;
            if (statusText.includes('new_token')) mappedStatus = 'active';
            else if (statusText.includes('used_token')) mappedStatus = 'used';
            else if (statusText.includes('expired_token')) mappedStatus = 'expired';
            
            matches = activeFilters[filterType].some(value => 
              mappedStatus.includes(value.toLowerCase())
            );
          }
        }
        else if (filterType === 'permission') {
          // Get all permission badges in the row
          const permBadges = row.querySelectorAll('.badge[data-permission]');
          if (permBadges.length > 0) {
            const rowPermissions = Array.from(permBadges).map(badge => 
              badge.getAttribute('data-permission') || badge.textContent.trim()
            );
            
            matches = activeFilters[filterType].some(permission => 
              rowPermissions.includes(permission)
            );
          }
        }
        
        if (!matches) {
          shouldShow = false;
          break;
        }
      }
      
      row.style.display = shouldShow ? '' : 'none';
      
      if (shouldShow) {
        visibleCount++;
      }
    });
    
    console.log('Filter applied, visible rows:', visibleCount);
  }
  
  /**
   * Add animation when table data loads
   */
  function animateTableLoad() {
    if (!dataTable) return;
    
    const rows = dataTable.querySelectorAll('tbody tr');
    
    // Add loading class to all rows
    rows.forEach(row => {
      row.classList.add('loading-row');
    });
    
    // Remove loading class with delay for animation effect
    rows.forEach((row, index) => {
      setTimeout(() => {
        row.classList.remove('loading-row');
      }, 100 + (index * 50));
    });
  }
})