/**
 * Password Reset Management JavaScript
 * Handles specific functionality for the password reset page
 * NOTE: This script should be loaded AFTER table.js
 */
document.addEventListener('DOMContentLoaded', function() {
  // Elements specific to password reset page
  const refreshButton = document.getElementById('refreshButton');
  const copyButtons = document.querySelectorAll('.action-btn.copy-link');
  const copyToast = document.getElementById('copyToast');
  const statusFilters = document.querySelectorAll('input[data-filter="status"]');
  
  // Initialize copy link functionality
  initCopyLink();
  
  // Initialize refresh button
  initRefreshButton();
  
  // Initialize status filters specifically for the password reset page
  initStatusFilters();
  
  /**
   * Initialize copy link functionality
   */
  function initCopyLink() {
    if (!copyButtons.length || !copyToast) {
      console.warn('Copy buttons or toast element not found');
      return;
    }
    
    copyButtons.forEach(button => {
      button.addEventListener('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        const resetLink = this.getAttribute('data-reset-link');
        if (!resetLink) {
          console.warn('No reset link found in data attribute');
          return;
        }
        
        // Copy to clipboard
        navigator.clipboard.writeText(resetLink)
          .then(() => {
            console.log('Text copied to clipboard');
            showToast();
          })
          .catch(err => {
            console.error('Could not copy text: ', err);
            
            // Fallback for browsers that don't support clipboard API
            fallbackCopyTextToClipboard(resetLink);
          });
      });
    });
    
    /**
     * Fallback copy method for older browsers
     */
    function fallbackCopyTextToClipboard(text) {
      const textArea = document.createElement('textarea');
      textArea.value = text;
      
      // Make the textarea out of viewport
      textArea.style.position = 'fixed';
      textArea.style.left = '-999999px';
      textArea.style.top = '-999999px';
      
      document.body.appendChild(textArea);
      textArea.focus();
      textArea.select();
      
      try {
        const successful = document.execCommand('copy');
        if (successful) {
          console.log('Fallback: Text copied to clipboard');
          showToast();
        } else {
          console.error('Fallback: Unable to copy');
        }
      } catch (err) {
        console.error('Fallback: Oops, unable to copy', err);
      }
      
      document.body.removeChild(textArea);
    }
    
    /**
     * Show success toast notification
     */
    function showToast() {
      copyToast.classList.add('show');
      
      // Start progress bar animation
      const progressBar = copyToast.querySelector('.toast-progress');
      if (progressBar) {
        progressBar.style.width = '0';
        
        // Animate progress bar
        setTimeout(() => {
          progressBar.style.width = '100%';
        }, 10);
      }
      
      // Hide toast after animation completes
      setTimeout(() => {
        copyToast.classList.remove('show');
      }, 3000);
    }
  }
  
  /**
   * Initialize refresh button
   */
  function initRefreshButton() {
    if (!refreshButton) {
      console.warn('Refresh button not found');
      return;
    }
    
    refreshButton.addEventListener('click', function() {
      // Add loading class to button
      this.classList.add('loading');
      console.log('Refresh button clicked, reloading page');
      
      // Add spinning animation
      const icon = this.querySelector('i');
      if (icon) {
        icon.classList.add('fa-spin');
      }
      
      // Reload the page after a short delay
      setTimeout(() => {
        window.location.reload();
      }, 300);
    });
  }
  
  /**
   * Initialize status filters specifically for password reset page
   */
  function initStatusFilters() {
    if (!statusFilters.length) {
      console.warn('Status filters not found');
      return;
    }
    
    // Map Croatian status names to filter values for easier matching
    const statusMap = {
      'new_token': 'active',
      'used_token': 'used',
      'expired_token': 'expired'
    };
    
    // Add custom translation for status filters
    statusFilters.forEach(filter => {
      const filterValue = filter.getAttribute('data-value');
      
      // Store original filter value for reference
      filter.setAttribute('data-original-value', filterValue);
      
      // We let table.js handle the actual filtering
      // This is just to enhance it with our translations
    });
  }
});