/**
 * User List JavaScript
 * Handles user management functionality
 */
document.addEventListener('DOMContentLoaded', function() {
    
    // Elements
    const deleteButtons = document.querySelectorAll('.action-btn.delete');
    const deleteModal = document.getElementById('deleteModal');
    const deleteForm = document.getElementById('deleteForm');
    const deleteNameSpan = document.getElementById('deleteName');
    const deleteIdInput = document.getElementById('deleteId');
    const modalCloseButtons = document.querySelectorAll('.modal-close, .modal-cancel');
    const exportToggle = document.getElementById('exportToggle');
    const exportPanel = document.getElementById('exportPanel');
    const closeExport = document.getElementById('closeExport');
    
    // Initialize delete functionality
    initDeleteFunctionality();
    
    // Initialize export functionality
    initExportFunctionality();
    
    /**
     * Initialize delete user functionality
     */
    function initDeleteFunctionality() {
        if (!deleteButtons.length || !deleteModal) return;
        
        deleteButtons.forEach(button => {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                
                const userId = this.getAttribute('data-id');
                const userName = this.getAttribute('data-name');
                
                if (!userId || !userName) {
                    console.error('Missing user ID or name for delete operation');
                    return;
                }
                
                // Set modal data
                if (deleteNameSpan) {
                    deleteNameSpan.textContent = userName;
                }
                
                if (deleteIdInput) {
                    deleteIdInput.value = userId;
                }
                
                // Show modal
                deleteModal.style.display = 'block';
                deleteModal.classList.add('show');
                
                // Focus on cancel button for better UX
                const cancelButton = deleteModal.querySelector('.modal-cancel');
                if (cancelButton) {
                    setTimeout(() => cancelButton.focus(), 100);
                }
            });
        });
        
        // Close modal functionality
        modalCloseButtons.forEach(button => {
            button.addEventListener('click', function() {
                closeDeleteModal();
            });
        });
        
        // Close modal on outside click
        deleteModal.addEventListener('click', function(e) {
            if (e.target === deleteModal) {
                closeDeleteModal();
            }
        });
        
        // Close modal on ESC key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && deleteModal.classList.contains('show')) {
                closeDeleteModal();
            }
        });
    }
    
    /**
     * Initialize export functionality
     */
    function initExportFunctionality() {
        if (!exportToggle || !exportPanel) return;
        
        exportToggle.addEventListener('click', function(e) {
            e.preventDefault();
            toggleExportPanel();
        });
        
        if (closeExport) {
            closeExport.addEventListener('click', function() {
                hideExportPanel();
            });
        }
        
        // Close export panel on outside click
        document.addEventListener('click', function(e) {
            if (exportPanel.style.display === 'block' && 
                !exportPanel.contains(e.target) && 
                !exportToggle.contains(e.target)) {
                hideExportPanel();
            }
        });
    }
    
    /**
     * Close delete modal
     */
    function closeDeleteModal() {
        if (deleteModal) {
            deleteModal.style.display = 'none';
            deleteModal.classList.remove('show');
            
            // Clear form data
            if (deleteIdInput) deleteIdInput.value = '';
            if (deleteNameSpan) deleteNameSpan.textContent = '';
        }
    }
    
    /**
     * Toggle export panel visibility
     */
    function toggleExportPanel() {
        if (exportPanel.style.display === 'block') {
            hideExportPanel();
        } else {
            showExportPanel();
        }
    }
    
    /**
     * Show export panel
     */
    function showExportPanel() {
        exportPanel.style.display = 'block';
        exportPanel.classList.add('show');
        
        // Focus on first form element
        const firstInput = exportPanel.querySelector('select, input');
        if (firstInput) {
            setTimeout(() => firstInput.focus(), 100);
        }
    }
    
    /**
     * Hide export panel
     */
    function hideExportPanel() {
        exportPanel.style.display = 'none';
        exportPanel.classList.remove('show');
    }
});