/**
 * DataTables Core Module - REFACTORED VERSION
 * Fokus SAMO na inicijalizaciju i orchestraciju servisa
 * Sve ostale odgovornosti premještene u odgovarajuće servise
 */

export class DataTablesCore {
  constructor(options = {}) {
    this.tableSelector = options.tableSelector || '#mojdatatables';
    this.services = options.services || {};
    this.config = options.config || {};
    this.routes = options.routes || {};
    
    this.table = null;
    this.columnMappings = null;
    this.tableConfig = null;
    this.initialData = [];
    this.uniqueValues = {};
    this.strategy = null;
    
    this.debug = options.debug || false;
    this.cascadingFiltersEnabled = options.cascadingFilters !== false;
    this.filterRefreshMode = options.filterRefreshMode || 'auto';
    
    // Bind main methods
    this.initialize = this.initialize.bind(this);
    this.setupTableEventHandlers = this.setupTableEventHandlers.bind(this);
  }
  
  /**
   * GLAVNA METODA: Inicijalizacija DataTable
   */
  async initialize(dataUrl) {
    const tableElement = document.querySelector(this.tableSelector);
    if (!tableElement) {
      console.error(`Table element not found: ${this.tableSelector}`);
      return null;
    }
    
    if (this.debug) {
      console.log('DataTablesCore: Initialization with URL', dataUrl);
    }
    
    try {
      // 1. Dohvati inicijalnu konfiguraciju
      const initData = await this.fetchInitialConfig(dataUrl);
      
      // 2. Procesuiraj konfiguraciju
      this.processConfiguration(initData);
      
      // 3. Generiraj table strukturu (delegiraj column servisu)
      this.generateTableStructure();
      
      // 4. Konfiguriraj filter servis
      this.configureFilterService();
      
      // 5. Use explicit strategy from configuration
      const strategy = this.config.strategy || 'client';
      if (strategy === 'server') {
        if (this.debug) console.log('Using server-side processing strategy');
        this.strategy = new ServerSideStrategy(this);
      } else {
        if (this.debug) console.log('Using client-side processing strategy');
        this.strategy = new ClientSideStrategy(this);
      }
      
      // 6. Inicijaliziraj strategiju
      return this.strategy.initialize(dataUrl);
      
    } catch (error) {
      console.error('Error initializing DataTable:', error);
      return this.showErrorTable('Error loading table');
    }
  }
  
  /**
   * Dohvati inicijalnu konfiguraciju s servera
   */
  async fetchInitialConfig(dataUrl) {
    return new Promise((resolve, reject) => {
      $.ajax({
        url: dataUrl,
        type: 'POST',
        data: {
          init: 'true',
          draw: 1,
          start: 0,
          length: 20,
          configOnly: true
        },
        success: (response) => {
          let parsedResponse = response;
          if (typeof response === 'string') {
            try {
              parsedResponse = JSON.parse(response);
            } catch (e) {
              console.error('Error parsing response string:', e);
              reject(e);
              return;
            }
          }
          resolve(parsedResponse);
        },
        error: (xhr, error, thrown) => {
          console.error('AJAX error:', error, thrown);
          reject(error);
        }
      });
    });
  }
  
  /**
   * Procesuiraj konfiguraciju podataka
   */
  processConfiguration(data) {
    if (!data || !data.config || !data.config.columns) {
      throw new Error('Invalid configuration data');
    }
    
    this.initialData = data.data || [];
    this.columnMappings = this.addIndexesToMapping(data.config.columns);
    
    this.tableConfig = {
      fixedColumns: data.config.fixedColumns || 0,
      routeName: data.config.routeName || '',
      routes: this.routes,
      rowStyling: data.config.rowStyling || null
    };
    
    if (data.uniqueValues) {
      this.uniqueValues = data.uniqueValues;
    }
    
    if (data.config.rowStyling) {
      this.initialData.config = this.initialData.config || {};
      this.initialData.config.rowStyling = data.config.rowStyling;
    }
    
    if (this.debug) {
      console.log('Row styling configuration:', data.config.rowStyling);
      console.log('Unique values loaded:', Object.keys(this.uniqueValues));
    }
    
    // Podijeli konfiguraciju sa servisima
    this.shareConfigurationWithServices();
  }
  
  /**
   * Podijeli konfiguraciju sa svim servisima
   */
  shareConfigurationWithServices() {
    Object.values(this.services).forEach(service => {
      if (typeof service.setTableConfig === 'function') {
        service.setTableConfig(this.tableConfig);
      }
      
      if (typeof service.setColumnMappings === 'function') {
        service.setColumnMappings(this.columnMappings);
      }
    });
  }
  
  /**
   * Generiraj HTML strukturu tablice (delegiraj column servisu)
   */
  generateTableStructure() {
    if (!this.services.column) {
      console.error('Column service not available');
      return;
    }
    
    const $table = $(this.tableSelector);
    $table.empty();
    
    const tableStructure = this.services.column.generateTableStructure(this.columnMappings);
    $table.append(tableStructure.thead);
    $table.append(tableStructure.tfoot);
  }
  
  /**
   * Konfiguriraj filter servis
   */
  configureFilterService() {
    if (!this.cascadingFiltersEnabled || !this.services.filter) {
      return;
    }
    
    if (this.services.filter.setRefreshMode) {
      const useServerSide = this.config.strategy === 'server';
      this.services.filter.setRefreshMode(useServerSide);
      
      if (this.debug) {
        console.log(`Filter refresh mode: ${useServerSide ? 'server' : 'client'}`);
      }
    }
  }
  
  /**
   * Pripremi osnovnu DataTables konfiguraciju
   */
  prepareBaseConfiguration() {
    if (!this.services.column || !this.services.i18n || !this.services.security) {
      throw new Error('Required services not available');
    }
    
    const columnUtil = this.services.column;
    const tableConfig = this.tableConfig;
    
    const excludedIndexes = [0];
    const excelColumns = columnUtil.excludeIndexes(excludedIndexes, this.columnMappings);
    const searchColumns = columnUtil.getIndexesForProperty('searchbox', 'yes', this.columnMappings);
    const dropdownColumns = columnUtil.getIndexesForProperty('dropdown', 'yes', this.columnMappings);
    const sumColumns = columnUtil.getIndexesForPropertyWithType('calculateinfoot', 'yes', this.columnMappings);
    const columnDefinitions = columnUtil.createColumnDefinitions(this.columnMappings);
    
    const csrfToken = this.services.security.getCSRFToken();
    const language = this.services.i18n.getLanguageOptions();
    
    return {
      processing: true,
      scrollX: true,
      scrollCollapse: true,
      fixedColumns: {
        left: tableConfig.fixedColumns || 0,
        leftColumns: tableConfig.fixedColumns || 0
      },
      stateSave: true,
      stateSaveCallback: (settings, data) => {
        localStorage.setItem('DataTables_' + tableConfig.routeName + '_' + settings.sTableId, JSON.stringify(data));
      },
      stateLoadCallback: (settings) => {
        return JSON.parse(localStorage.getItem('DataTables_' + tableConfig.routeName + '_' + settings.sTableId));
      },
      lengthMenu: [[20, 50, 100, -1], [20, 50, 100, "All"]],
      pageLength: 20,
      language: language,
      columns: columnDefinitions,
      columnDefs: columnUtil.generateColumnDefs(this.columnMappings, this.getTypeAlignments()),
      dom: '<"top"<"d-flex justify-content-between"B<"float-right"f>>>rt<"bottom"<"d-flex justify-content-between"<"float-left"il><"float-right"p>>>',
      buttons: this.generateCustomButtons(excelColumns, tableConfig.routeName, csrfToken),
      initComplete: (settings, json) => {
        this.handleInitComplete(settings, json, searchColumns, dropdownColumns, sumColumns);
      },
      rowCallback: (row, data, index) => {
        $(row).find('td').css('padding', '0.15rem 0.3rem');
        this.applyRowStyling(row, data);
      },
      drawCallback: (settings) => {
        this.handleDrawCallback(settings);
      }
    };
  }
  
  /**
   * Get type alignments
   */
  getTypeAlignments() {
    return {
      'number': 'right',
      'currency': 'right', 
      'datetime': 'right',
      'text': 'left'
    };
  }
  
  /**
   * Generiraj custom buttons
   */
  generateCustomButtons(excelColumns, routeName, csrfToken) {
    return [
      {
        extend: 'excelHtml5',
        text: '<i class="fas fa-file-excel"></i> Excel',
        className: 'dt-button--excel',
        exportOptions: {
          columns: excelColumns
        },
        title: routeName
      },
      {
        text: '<i class="fas fa-sync-alt"></i> Reset',
        className: 'dt-button--reset',
        action: (e, dt, node, config) => {
          if (this.services.filter && this.services.filter.clearAllFilters) {
            this.services.filter.clearAllFilters(dt);
          } else {
            dt.columns().every(function() {
              $(this.header()).find('.dt-filter__icon').removeClass('filtered');
              this.search('').draw();
            });
            localStorage.removeItem('DataTables_' + routeName + '_mojdatatables');
          }
        }
      }
    ];
  }
  
  /**
   * Handle initComplete - orchestracija svih servisa
   */
  handleInitComplete(settings, json, searchColumns, dropdownColumns, sumColumns) {
    const api = settings.oInstance.api();
    
    try {
      // STEP 1: Čekaj da se sve kolumne potpuno inicijaliziraju
      setTimeout(() => {
        // 1.1 Theme servis - column visibility (DELAYED)
        if (this.services.theme) {
          this.services.theme.restoreColumnVisibilityState(api);
          this.services.theme.addColumnVisibilityControl(api);
          this.services.theme.applyCompactStyles();
          this.services.theme.setupColumnVisibilityHandling(api, searchColumns, sumColumns);
        }
        
        // 1.2 Filter servis - postavi filtere (DELAYED)
        if (this.services.filter && this.cascadingFiltersEnabled) {
          this.services.filter.setupFilters(api, searchColumns, dropdownColumns, this.columnMappings);
          this.setupCascadingFilterRefresh(api);
        } else if (this.services.filter) {
          this.services.filter.setupFilters(api, searchColumns, dropdownColumns, this.columnMappings);
        }
        
        // 1.3 Column servis - sumiranje (DELAYED)
        if (this.services.column) {
          this.services.column.sumColumns(api, sumColumns);
        }
        
        // 1.4 Editor servis - inline editing (DELAYED)
        if (this.services.editor) {
          this.services.editor.setupInlineEditing(api, this.columnMappings);
        }
        
        // 1.5 Theme servis - finishing touches (DELAYED)
        if (this.services.theme) {
          setTimeout(() => {
            this.services.theme.fixFixedColumnsOnThemeChange(api);
          }, 100);
        }
      }, 200); // Povećaj delay za sigurnost
      
    } catch (error) {
      console.error('Error in initComplete:', error);
    }
  }
  
  /**
   * Setup cascading filter refresh
   */
  setupCascadingFilterRefresh(api) {
    if (!this.cascadingFiltersEnabled) {
      return;
    }
    
    api.on('draw.cascadingFilters', () => {
      let hasActiveFilters = false;
      api.columns().every(function() {
        if (this.search()) {
          hasActiveFilters = true;
          return false;
        }
      });
      
      if (hasActiveFilters && this.services.filter && !this.services.filter.refreshInProgress) {
        setTimeout(() => {
          if (this.services.filter.refreshUniqueValues) {
            this.services.filter.refreshUniqueValues(api);
          }
        }, 100);
      }
    });
  }
  
  /**
   * Handle drawCallback
   */
  handleDrawCallback(settings) {
    try {
      if (this.services.theme) {
        this.services.theme.applyCompactStyles();
        setTimeout(() => {
          this.services.theme.fixFixedColumnsOnThemeChange(this.table);
        }, 100);
      }
      
      if (this.services.filter && this.table) {
        this.services.filter.restoreFilterIcons(this.table);
      }
    } catch (error) {
      console.error('Error in drawCallback:', error);
    }
  }
  
  /**
   * Setup osnovnih table event handlera
   */
  setupTableEventHandlers() {
    const $table = $(this.tableSelector);
    
    // Tooltip na hover
    $table.on("mouseenter", "tbody td", (e) => {
      const cell = this.table.cell(e.currentTarget);
      const cellData = cell.data();
      if (cellData) {
        $(e.currentTarget).attr('title', cellData.toString());
      }
    });
  }
  
  /**
   * Primijeni row styling (delegiraj theme servisu u budućnosti)
   */
  applyRowStyling(row, data) {
    try {
      const rowStylingConfig = this.initialData?.config?.rowStyling || 
                               this.tableConfig?.rowStyling;
      
      if (!rowStylingConfig || rowStylingConfig.enabled !== 'yes') {
        return;
      }
      
      const rules = rowStylingConfig.rules || [];
      const sortedRules = rules.slice().sort((a, b) => (a.priority || 999) - (b.priority || 999));
      
      let appliedHighestPriority = null;
      
      sortedRules.forEach(rule => {
        if (!rule.condition_column || !rule.css_class) {
          return;
        }
        
        if (appliedHighestPriority !== null && (rule.priority || 999) > appliedHighestPriority) {
          return;
        }
        
        const columnIndex = this.findColumnIndexByName(rule.condition_column);
        if (columnIndex === -1) {
          if (this.debug) {
            console.warn(`Column '${rule.condition_column}' not found for row styling`);
          }
          return;
        }
        
        const cellValue = data[columnIndex];
        const conditionValue = rule.condition_value;
        const operator = rule.condition_operator || '==';
        
        let conditionMet = false;
        switch (operator) {
          case '==':
          case '===':
            conditionMet = cellValue == conditionValue;
            break;
          case '!=':
          case '!==':
            conditionMet = cellValue != conditionValue;
            break;
          case '>':
            conditionMet = parseFloat(cellValue) > parseFloat(conditionValue);
            break;
          case '<':
            conditionMet = parseFloat(cellValue) < parseFloat(conditionValue);
            break;
          case '>=':
            conditionMet = parseFloat(cellValue) >= parseFloat(conditionValue);
            break;
          case '<=':
            conditionMet = parseFloat(cellValue) <= parseFloat(conditionValue);
            break;
          case 'contains':
            conditionMet = String(cellValue).toLowerCase().includes(String(conditionValue).toLowerCase());
            break;
          case 'startsWith':
            conditionMet = String(cellValue).toLowerCase().startsWith(String(conditionValue).toLowerCase());
            break;
          default:
            conditionMet = cellValue == conditionValue;
        }
        
        if (conditionMet) {
          $(row).addClass(rule.css_class);
          
          if (appliedHighestPriority === null || (rule.priority || 999) < appliedHighestPriority) {
            appliedHighestPriority = rule.priority || 999;
          }
          
          if (this.debug) {
            console.log(`Applied row styling: ${rule.css_class} for ${rule.condition_column} = ${cellValue} (priority: ${rule.priority || 'none'})`);
          }
        }
      });
    } catch (error) {
      if (this.debug) {
        console.error('Error applying row styling:', error);
      }
    }
  }
  
  /**
   * Utility metode
   */
  addIndexesToMapping(columnMappings) {
    return columnMappings.map((column, index) => ({
      ...column,
      index: index
    }));
  }
  
  findColumnIndexByName(sqlColumnName) {
    if (!this.columnMappings) {
      return -1;
    }
    
    const column = this.columnMappings.find(col => col.sqlcolumnname === sqlColumnName);
    return column ? column.index : -1;
  }
  
  showErrorTable(errorMessage) {
    const $table = $(this.tableSelector);
    this.ensureTableStructure($table);
    
    this.table = $table.DataTable({
      data: [],
      columns: [{ title: errorMessage, data: null }],
      language: {
        emptyTable: errorMessage
      },
      paging: false,
      searching: false,
      ordering: false,
      info: false
    });
    
    return this.table;
  }
  
  ensureTableStructure($table) {
    if ($table.find('thead').length === 0) {
      $table.append('<thead><tr><th>Loading...</th></tr></thead>');
    }
    
    if ($table.find('tbody').length === 0) {
      $table.append('<tbody></tbody>');
    }
  }
  
  /**
   * Setter metode za vanjsko konfiguriranje
   */
  setCascadingFilters(enabled) {
    this.cascadingFiltersEnabled = enabled;
    
    if (this.debug) {
      console.log(`Cascading filters ${enabled ? 'enabled' : 'disabled'}`);
    }
    
    if (this.table && this.services.filter) {
      this.configureFilterService();
    }
  }

  setFilterRefreshMode(mode) {
    this.filterRefreshMode = mode;
    
    if (this.debug) {
      console.log(`Filter refresh mode set to: ${mode}`);
    }
    
    if (this.services.filter && this.services.filter.setRefreshMode) {
      const useServerSide = mode === 'server' ? true : 
                            mode === 'client' ? false : 'auto';
      this.services.filter.setRefreshMode(useServerSide);
    }
  }
  
  /**
   * Cleanup
   */
  destroy() {
    if (this.table) {
      this.table.off('draw.cascadingFilters');
    }
    
    Object.values(this.services).forEach(service => {
      if (typeof service.destroy === 'function') {
        service.destroy();
      }
    });
    
    if (this.table) {
      this.table.destroy();
      this.table = null;
    }
  }
}

// Strategy classes - bez promjena, samo čišći kod
class DataProcessingStrategy {
  constructor(core) {
    this.core = core;
    this.debug = core.debug;
  }
  
  prepareConfiguration(dataUrl) {
    throw new Error('Method prepareConfiguration must be implemented');
  }
  
  initialize(dataUrl) {
    throw new Error('Method initialize must be implemented');
  }
}

class ServerSideStrategy extends DataProcessingStrategy {
  prepareConfiguration(dataUrl) {
    const baseConfig = this.core.prepareBaseConfiguration();
    
    baseConfig.serverSide = true;
    
    baseConfig.ajax = {
      url: dataUrl,
      type: 'POST',
      data: (d) => {
        if (this.debug) {
          console.log('Server-side data request', d);
        }
        return d;
      },
      dataSrc: (response) => {
        if (this.debug) {
          console.log('Server-side response', response);
        }
        
        if (response?.uniqueValues) {
          this.core.uniqueValues = response.uniqueValues;
          
          if (this.core.services.filter && this.core.services.filter.updateAllDropdownOptions) {
            this.core.services.filter.updateAllDropdownOptions(response.uniqueValues);
          }
        }
        
        return response.data || [];
      },
      error: (xhr, error, thrown) => {
        console.error('AJAX error', error, thrown);
      }
    };
    
    return baseConfig;
  }

  initialize(dataUrl) {
    const config = this.prepareConfiguration(dataUrl);
    
    if (this.debug) {
      console.log('Initializing table with server-side configuration', config);
    }
    
    const $table = $(this.core.tableSelector);
    this.core.table = $table.DataTable(config);
    
    this.core.setupTableEventHandlers();
    
    if (this.core.services.theme) {
      this.core.services.theme.applyThemeStyling(this.core.table);
    }
    
    return Promise.resolve(this.core.table);
  }
}

class ClientSideStrategy extends DataProcessingStrategy {
  prepareConfiguration(dataUrl) {
    const baseConfig = this.core.prepareBaseConfiguration();
    
    baseConfig.serverSide = false;
    baseConfig.ajax = null;
    baseConfig.data = this.core.initialData || [];
    
    const originalInitComplete = baseConfig.initComplete;
    baseConfig.initComplete = (settings, json) => {
      settings.json = { uniqueValues: this.core.uniqueValues };
      
      if (originalInitComplete) {
        originalInitComplete.call(this.core, settings, { uniqueValues: this.core.uniqueValues });
      }
    };
    
    return baseConfig;
  }

  initialize(dataUrl) {
    return new Promise((resolve, reject) => {
      if (this.debug) {
        console.log('Client-side strategy: Fetching all records');
      }
      
      $.ajax({
        url: dataUrl,
        type: 'POST',
        data: {
          init: 'true',
          draw: 1,
          start: 0,
          length: -1,
          configOnly: false
        },
        success: (response) => {
          try {
            let parsedResponse = typeof response === 'string' ? JSON.parse(response) : response;
            
            this.core.initialData = parsedResponse.data || [];
            
            if (parsedResponse.uniqueValues) {
              this.core.uniqueValues = parsedResponse.uniqueValues;
            }
            
            if (this.debug) {
              console.log(`Fetched ${this.core.initialData.length} records for client-side processing`);
            }
            
            const config = this.prepareConfiguration(dataUrl);
            
            const $table = $(this.core.tableSelector);
            this.core.table = $table.DataTable(config);
            
            this.core.setupTableEventHandlers();
            
            if (this.core.services.theme) {
              this.core.services.theme.applyThemeStyling(this.core.table);
            }
            
            resolve(this.core.table);
          } catch (error) {
            console.error('Error initializing client-side DataTable:', error);
            reject(error);
          }
        },
        error: (xhr, error, thrown) => {
          console.error('AJAX error during data fetch:', error, thrown);
          reject(error);
        }
      });
    });
  }
}