/**
 * DataTables Factory - UPDATED VERSION
 * Dodana podrška za fiksnu strategy konfiguraciju
 */

import { DataTablesCore } from './core/datatables-core.js';
import { ColumnService } from './services/column-service.js';
import { FilterService } from './services/filter-service.js';
import { FormatService } from './services/format-service.js';
import { EditorService } from './services/editor-service.js';
import { SecurityService } from './services/security-service.js';
import { I18nService } from './services/i18n-service.js';
import { ThemeService } from './services/theme-service.js';

/**
 * Creates services for DataTables with clear dependency management
 * @param {Object} options - Configuration options
 * @return {Object} - Object with all service instances
 */
function createServices(options = {}) {
  const debug = options.debug || false;
  const locale = options.locale || 'hr';
  const strategy = options.strategy || 'client';
  
  if (debug) {
    console.log('Creating DataTables services for strategy:', strategy);
  }
  
  // STEP 1: Create basic services with no dependencies
  const services = {
    security: new SecurityService({
      debug,
      tokenRefreshInterval: 0, // Disable auto-refresh for now
      csrfHeaderName: 'X-CSRF-Token',
      csrfInputName: 'csrf_token'
    }),
    
    format: new FormatService(locale),
    
    i18n: new I18nService(locale, {
      debug,
      fallbackLocale: 'en'
    }),
    
    theme: new ThemeService({ 
      debug 
    })
  };
  
  // STEP 2: Create services that depend on basic services
  services.column = new ColumnService({
    formatService: services.format,
    debug
  });
  
  services.filter = new FilterService({
    security: services.security,
    i18n: services.i18n,
    useServerSideRefresh: strategy === 'server',
    debug
  });
  
  services.editor = new EditorService({
    security: services.security,
    formatService: services.format,  // ADD THIS LINE
    useGlobalCsrf: true,
    debug
  });
  
  if (debug) {
    console.log('All services created successfully for strategy:', strategy);
    console.log('Services:', Object.keys(services));
  }
  
  return services;
}

/**
 * Ensures jQuery is ready and CSRF is configured
 * @return {Promise} - Promise that resolves when jQuery is ready
 */
function ensureJQueryReady() {
  return new Promise((resolve) => {
    const checkJquery = () => {
      if (!window.jQuery) {
        if (typeof document !== 'undefined') {
          setTimeout(checkJquery, 100);
        }
        return;
      }
      
      // Check if CSRF settings are already applied
      if ($.ajaxSettings?.headers?.['X-CSRF-Token']) {
        resolve();
      } else {
        // Wait a bit more for CSRF to be set
        setTimeout(() => {
          // Try to get CSRF token from various sources
          const csrfToken = document.querySelector('input[name="csrf_token"]')?.value;
          if (csrfToken) {
            // Set CSRF token in jQuery AJAX headers if not set
            $.ajaxSetup({
              headers: {
                'X-CSRF-Token': csrfToken
              }
            });
          }
          resolve();
        }, 100);
      }
    };
    
    if (typeof document !== 'undefined') {
      if (document.readyState === 'complete' || document.readyState === 'interactive') {
        checkJquery();
      } else {
        document.addEventListener('DOMContentLoaded', checkJquery);
      }
    } else {
      // Server-side or non-browser environment
      resolve();
    }
  });
}

/**
 * Creates a DataTables instance with all necessary services
 * @param {string} selector - CSS selector for the table
 * @param {string} dataUrl - URL for data retrieval
 * @param {Object} options - Configuration options
 * @return {Promise<Object>} - Promise that returns the DataTables instance
 */
export async function createDataTable(selector, dataUrl, options = {}) {
  const debug = options.debug || false;
  const strategy = options.strategy || 'client'; // DEFAULT to client-side
  
  if (debug) {
    console.log('=== DATATABLES FACTORY START ===');
    console.log('Selector:', selector);
    console.log('Data URL:', dataUrl);
    console.log('Strategy:', strategy);
    console.log('Options:', options);
  }
  
  try {
    // STEP 1: Ensure jQuery is ready
    await ensureJQueryReady();
    
    if (debug) console.log("✓ jQuery is ready, creating services");
    
    // STEP 2: Create services with strategy awareness
    const services = createServices({
      locale: options.locale || 'hr',
      routes: options.routes || {},
      strategy: strategy,  // PASS STRATEGY TO SERVICES
      debug
    });
    
    if (debug) console.log("✓ Services created, initializing DataTablesCore");
    
    // STEP 3: Create main DataTables instance with dependencies
    const core = new DataTablesCore({
      tableSelector: selector,
      services,
      config: options.config || {},
      routes: options.routes || {},
      strategy: strategy,  // PASS STRATEGY TO CORE
      cascadingFilters: options.cascadingFilters !== false,
      debug
    });
    
    // STEP 4: Make services globally available for cross-service communication
    if (typeof window !== 'undefined') {
      window.filterService = services.filter;
      window.columnService = services.column;
      window.editorService = services.editor;
      window.themeService = services.theme;
    }
    
    if (debug) console.log("✓ Global service references set");
    
    // STEP 5: Initialize and return the table
    const table = await core.initialize(dataUrl);
    
    if (debug) {
      console.log("✓ Initialization completed with strategy:", strategy);
      console.log('=== DATATABLES FACTORY END ===');
    }
    
    return table;
    
  } catch (error) {
    console.error('=== DATATABLES FACTORY ERROR ===');
    console.error('Error in createDataTable:', error);
    console.error('Strategy was:', strategy);
    console.error('Stack trace:', error.stack);
    throw error;
  }
}

/**
 * Global method for compatibility with legacy code
 * @param {string} dataUrl - URL for data retrieval
 * @return {Promise<Object>} - Promise that returns the DataTables instance
 */
export function initializeDataTable(dataUrl) {
  return createDataTable('#mojdatatables', dataUrl, {
    strategy: 'client', // Default to client-side for legacy compatibility
    debug: false
  });
}

/**
 * Advanced initialization with full configuration
 * @param {Object} config - Full configuration object
 * @return {Promise<Object>} - Promise that returns the DataTables instance
 */
export function createAdvancedDataTable(config) {
  const {
    selector = '#mojdatatables',
    dataUrl,
    locale = 'hr',
    routes = {},
    strategy = 'client',  // DEFAULT to client-side
    cascadingFilters = true,
    debug = false,
    ...otherOptions
  } = config;
  
  return createDataTable(selector, dataUrl, {
    locale,
    routes,
    strategy,  // PASS STRATEGY
    cascadingFilters,
    debug,
    config: otherOptions
  });
}

/**
 * Create DataTable with specific service overrides
 * @param {string} selector - CSS selector for the table
 * @param {string} dataUrl - URL for data retrieval
 * @param {Object} services - Service overrides
 * @param {Object} options - Additional options
 * @return {Promise<Object>} - Promise that returns the DataTables instance
 */
export function createDataTableWithServices(selector, dataUrl, services = {}, options = {}) {
  const strategy = options.strategy || 'client';
  
  const defaultServices = createServices({
    ...options,
    strategy: strategy
  });
  const mergedServices = { ...defaultServices, ...services };
  
  const core = new DataTablesCore({
    tableSelector: selector,
    services: mergedServices,
    config: options.config || {},
    routes: options.routes || {},
    strategy: strategy,  // PASS STRATEGY
    debug: options.debug || false
  });
  
  return core.initialize(dataUrl);
}

/**
 * Utility function to destroy DataTable instance
 * @param {Object} table - DataTables instance
 */
export function destroyDataTable(table) {
  if (table && typeof table.destroy === 'function') {
    table.destroy();
  }
  
  // Clean up global references
  if (typeof window !== 'undefined') {
    delete window.filterService;
    delete window.columnService;
    delete window.editorService;
    delete window.themeService;
    delete window.currentColumnMappings;
  }
}