/**
 * DataTables Module Initializer - UPDATED VERSION
 * Dodana podrška za fiksnu strategy konfiguraciju
 */

class DataTablesModuleInitializer {
  constructor(options = {}) {
    this.debug = options.debug || false;
    this.loadingStylesApplied = new Set();
    this.initializedModules = new Map();
    
    // Global loading flag
    window.dtIsLoading = false;
  }

  /**
   * GLAVNA METODA: Inicijaliziraj bilo koji modul
   * @param {Object} moduleConfig - Konfiguracija modula
   * @return {Promise<Object>} - DataTable instance
   */
  async initializeModule(moduleConfig) {
    try {
      // Validate config
      if (!this.validateModuleConfig(moduleConfig)) {
        throw new Error('Invalid module configuration');
      }

      // Set loading state
      window.dtIsLoading = true;

      if (this.debug) {
        console.log(`=== ${moduleConfig.module.name.toUpperCase()} MODULE INITIALIZATION ===`);
        console.log('Strategy:', moduleConfig.table.strategy || 'client');
        console.log('Configuration:', moduleConfig);
      }

      // Apply loading styles if not already applied
      this.setupLoadingStyles(moduleConfig.table.selector);

      // Check if table element exists
      if (!document.querySelector(moduleConfig.table.selector)) {
        throw new Error(`Table element ${moduleConfig.table.selector} not found`);
      }

      // Get routes from app config
      const routes = this.getRoutesFromAppConfig(moduleConfig.routes);

      // Prepare configuration for DataTables
      const config = this.prepareConfiguration(moduleConfig, routes);

      // Import DataTables factory dynamically
      const factory = await this.importDataTablesFactory();

      if (this.debug) {
        console.log('Final configuration for DataTables:', config);
      }

      // Create DataTable instance
      const table = await factory.createDataTable(
        config.selector,
        config.routes.fetchAjax,
        config
      );

      // Store global reference
      this.storeGlobalReference(moduleConfig.module.name, table);

      // Apply completion styles
      this.applyCompletionStyles(moduleConfig.table.selector, moduleConfig.module.name);

      // Store module as initialized
      this.initializedModules.set(moduleConfig.module.name, {
        config: moduleConfig,
        table: table,
        timestamp: Date.now()
      });

      return table;

    } catch (error) {
      console.error(`Error initializing ${moduleConfig.module.name} module:`, error);
      
      // Reset loading state on error
      window.dtIsLoading = false;
      
      if (this.debug) {
        alert(`Error loading ${moduleConfig.module.name} table. Check console for details.`);
      }
      
      throw error;
    }
  }

  /**
   * Validiraj konfiguraciju modula
   * @param {Object} config - Module config
   * @return {boolean} - Is valid
   */
  validateModuleConfig(config) {
    const required = ['module', 'table', 'routes'];
    const moduleRequired = ['name', 'title'];
    const tableRequired = ['selector'];
    const routesRequired = ['fetchAjax'];

    // Check main structure
    for (const key of required) {
      if (!config[key]) {
        console.error(`Missing required config: ${key}`);
        return false;
      }
    }

    // Check module structure
    for (const key of moduleRequired) {
      if (!config.module[key]) {
        console.error(`Missing required module config: ${key}`);
        return false;
      }
    }

    // Check table structure
    for (const key of tableRequired) {
      if (!config.table[key]) {
        console.error(`Missing required table config: ${key}`);
        return false;
      }
    }

    // Check routes structure
    for (const key of routesRequired) {
      if (!config.routes[key]) {
        console.error(`Missing required routes config: ${key}`);
        return false;
      }
    }

    // Validate strategy
    const strategy = config.table.strategy || 'client';
    if (!['client', 'server'].includes(strategy)) {
      console.error(`Invalid strategy: ${strategy}. Must be 'client' or 'server'`);
      return false;
    }

    return true;
  }

  /**
   * Dohvati rute iz AppConfig
   * @param {Object} routesConfig - Routes configuration
   * @return {Object} - Full routes with base URL
   */
  getRoutesFromAppConfig(routesConfig) {
    const appConfigElement = document.getElementById('app-config');
    if (!appConfigElement) {
      throw new Error('app-config element not found');
    }

    let publicUrl;
    try {
      const appConfig = JSON.parse(appConfigElement.textContent);
      publicUrl = appConfig.paths.app_url;
    } catch (error) {
      throw new Error('Failed to parse app-config: ' + error.message);
    }

    const routes = {};
    Object.keys(routesConfig).forEach(key => {
      const route = routesConfig[key];
      routes[key] = route.startsWith('http') || route.startsWith('/') ? 
        route : 
        `${publicUrl}${route}`;
    });

    return routes;
  }

  /**
   * Pripremi konfiguraciju za DataTables - UPDATED s strategy
   * @param {Object} moduleConfig - Module config
   * @param {Object} routes - Resolved routes
   * @return {Object} - DataTables config
   */
  prepareConfiguration(moduleConfig, routes) {
    return {
      // Table settings
      selector: moduleConfig.table.selector,
      locale: moduleConfig.locale?.language || 'hr',
      
      // Routes
      routes: routes,
      
      // STRATEGY - ključna promjena
      strategy: moduleConfig.table.strategy || 'client',
      
      // Features
      cascadingFilters: moduleConfig.features?.cascadingFilters !== false,
      
      // Debug
      debug: moduleConfig.debug?.enabled || false,
      
      // Table config
      config: {
        pageLength: moduleConfig.table?.pageLength || 20,
        lengthMenu: moduleConfig.table?.lengthMenu || [[20, 50, 100, -1], [20, 50, 100, "All"]],
        scrollX: moduleConfig.table?.scrollX !== false,
        scrollCollapse: moduleConfig.table?.scrollCollapse !== false,
        stateSave: moduleConfig.table?.stateSave !== false,
        
        // Default order if specified
        order: moduleConfig.table?.defaultOrder || undefined
      }
    };
  }

  /**
   * Dinamički uvezi DataTables factory
   * @return {Object} - Factory module
   */
  async importDataTablesFactory() {
    const appConfigElement = document.getElementById('app-config');
    if (!appConfigElement) {
      throw new Error('app-config element not found');
    }

    let publicUrl;
    try {
      const appConfig = JSON.parse(appConfigElement.textContent);
      publicUrl = appConfig.paths.app_url;
    } catch (error) {
      throw new Error('Failed to parse app-config: ' + error.message);
    }

    const factoryUrl = `${publicUrl}/srcassets/datatables/js/datatables-factory.js`;
    
    try {
      return await import(factoryUrl);
    } catch (error) {
      throw new Error(`Failed to import DataTables factory from ${factoryUrl}: ` + error.message);
    }
  }

  /**
   * Postavi loading styles
   * @param {string} tableSelector - Table selector
   */
  setupLoadingStyles(tableSelector) {
    const selectorKey = tableSelector.replace('#', '');
    
    // Skip if already applied
    if (this.loadingStylesApplied.has(selectorKey)) {
      return;
    }

    // Special handling for rashodi module
    const isRashodi = tableSelector === '#rashodidatatables';
    const styleId = isRashodi ? 'dt-rashodi-init-style' : `dt-init-style-${selectorKey}`;
    const loadedClass = isRashodi ? 'dt-rashodi-loaded' : 'dt-loaded';

    const styleEl = document.createElement('style');
    styleEl.id = styleId;
    styleEl.innerHTML = `
      ${tableSelector}, ${tableSelector}_wrapper {
        visibility: hidden;
        opacity: 0;
        transition: opacity 0.3s ease;
      }
      .${loadedClass} {
        visibility: visible !important;
        opacity: 1 !important;
      }
    `;
    document.head.appendChild(styleEl);
    
    this.loadingStylesApplied.add(selectorKey);
  }

  /**
   * Primijeni completion styles
   * @param {string} tableSelector - Table selector
   * @param {string} moduleName - Module name
   */
  applyCompletionStyles(tableSelector, moduleName) {
    setTimeout(() => {
      const selectorKey = tableSelector.replace('#', '');
      
      const wrapper = document.getElementById(`${selectorKey}_wrapper`);
      const tableEl = document.querySelector(tableSelector);
      
      // Use module-specific class names for loading states
      const loadedClass = moduleName === 'rashodi' ? 'dt-rashodi-loaded' : 'dt-loaded';
      
      if (wrapper) wrapper.classList.add(loadedClass);
      if (tableEl) tableEl.classList.add(loadedClass);
      
      // Remove init style
      const styleId = moduleName === 'rashodi' ? `dt-rashodi-init-style` : `dt-init-style-${selectorKey}`;
      const styleEl = document.getElementById(styleId);
      if (styleEl) styleEl.remove();
      
      // Reset loading flag
      window.dtIsLoading = false;
      
      if (this.debug) {
        console.log(`✓ ${moduleName} table initialized successfully`);
      }
    }, 300);
  }

  /**
   * Spremi global reference
   * @param {string} moduleName - Module name
   * @param {Object} table - DataTable instance
   */
  storeGlobalReference(moduleName, table) {
    // Store module-specific reference
    window[`${moduleName}Table`] = table;
    
    // Store in global functions object
    const functionsKey = `${this.capitalizeFirst(moduleName)}TableFunctions`;
    
    if (!window[functionsKey]) {
      window[functionsKey] = {};
    }
    
    // Add standard functions
    Object.assign(window[functionsKey], {
      reloadTable: () => {
        if (table && table.ajax && table.ajax.reload) {
          table.ajax.reload();
        }
      },
      
      applyFilter: (column, value) => {
        if (table && table.column) {
          table.column(column).search(value).draw();
        }
      },
      
      clearFilters: () => {
        if (table && table.search && table.columns) {
          table.search('').columns().search('').draw();
        }
      },
      
      getTable: () => table
    });
  }

  /**
   * Capitalize first letter
   * @param {string} str - String to capitalize
   * @return {string} - Capitalized string
   */
  capitalizeFirst(str) {
    return str.charAt(0).toUpperCase() + str.slice(1);
  }

  /**
   * Auto-inicijalizacija modula
   * @param {Object} moduleConfig - Module config
   */
  autoInitialize(moduleConfig) {
    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', () => {
        this.initializeModule(moduleConfig);
      });
    } else {
      // DOM is already ready
      setTimeout(() => {
        this.initializeModule(moduleConfig);
      }, 50);
    }
  }

  /**
   * Get initialized module
   * @param {string} moduleName - Module name
   * @return {Object|null} - Module info or null
   */
  getInitializedModule(moduleName) {
    return this.initializedModules.get(moduleName) || null;
  }

  /**
   * Check if module is initialized
   * @param {string} moduleName - Module name
   * @return {boolean} - Is initialized
   */
  isModuleInitialized(moduleName) {
    return this.initializedModules.has(moduleName);
  }

  /**
   * Destroy module
   * @param {string} moduleName - Module name
   */
  destroyModule(moduleName) {
    const moduleInfo = this.initializedModules.get(moduleName);
    
    if (moduleInfo && moduleInfo.table) {
      // Destroy DataTable
      if (typeof moduleInfo.table.destroy === 'function') {
        moduleInfo.table.destroy();
      }
      
      // Remove global references
      delete window[`${moduleName}Table`];
      delete window[`${this.capitalizeFirst(moduleName)}TableFunctions`];
      
      // Remove from initialized modules
      this.initializedModules.delete(moduleName);
      
      if (this.debug) {
        console.log(`✓ ${moduleName} module destroyed`);
      }
    }
  }

  /**
   * Destroy all modules
   */
  destroyAll() {
    const moduleNames = Array.from(this.initializedModules.keys());
    moduleNames.forEach(moduleName => {
      this.destroyModule(moduleName);
    });
    
    // Clean up loading styles
    this.loadingStylesApplied.forEach(selectorKey => {
      const styleEl = document.getElementById(`dt-init-style-${selectorKey}`);
      if (styleEl) styleEl.remove();
    });
    
    this.loadingStylesApplied.clear();
    window.dtIsLoading = false;
  }
}

// Create global singleton instance
window.dataTablesInitializer = new DataTablesModuleInitializer({
  debug: false // Can be overridden per module
});

/**
 * Global convenience function for initializing modules
 * @param {Object} moduleConfig - Module configuration
 * @return {Promise<Object>} - DataTable instance
 */
window.initializeDataTablesModule = function(moduleConfig) {
  return window.dataTablesInitializer.initializeModule(moduleConfig);
};

/**
 * Global convenience function for auto-initialization
 * @param {Object} moduleConfig - Module configuration
 */
window.autoInitializeDataTablesModule = function(moduleConfig) {
  window.dataTablesInitializer.autoInitialize(moduleConfig);
};