/**
 * DataTables Column Service - AŽURIRANA VERZIJA s hrvatskim formatiranjem
 * Poboljšana integracija s format servisom za valute
 */

export class ColumnService {
  /**
   * Creates a new ColumnService instance
   * @param {Object} options - Configuration options
   */
  constructor(options = {}) {
    this.formatService = options.formatService || null;
    this.tableConfig = null;
    this.columnMappings = null;
    this.debug = options.debug || false;
    
    // Default settings - AŽURIRANO za hrvatski format
    this.sumFormatters = {
      'currency': this.formatCurrency.bind(this),
      'number': this.formatNumber.bind(this)
    };
  }
  
  /**
   * Set table configuration
   * @param {Object} config - Table configuration
   */
  setTableConfig(config) {
    this.tableConfig = config;
  }
  
  /**
   * Set column mappings
   * @param {Array} mappings - Array of column mapping objects
   */
  setColumnMappings(mappings) {
    this.columnMappings = mappings;
    
    // Make globally available for theme service
    if (typeof window !== 'undefined') {
      window.currentColumnMappings = mappings;
    }
  }
  
  /**
   * Get column indexes where property has a specific value
   * @param {string} property - Property name to check
   * @param {string} value - Property value to match
   * @param {Array} mappings - Array of column mapping objects
   * @return {Array} - Array of matching column indexes
   */
  getIndexesForProperty(property, value, mappings) {
    return mappings.reduce((indexes, item) => {
      if (item[property] === value) {
        indexes.push(item.index);
      }
      return indexes;
    }, []);
  }
  
  /**
   * Get column indexes and types where property has a specific value
   * @param {string} property - Property name to check
   * @param {string} value - Property value to match
   * @param {Array} mappings - Array of column mapping objects
   * @return {Array} - Array of objects with index and type
   */
  getIndexesForPropertyWithType(property, value, mappings) {
    return mappings
      .filter(item => item[property] === value)
      .map(item => ({ 
        index: item.index, 
        type: item.type 
      }));
  }
  
  /**
   * Exclude specific indexes from array
   * @param {Array} indexesToExclude - Array of indexes to exclude
   * @param {Array} mappings - Array of column mapping objects
   * @return {Array} - Array of remaining indexes
   */
  excludeIndexes(indexesToExclude, mappings) {
    return mappings
      .filter(column => !indexesToExclude.includes(column.index))
      .map(column => column.index);
  }
  
  /**
   * Create column definitions for DataTables - AŽURIRANO za valute
   * @param {Array} mappings - Array of column mapping objects
   * @return {Array} - Array of column definitions for DataTables
   */
  createColumnDefinitions(mappings) {
    if (!mappings || !Array.isArray(mappings)) {
      if (this.debug) console.error('Invalid column mappings provided:', mappings);
      return [];
    }
    
    return mappings.map((column, index) => {
      const columnDef = {
        data: index.toString(),
        title: column.headername,
        name: column.sqlcolumnname
      };
      
      // Set renderer based on column type - KORISTI FORMAT SERVICE
      if (column.type && this.formatService) {
        switch (column.type) {
          case 'date':
            columnDef.render = (data, type, row, meta) => {
              if (type !== 'display') return data;
              return this.formatService.formatDate(data);
            };
            columnDef.type = 'date-hr';
            break;
          case 'datetime':
            columnDef.render = (data, type, row, meta) => {
              if (type !== 'display') return data;
              return this.formatService.formatDateTime(data);
            };
            columnDef.type = 'datetime-hr';
            break;
          case 'currency':
            // KLJUČNA PROMJENA: Pravilno formatiranje valuta
            columnDef.render = (data, type, row, meta) => {
              return this.formatService.formatCurrencyForDataTables(data, type);
            };
            // Set custom sorting type for Croatian currency
            columnDef.type = 'currency-hr';
            break;
        }
      } else {
        // Fallback to local renderers if no format service
        switch (column.type) {
          case 'date':
            columnDef.render = this.dateRenderer;
            columnDef.type = 'date-hr';
            break;
          case 'datetime':
            columnDef.render = this.datetimeRenderer;
            columnDef.type = 'datetime-hr';
            break;
          case 'currency':
            // POBOLJŠANI FALLBACK za hrvatski format
            columnDef.render = this.croatianCurrencyRenderer;
            columnDef.type = 'currency-hr';
            break;
          case 'boolean':
            columnDef.render = this.booleanRenderer;
            break;
        }
      }
      
      // Set custom renderer if specified
      if (column.render) {
        columnDef.render = (data, type, row, meta) => this.renderColumn(data, type, row, meta, column.render);
      }
      
      return columnDef;
    });
  }
  
  /**
   * Render column with specific renderer
   * @param {*} data - Cell data
   * @param {string} type - Render type
   * @param {Array} row - Row data
   * @param {Object} meta - Meta information
   * @param {string} renderer - Renderer name
   * @return {string} - Rendered cell content
   */
  renderColumn(data, type, row, meta, renderer) {
    // Only apply rendering for display type
    if (type !== 'display') {
      return data;
    }
    
    switch (renderer) {
      case 'actions':
        return this.actionsRenderer(data, type, row, meta);
      case 'sharelink':
        return this.sharelinkRenderer(data, type, row, meta);
      case 'currency':
        // NOVA OPCIJA: Eksplicitno currency rendering
        return this.formatService ? 
          this.formatService.formatCurrency(data) : 
          this.croatianCurrencyRenderer(data, type);
      default:
        return data;
    }
  }
  
  /**
   * FALLBACK RENDERERS - koristi se ako nema format service
   */
  
  /**
   * Render date values
   * @param {*} data - Cell data
   * @param {string} type - Render type
   * @return {string} - Formatted date
   */
  dateRenderer(data, type) {
    if (type !== 'display') return data;
    if (!data) return '';
    
    // Use FormatService if available
    if (this.formatService) {
      return this.formatService.formatDate(data);
    }
    
    // Simple fallback
    return String(data).replace(/^(\d{4})-(\d{2})-(\d{2})/, '$3-$2-$1');
  }
  
  /**
   * Render datetime values
   * @param {*} data - Cell data
   * @param {string} type - Render type
   * @return {string} - Formatted datetime
   */
  datetimeRenderer(data, type) {
    if (type !== 'display') return data;
    if (!data) return '';
    
    // Use FormatService if available
    if (this.formatService) {
      return this.formatService.formatDateTime(data);
    }
    
    // Simple fallback
    return String(data).replace(/^(\d{4})-(\d{2})-(\d{2})\s+(\d{2}):(\d{2})/, '$3-$2-$1 $4:$5');
  }
  
  /**
   * NOVA FUNKCIJA: Croatian currency renderer - POBOLJŠANA VERZIJA
   * @param {*} data - Cell data
   * @param {string} type - Render type
   * @return {string|number} - Formatted currency
   */
  croatianCurrencyRenderer(data, type) {
    if (type === 'display') {
      const value = parseFloat(data);
      if (isNaN(value)) return '';
      
      // Croatian format: 1.234,56 €
      const formattedNumber = value.toLocaleString('hr-HR', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
      });
      
      return `${formattedNumber} €`;
    } else {
      // For sorting and filtering, return numeric value
      return parseFloat(data) || 0;
    }
  }
  
  /**
   * Render currency values - DEPRECATED, koristi croatianCurrencyRenderer
   * @param {*} data - Cell data
   * @param {string} type - Render type
   * @return {string} - Formatted currency
   */
  currencyRenderer(data, type) {
    return this.croatianCurrencyRenderer(data, type);
  }
  
  /**
   * Render boolean values
   * @param {*} data - Cell data
   * @param {string} type - Render type
   * @return {string} - Formatted boolean
   */
  booleanRenderer(data, type) {
    if (type !== 'display') return data;
    
    let isTrue = false;
    if (typeof data === 'boolean') {
      isTrue = data;
    } else if (typeof data === 'number') {
      isTrue = data > 0;
    } else if (typeof data === 'string') {
      const lowerData = data.toLowerCase();
      isTrue = lowerData === 'true' || lowerData === 'yes' || lowerData === 'da' || lowerData === '1';
    }
    
    return isTrue
      ? '<i class="fas fa-check text-success"></i>'
      : '<i class="fas fa-times text-danger"></i>';
  }
  
  /**
   * Render action buttons
   * @param {*} data - Cell data
   * @param {string} type - Render type
   * @param {Array} row - Row data
   * @return {string} - HTML for action buttons
   */
  actionsRenderer(data, type, row) {
    if (type !== 'display') return data;
    
    // Get routes safely from table config
    const editRoute = (this.tableConfig && this.tableConfig.routes) ? 
      this.tableConfig.routes.editRoute || '' : '';
    const deleteRoute = (this.tableConfig && this.tableConfig.routes) ? 
      this.tableConfig.routes.deleteRoute || '' : '';
    
    // Get ID safely
    const id = row[0] ? this.escapeHtml(row[0]) : '';
    
    if (!id) return '&nbsp;';
    
    // Get CSRF token with XSS protection
    const csrfToken = $('input[name="csrf_token"]').val() || '';
    const sanitizedCsrfToken = this.escapeHtml(csrfToken);
    
    // Create edit button
    const editButton = '<a href="' + editRoute + '/' + id + '" class="dt-action dt-action--edit"><button type="button" class="btn btn-secondary btn-sm">Edit</button></a>';
    
    // Create delete form with CSRF protection
    const deleteForm = '<form action="' + deleteRoute + '" method="post" class="dt-action dt-action--delete" onsubmit="return confirm(\'Are you sure you want to delete this record?\')">' +
      '<input type="hidden" name="csrf_token" value="' + sanitizedCsrfToken + '">' +
      '<button type="submit" class="btn btn-danger btn-sm" name="id" value="' + id + '">Delete</button>' +
      '</form>';
    
    return '<div class="dt-actions">' + editButton + deleteForm + '</div>';
  }

  /**
   * Render sharelink values as clickable buttons
   * @param {*} data - Cell data (URL)
   * @param {string} type - Render type
   * @param {Array} row - Row data
   * @return {string} - HTML for sharelink button
   */
  sharelinkRenderer(data, type, row) {
    if (type !== 'display') return data;
    
    const hasData = data && data !== null && data !== '' && data.toString().trim() !== '';
    
    if (hasData) {
      let url = String(data).trim();
      
      // Ensure URL has protocol
      if (!url.match(/^https?:\/\//)) {
        url = 'https://' + url;
      }
      
      // Sanitize URL to prevent XSS
      const sanitizedUrl = this.escapeHtml(url);
      
      return `<a href="${sanitizedUrl}" 
                target="_blank" 
                rel="noopener noreferrer" 
                class="btn btn-primary btn-sm sharelink-btn" 
                title="Otvori: ${sanitizedUrl}">
        <i class="fas fa-external-link-alt"></i>
      </a>`;
    } else {
      return `<span class="btn btn-secondary btn-sm sharelink-btn disabled" 
                title="Nema linka">
        <i class="fas fa-ban"></i>
      </span>`;
    }
  }
  
  /**
   * Generate column definitions with styling
   * @param {Array} mappings - Array of column mapping objects
   * @param {Object} typeAlignments - Object mapping types to alignments
   * @return {Array} - Array of column definition objects
   */
  generateColumnDefs(mappings, typeAlignments) {
    if (!mappings || !Array.isArray(mappings)) {
      if (this.debug) console.error('Invalid column mappings provided for column defs:', mappings);
      return [];
    }
    
    return mappings.map(column => {
      const def = {
        targets: column.index
      };
      
      if (column.width) {
        def.width = column.width + 'px';
      }
      
      if (typeAlignments && typeAlignments[column.type]) {
        def.className = 'dt-body-' + typeAlignments[column.type];
      }
      
      if (column.type === 'currency') {
        def.className = (def.className || '') + ' dt-currency';
      } else if (column.type === 'datetime') {
        def.className = (def.className || '') + ' dt-datetime';
      }
      
      if (column.inlineediting && column.inlineediting.enabled === 'yes') {
        def.className = (def.className || '') + ' dt-editable';
      }
      
      return def;
    });
  }
  
  /**
   * Generate table HTML structure
   * @param {Array} mappings - Array of column mapping objects
   * @return {Object} - Object with thead and tfoot HTML strings
   */
  generateTableStructure(mappings) {
    if (!mappings || !Array.isArray(mappings)) {
      if (this.debug) console.error('Invalid column mappings for table structure:', mappings);
      return { thead: '<thead><tr><th>Error: Invalid mappings</th></tr></thead>', tfoot: '<tfoot><tr><th></th></tr></tfoot>' };
    }
    
    let thead = '<thead><tr>';
    let tfoot = '<tfoot><tr>';

    mappings.forEach(column => {
      thead += `<th>${this.escapeHtml(column.headername)}</th>`;

      if (column.calculateinfoot === 'yes') {
        tfoot += `<th class="dt-sum" data-column-type="${column.type || 'text'}">${this.escapeHtml(column.headername)}</th>`;
      } else {
        tfoot += '<th></th>';
      }
    });

    thead += '</tr></thead>';
    tfoot += '</tr></tfoot>';

    return {
      thead: thead,
      tfoot: tfoot
    };
  }
  
  /**
   * Set up sum functions for footer columns
   * @param {Object} table - DataTables API instance
   * @param {Array} columnDetails - Array of column objects to sum
   */
  sumColumns(table, columnDetails) {
    if (!table || !columnDetails || !Array.isArray(columnDetails)) {
      if (this.debug) console.error('Invalid parameters for sumColumns:', { table, columnDetails });
      return;
    }
    
    this.recalculateSums(table, columnDetails);
    
    table.on('draw', () => {
      this.recalculateSums(table, columnDetails);
    });
  }
  
  /**
   * Recalculate sums for specified columns - POBOLJŠANA VERZIJA za hrvatski format
   * @param {Object} table - DataTables API instance
   * @param {Array} columnDetails - Array of column objects to sum
   */
  recalculateSums(table, columnDetails) {
    if (!table || !columnDetails || !Array.isArray(columnDetails)) {
      if (this.debug) console.error('Invalid parameters for recalculateSums:', { table, columnDetails });
      return;
    }
    
    Promise.all(columnDetails.map(columnInfo => {
      return new Promise(resolve => {
        const columnIndex = columnInfo.index;
        const dataType = columnInfo.type;
        
        const column = table.column(columnIndex, { search: 'applied' });
        
        const sum = column.data().reduce((acc, val) => {
          if (val !== null && val !== undefined && val !== '') {
            let numericValue;
            
            if (dataType === 'currency') {
              // POBOLJŠANO: Handle Croatian and American formats
              if (typeof val === 'string') {
                // Croatian format: 1.234,56 € -> 1234.56
                if (val.includes(',') && val.includes('.')) {
                  // Croatian: 1.234,56 €
                  const cleanValue = val.replace(/\./g, '').replace(',', '.').replace(/\s*€\s*/, '');
                  numericValue = parseFloat(cleanValue);
                } else if (val.includes(',')) {
                  // Simple comma decimal: 1234,56 €
                  const cleanValue = val.replace(',', '.').replace(/\s*€\s*/, '');
                  numericValue = parseFloat(cleanValue);
                } else {
                  // American format: 1234.56
                  const cleanValue = val.replace(/\s*€\s*/, '');
                  numericValue = parseFloat(cleanValue);
                }
              } else {
                numericValue = parseFloat(val);
              }
            } else {
              numericValue = parseFloat(val);
            }
            
            if (!isNaN(numericValue)) {
              return acc + numericValue;
            }
          }
          return acc;
        }, 0);
        
        const formattedSum = this.formatDataForFooter(sum, dataType);
        
        $(column.footer()).html(
          '<div class="dt-footer__sum">' + formattedSum + '</div>'
        );
        
        resolve();
      });
    })).catch(error => {
      console.error('Error calculating sums:', error);
    });
  }
  
  /**
   * Format data for footer based on type - POBOLJŠANO za hrvatski format
   * @param {number} value - Value to format
   * @param {string} dataType - Data type for formatting
   * @return {string} - Formatted value
   */
  formatDataForFooter(value, dataType) {
    // Use format service if available
    if (this.formatService) {
      switch (dataType) {
        case 'currency':
          return this.formatService.formatCurrency(value);
        case 'number':
          return this.formatService.formatNumber(value);
        default:
          return value.toString();
      }
    }
    
    // Fallback to local formatters
    if (this.sumFormatters[dataType]) {
      return this.sumFormatters[dataType](value);
    }
    
    return value.toString();
  }
  
  /**
   * Format currency values - POBOLJŠANI FALLBACK za hrvatski format
   * @param {number} value - Value to format
   * @return {string} - Formatted currency value
   */
  formatCurrency(value) {
    const floatValue = parseFloat(value);
    if (!isNaN(floatValue)) {
      // Croatian format: 1.234,56 €
      const formattedValue = floatValue.toLocaleString('hr-HR', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
      });
      return formattedValue + ' €';
    }
    return value;
  }
  
  /**
   * Format number values - POBOLJŠANI FALLBACK za hrvatski format
   * @param {number} value - Value to format
   * @return {string} - Formatted number value
   */
  formatNumber(value) {
    const floatValue = parseFloat(value);
    if (!isNaN(floatValue)) {
      return floatValue.toLocaleString('hr-HR');
    }
    return value;
  }
  
  /**
   * Escape HTML special characters to prevent XSS attacks
   * @param {string} html - String that may contain HTML
   * @return {string} - Escaped HTML string
   */
  escapeHtml(html) {
    if (html === null || html === undefined) {
      return '';
    }
    
    if (typeof html !== 'string') {
      html = String(html);
    }
    
    return html
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#39;');
  }
}