/**
 * ENHANCED Editor Service - REFACTORED VERSION
 * Dodana podrška za date picker umjesto dropdown-a za date kolumne
 * Minimalne optimizacije - service je bio izvrsno napisan
 */

export class EditorService {
  constructor(options = {}) {
    this.security = options.security || null;
    this.formatService = options.formatService || null;  // ADD THIS LINE
    this.tableConfig = null;
    this.columnMappings = null;
    this.activeEdits = new Map();
    this.useGlobalCsrf = options.useGlobalCsrf || false;
    this.debug = options.debug || false;
    
    this.eventListeners = new Set();
    
    // Bind methods
    this.setupInlineEditing = this.setupInlineEditing.bind(this);
    this.handleCellClick = this.handleCellClick.bind(this);
    this.setupDropdownEdit = this.setupDropdownEdit.bind(this);
    this.setupTextEdit = this.setupTextEdit.bind(this);
    this.setupDateEdit = this.setupDateEdit.bind(this);
    this.saveInlineEdit = this.saveInlineEdit.bind(this);
    this.clearEditorEvents = this.clearEditorEvents.bind(this);
    this.cancelEdit = this.cancelEdit.bind(this);
    this.closeAllActiveEdits = this.closeAllActiveEdits.bind(this);
    
    // Ensure date picker styles are loaded
    this.loadDatePickerStyles();
  }
  
  setTableConfig(config) {
    this.tableConfig = config;
  }
  
  setColumnMappings(mappings) {
    this.columnMappings = mappings;
  }
  
  /**
   * Učitaj CSS za date picker
   */
  loadDatePickerStyles() {
    if (!document.getElementById('dt-datepicker-styles')) {
      const style = document.createElement('style');
      style.id = 'dt-datepicker-styles';
      style.textContent = `
        .dt-datepicker {
          position: absolute;
          z-index: 9999;
          background: var(--dt-bg-primary, #ffffff);
          border: 1px solid var(--dt-border, #ddd);
          border-radius: 4px;
          box-shadow: 0 4px 12px rgba(0,0,0,0.15);
          padding: 8px;
          font-size: 12px;
          min-width: 250px;
        }
        
        .dt-datepicker-header {
          display: flex;
          justify-content: space-between;
          align-items: center;
          margin-bottom: 8px;
          padding: 4px;
        }
        
        .dt-datepicker-nav {
          background: none;
          border: none;
          cursor: pointer;
          font-size: 14px;
          padding: 4px 8px;
          border-radius: 3px;
          color: var(--dt-text-primary, #333);
        }
        
        .dt-datepicker-nav:hover {
          background: var(--dt-bg-hover, #f5f5f5);
        }
        
        .dt-datepicker-month-year {
          font-weight: bold;
          color: var(--dt-text-primary, #333);
        }
        
        .dt-datepicker-grid {
          display: grid;
          grid-template-columns: repeat(7, 1fr);
          gap: 1px;
          margin-bottom: 8px;
        }
        
        .dt-datepicker-day-header {
          text-align: center;
          font-weight: bold;
          padding: 4px;
          font-size: 10px;
          color: var(--dt-text-secondary, #666);
          background: var(--dt-bg-secondary, #f8f9fa);
        }
        
        .dt-datepicker-day {
          text-align: center;
          padding: 6px 4px;
          cursor: pointer;
          border-radius: 3px;
          transition: background-color 0.2s;
          color: var(--dt-text-primary, #333);
        }
        
        .dt-datepicker-day:hover {
          background: var(--dt-bg-hover, #e3f2fd);
        }
        
        .dt-datepicker-day.selected {
          background: var(--dt-primary, #4a6cf7);
          color: white;
        }
        
        .dt-datepicker-day.today {
          background: var(--dt-warning, #ffc107);
          color: var(--dt-text-primary, #333);
          font-weight: bold;
        }
        
        .dt-datepicker-day.other-month {
          color: var(--dt-text-muted, #999);
        }
        
        .dt-datepicker-actions {
          display: flex;
          justify-content: space-between;
          gap: 8px;
        }
        
        .dt-datepicker-btn {
          padding: 6px 12px;
          border: 1px solid var(--dt-border, #ddd);
          background: var(--dt-bg-primary, #fff);
          color: var(--dt-text-primary, #333);
          border-radius: 3px;
          cursor: pointer;
          font-size: 11px;
          flex: 1;
        }
        
        .dt-datepicker-btn:hover {
          background: var(--dt-bg-hover, #f5f5f5);
        }
        
        .dt-datepicker-btn.primary {
          background: var(--dt-primary, #4a6cf7);
          color: white;
          border-color: var(--dt-primary, #4a6cf7);
        }
        
        .dt-datepicker-btn.primary:hover {
          background: var(--dt-primary-dark, #3b5bdb);
        }
      `;
      document.head.appendChild(style);
    }
  }
  
  setupInlineEditing(table, mappings) {
    if (this.debug) {
      console.log('EditorService.setupInlineEditing', { tableId: table.table().node().id });
    }
    
    this.clearEditorEvents(table);
    
    const editableColumns = mappings.filter(column => 
      column.inlineediting && 
      column.inlineediting.enabled === 'yes'
    );
    
    if (this.debug) {
      console.log('Found columns with inline editing:', editableColumns.length);
    }
    
    if (editableColumns.length === 0) {
      console.log("No columns with enabled inline editing");
      return;
    }
    
    const tableNode = table.table().node();
    $(tableNode).on('click.inlineEdit', 'td.editable-cell', this.handleCellClick);
    this.eventListeners.add('click.inlineEdit');
    
    this.addEditableStyles(editableColumns);
    this.markEditableCells(table, editableColumns);
    
    table.on('draw.dt', () => {
      this.markEditableCells(table, editableColumns);
    });
  }
  
  addEditableStyles(editableColumns) {
    let styleElement = document.getElementById('datatables-inline-edit-styles');
    
    if (!styleElement) {
      styleElement = document.createElement('style');
      styleElement.id = 'datatables-inline-edit-styles';
      
      let css = `
        .editable-cell {
          cursor: pointer;
          position: relative;
        }
        
        .editable-cell:hover {
          background-color: var(--hover, #f3f4f6);
        }
        
        .editable-cell:hover::after {
          content: '✎';
          position: absolute;
          right: 5px;
          top: 50%;
          transform: translateY(-50%);
          font-size: 12px;
          opacity: 0.5;
        }
        
        .editable-cell.date-cell:hover::after {
          content: '📅';
        }
        
        .dt-editor__input,
        .dt-editor__select {
          width: 100%;
          padding: 2px 4px;
          border: 1px solid var(--dt-primary, #4a6cf7);
          border-radius: 3px;
          font-size: 12px;
          background-color: var(--dt-bg-primary, #ffffff);
          color: var(--dt-text-primary, #111827);
        }
      `;
      
      editableColumns.forEach(column => {
        css += `
          td.column-${column.index} {
            transition: background-color 0.2s;
          }
        `;
      });
      
      styleElement.textContent = css;
      document.head.appendChild(styleElement);
    }
  }
  
  markEditableCells(table, editableColumns) {
    editableColumns.forEach(column => {
      const originalIndex = column.index;
      let visibleIndex = null;
      
      table.columns().every(function(idx) {
        if (table.column(idx).dataSrc() == originalIndex) {
          visibleIndex = idx;
          return false;
        }
      });
      
      if (visibleIndex !== null && table.column(visibleIndex).visible()) {
        $(table.cells(null, visibleIndex).nodes()).each(function() {
          const $cell = $(this);
          $cell
            .addClass('editable-cell')
            .addClass('column-' + originalIndex)
            .attr('data-original-column', originalIndex);
          
          // Add date-cell class for date columns
          if (column.type === 'date' || column.type === 'datetime') {
            $cell.addClass('date-cell');
          }
          
          $cell.data('column-info', {
            index: originalIndex,
            name: column.sqlcolumnname,
            type: column.type || 'text',
            isDropdown: column.dropdown === 'yes',
            isDate: column.type === 'date' || column.type === 'datetime',
            editorUrl: column.inlineediting.url
          });
        });
      }
    });
  }

  closeAllActiveEdits() {
    this.activeEdits.forEach((active, key) => {
      const [recordId, columnName] = key.split('_');
      
      const $cell = $(`[data-original-column="${columnName}"][data-record-id="${recordId}"]`);
      if ($cell.length) {
        const table = $.fn.dataTable.Api($cell.closest('table'));
        const cell = table.cell($cell[0]);
        
        if (cell) {
          const originalValue = $cell.data('original-value');
          cell.data(originalValue).draw(false);
        }
      }
    });
    
    this.activeEdits.clear();
    $(document).off('click.editText').off('click.editDropdown').off('click.editDate');
    
    // Remove all date pickers
    $('.dt-datepicker').remove();
  }
  
  /**
   * Enhanced handleCellClick with date picker support
   */
  handleCellClick(e) {
    const $cellNode = $(e.currentTarget);

    this.closeAllActiveEdits();
    
    if ($cellNode.find('select, input').length > 0) return;
    
    const columnInfo = $cellNode.data('column-info');
    if (!columnInfo) {
      console.error("Missing column information");
      return;
    }
    
    const table = $.fn.dataTable.Api($cellNode.closest('table'));
    const cell = table.cell(e.currentTarget);
    if (!cell) {
      console.error("Cell not found");
      return;
    }
    
    const currentValue = cell.data();
    const row = table.row(e.currentTarget).data();
    if (!row) {
      console.error("Row not found");
      return;
    }
    
    const recordId = row[0];
    if (!recordId) {
      console.error("Record ID not found in first column");
      return;
    }
    
    const editKey = `${recordId}_${columnInfo.name}`;
    if (this.activeEdits.has(editKey)) {
      return;
    }
    
    this.activeEdits.set(editKey, true);
    
    // Check column type and call appropriate method
    if (columnInfo.isDate) {
      this.setupDateEdit(cell, currentValue, recordId, columnInfo, table);
    } else if (columnInfo.isDropdown) {
      this.setupDropdownEdit(cell, currentValue, recordId, columnInfo, table);
    } else {
      this.setupTextEdit(cell, currentValue, recordId, columnInfo);
    }
  }

  /**
   * Setup date picker for inline editing
   */
  setupDateEdit(cell, currentValue, recordId, columnInfo, table) {
    if (this.debug) {
      console.log("setupDateEdit", { columnName: columnInfo.name, currentValue });
    }
    
    $(document).off('click.editDate');
    
    const $cellNode = $(cell.node());
    const cellPosition = $cellNode.offset();
    
    // Parse current date
    let selectedDate = null;
    if (currentValue) {
      selectedDate = this.parseDateValue(currentValue);
    }
    
    if (!selectedDate) {
      selectedDate = new Date();
    }
    
    // Create date picker
    const $datePicker = this.createDatePicker(selectedDate, cellPosition);
    $('body').append($datePicker);
    
    // Position date picker
    this.positionDatePicker($datePicker, $cellNode);
    
    // Setup event handlers
    this.setupDatePickerEvents($datePicker, cell, currentValue, recordId, columnInfo);
    
    // Click outside handler
    setTimeout(() => {
      $(document).on('click.editDate', e => {
        if (!$(e.target).closest('.dt-datepicker').length) {
          this.cancelEdit(cell, currentValue, recordId, columnInfo.name);
          $datePicker.remove();
        }
      });
      this.eventListeners.add('click.editDate');
    }, 100);
  }
  
  /**
   * Create date picker HTML
   */
  createDatePicker(selectedDate, position) {
    const today = new Date();
    const currentMonth = selectedDate.getMonth();
    const currentYear = selectedDate.getFullYear();
    
    const monthNames = [
      'Siječanj', 'Veljača', 'Ožujak', 'Travanj', 'Svibanj', 'Lipanj',
      'Srpanj', 'Kolovoz', 'Rujan', 'Listopad', 'Studeni', 'Prosinac'
    ];
    
    const dayNames = ['Pon', 'Uto', 'Sri', 'Čet', 'Pet', 'Sub', 'Ned'];
    
    const $datePicker = $(`
      <div class="dt-datepicker">
        <div class="dt-datepicker-header">
          <button type="button" class="dt-datepicker-nav" data-action="prev-month">‹</button>
          <span class="dt-datepicker-month-year">${monthNames[currentMonth]} ${currentYear}</span>
          <button type="button" class="dt-datepicker-nav" data-action="next-month">›</button>
        </div>
        <div class="dt-datepicker-grid">
          ${dayNames.map(day => `<div class="dt-datepicker-day-header">${day}</div>`).join('')}
        </div>
        <div class="dt-datepicker-actions">
          <button type="button" class="dt-datepicker-btn" data-action="today">Danas</button>
          <button type="button" class="dt-datepicker-btn" data-action="cancel">Odustani</button>
          <button type="button" class="dt-datepicker-btn primary" data-action="select">Odaberi</button>
        </div>
      </div>
    `);
    
    // Store selected date
    $datePicker.data('selectedDate', selectedDate);
    $datePicker.data('currentMonth', currentMonth);
    $datePicker.data('currentYear', currentYear);
    
    // Generate calendar days
    this.generateCalendarDays($datePicker, selectedDate, currentMonth, currentYear, today);
    
    return $datePicker;
  }
  
  /**
   * Generate days in calendar
   */
  generateCalendarDays($datePicker, selectedDate, month, year, today) {
    const $grid = $datePicker.find('.dt-datepicker-grid');
    
    // Remove existing days (keep headers)
    $grid.find('.dt-datepicker-day').remove();
    
    const firstDay = new Date(year, month, 1);
    const lastDay = new Date(year, month + 1, 0);
    const startDate = new Date(firstDay);
    
    // Adjust to Monday start (Croatian standard)
    const dayOfWeek = (firstDay.getDay() + 6) % 7; // Convert Sunday=0 to Monday=0
    startDate.setDate(startDate.getDate() - dayOfWeek);
    
    // Generate 42 days (6 weeks)
    for (let i = 0; i < 42; i++) {
      const currentDate = new Date(startDate);
      currentDate.setDate(startDate.getDate() + i);
      
      const isCurrentMonth = currentDate.getMonth() === month;
      const isSelected = selectedDate && 
                        currentDate.getDate() === selectedDate.getDate() &&
                        currentDate.getMonth() === selectedDate.getMonth() &&
                        currentDate.getFullYear() === selectedDate.getFullYear();
      const isToday = currentDate.getDate() === today.getDate() &&
                     currentDate.getMonth() === today.getMonth() &&
                     currentDate.getFullYear() === today.getFullYear();
      
      const dayClasses = ['dt-datepicker-day'];
      if (!isCurrentMonth) dayClasses.push('other-month');
      if (isSelected) dayClasses.push('selected');
      if (isToday) dayClasses.push('today');
      
      const $day = $(`<div class="${dayClasses.join(' ')}" data-date="${currentDate.getFullYear()}-${String(currentDate.getMonth() + 1).padStart(2, '0')}-${String(currentDate.getDate()).padStart(2, '0')}">${currentDate.getDate()}</div>`);
      $grid.append($day);
    }
  }
  
  /**
   * Position date picker
   */
  positionDatePicker($datePicker, $cellNode) {
    const cellOffset = $cellNode.offset();
    const cellHeight = $cellNode.outerHeight();
    const pickerHeight = $datePicker.outerHeight();
    const windowHeight = $(window).height();
    const scrollTop = $(window).scrollTop();
    
    let top = cellOffset.top + cellHeight + 5;
    
    // Check if picker would go below viewport
    if (top + pickerHeight > scrollTop + windowHeight) {
      top = cellOffset.top - pickerHeight - 5;
    }
    
    $datePicker.css({
      position: 'absolute',
      top: top,
      left: cellOffset.left,
      zIndex: 9999
    });
  }
  
  /**
   * Setup date picker event handlers
   */
  setupDatePickerEvents($datePicker, cell, originalValue, recordId, columnInfo) {
    // Navigation buttons
    $datePicker.on('click', '.dt-datepicker-nav', (e) => {
      const action = $(e.target).data('action');
      let currentMonth = $datePicker.data('currentMonth');
      let currentYear = $datePicker.data('currentYear');
      const selectedDate = $datePicker.data('selectedDate');
      
      if (action === 'prev-month') {
        currentMonth--;
        if (currentMonth < 0) {
          currentMonth = 11;
          currentYear--;
        }
      } else if (action === 'next-month') {
        currentMonth++;
        if (currentMonth > 11) {
          currentMonth = 0;
          currentYear++;
        }
      }
      
      $datePicker.data('currentMonth', currentMonth);
      $datePicker.data('currentYear', currentYear);
      
      // Update header
      const monthNames = [
        'Siječanj', 'Veljača', 'Ožujak', 'Travanj', 'Svibanj', 'Lipanj',
        'Srpanj', 'Kolovoz', 'Rujan', 'Listopad', 'Studeni', 'Prosinac'
      ];
      $datePicker.find('.dt-datepicker-month-year').text(`${monthNames[currentMonth]} ${currentYear}`);
      
      // Regenerate days
      this.generateCalendarDays($datePicker, selectedDate, currentMonth, currentYear, new Date());
    });
    
    // Day selection
    $datePicker.on('click', '.dt-datepicker-day:not(.other-month)', (e) => {
      const dateStr = $(e.target).data('date');
      const selectedDate = new Date(dateStr);
      
      // Update selection
      $datePicker.find('.dt-datepicker-day').removeClass('selected');
      $(e.target).addClass('selected');
      $datePicker.data('selectedDate', selectedDate);
    });
    
    // Action buttons
    $datePicker.on('click', '.dt-datepicker-btn', (e) => {
      const action = $(e.target).data('action');
      
      if (action === 'today') {
        const today = new Date();
        $datePicker.find('.dt-datepicker-day').removeClass('selected');
        $datePicker.find(`[data-date="${today.getFullYear()}-${String(today.getMonth() + 1).padStart(2, '0')}-${String(today.getDate()).padStart(2, '0')}"]`).addClass('selected');
        $datePicker.data('selectedDate', today);
        
      } else if (action === 'cancel') {
        this.cancelEdit(cell, originalValue, recordId, columnInfo.name);
        $datePicker.remove();
        
      } else if (action === 'select') {
        const selectedDate = $datePicker.data('selectedDate');
        if (selectedDate) {
          const formattedDate = this.formatDateForSave(selectedDate);
          
          if (this.debug) {
            console.log(`Saving date for ${columnInfo.name}:`, formattedDate);
          }
          
          this.saveInlineEdit(
            recordId,
            columnInfo.name,
            formattedDate,
            columnInfo.editorUrl,
            success => {
              if (success) {
                const displayDate = this.formatDateForDisplay(selectedDate);
                cell.data(displayDate).draw(false);
              } else {
                this.cancelEdit(cell, originalValue, recordId, columnInfo.name);
              }
              $datePicker.remove();
            }
          );
        }
      }
    });
  }
  
  /**
   * Parse date value from different formats
   */
  parseDateValue(value) {
    // Use FormatService if available
    if (this.formatService) {
      return this.formatService.parseDate(value);
    }
    
    // Simple fallback
    if (!value) return null;
    const date = new Date(value);
    return isNaN(date.getTime()) ? null : date;
  }
  
  /**
   * Format date for saving (YYYY-MM-DD)
   */
  formatDateForSave(date) {
    // Use FormatService if available
    if (this.formatService) {
      return this.formatService.formatDateForSave(date);
    }
    
    // Simple fallback
    if (!date || !(date instanceof Date)) return '';
    return date.toISOString().split('T')[0]; // YYYY-MM-DD
  }
  
  /**
   * Format date for display (DD-MM-YYYY)
   */
  formatDateForDisplay(date) {
    // Use FormatService if available
    if (this.formatService) {
      return this.formatService.formatDate(date);
    }
    
    // Simple fallback
    if (!date || !(date instanceof Date)) return '';
    return date.toLocaleDateString('hr-HR'); // DD.MM.YYYY
  }

  // Existing methods for dropdown and text editing...
  setupDropdownEdit(cell, currentValue, recordId, columnInfo, table) {
    if (this.debug) {
      console.log("setupDropdownEdit", { columnName: columnInfo.name });
    }
    
    $(document).off('click.editDropdown');
    
    let uniqueValues = [];
    const response = table.ajax.json();
    
    if (response && response.uniqueValues && response.uniqueValues[columnInfo.name]) {
      uniqueValues = response.uniqueValues[columnInfo.name];
      
      if (this.debug) {
        console.log(`Retrieved unique values for ${columnInfo.name}:`, uniqueValues.length);
      }
      
      const select = $('<select class="form-control form-control-sm dt-editor__select">');
      uniqueValues.forEach(value => {
        if (value !== null && value !== "") {
          const option = $('<option>')
            .val(value)
            .text(value)
            .prop('selected', value === currentValue);
          select.append(option);
        }
      });
      
      const $cellNode = $(cell.node());
      $cellNode.empty().append(select);
      
      setTimeout(() => {
        $(document).on('click.editDropdown', e => {
          if (!$(e.target).closest('select').length) {
            this.cancelEdit(cell, currentValue, recordId, columnInfo.name);
          }
        });
        this.eventListeners.add('click.editDropdown');
      }, 100);
      
      select.on('change', () => {
        const newValue = select.val();
        
        if (this.debug) {
          console.log(`Saving new value for ${columnInfo.name}:`, newValue);
        }
        
        this.saveInlineEdit(
          recordId, 
          columnInfo.name, 
          newValue, 
          columnInfo.editorUrl,
          success => {
            if (success) {
              cell.data(newValue).draw(false);
            } else {
              this.cancelEdit(cell, currentValue, recordId, columnInfo.name);
            }
          }
        );
      });
      
      select.on('keypress', e => {
        if (e.which === 13) {
          e.preventDefault();
          const newValue = select.val();
          
          if (this.debug) {
            console.log(`Saving new value from keypress for ${columnInfo.name}:`, newValue);
          }
          
          this.saveInlineEdit(
            recordId, 
            columnInfo.name, 
            newValue, 
            columnInfo.editorUrl,
            success => {
              if (success) {
                cell.data(newValue).draw(false);
              } else {
                this.cancelEdit(cell, currentValue, recordId, columnInfo.name);
              }
            }
          );
        } else if (e.which === 27) {
          this.cancelEdit(cell, currentValue, recordId, columnInfo.name);
        }
      });
      
      select.focus();
    } else {
      console.warn(`No unique values for column ${columnInfo.name}`);
      this.cancelEdit(cell, currentValue, recordId, columnInfo.name);
    }
  }

  setupTextEdit(cell, currentValue, recordId, columnInfo) {
    if (this.debug) {
      console.log("setupTextEdit", { columnName: columnInfo.name });
    }
    
    $(document).off('click.editText');
    
    const input = $('<input type="text" class="form-control form-control-sm dt-editor__input">')
      .val(currentValue)
      .css('width', '100%');
    
    const $cellNode = $(cell.node());
    $cellNode.empty().append(input);
    
    setTimeout(() => {
      $(document).on('click.editText', e => {
        if (!$(e.target).closest('input').length) {
          this.cancelEdit(cell, currentValue, recordId, columnInfo.name);
        }
      });
      this.eventListeners.add('click.editText');
    }, 100);
    
    input.on('keypress', e => {
      if (e.which === 13) {
        const newValue = input.val();
        
        if (this.debug) {
          console.log(`Saving new value for ${columnInfo.name}:`, newValue);
        }
        
        this.saveInlineEdit(
          recordId, 
          columnInfo.name, 
          newValue, 
          columnInfo.editorUrl,
          success => {
            if (success) {
              cell.data(newValue).draw(false);
            } else {
              this.cancelEdit(cell, currentValue, recordId, columnInfo.name);
            }
          }
        );
      } else if (e.which === 27) {
        this.cancelEdit(cell, currentValue, recordId, columnInfo.name);
      }
    });
    
    input.focus();
  }

  cancelEdit(cell, originalValue, recordId, columnName) {
    cell.data(originalValue).draw(false);
    $(document).off('click.editText').off('click.editDropdown').off('click.editDate');
    this.activeEdits.delete(`${recordId}_${columnName}`);
    
    // Remove all date pickers
    $('.dt-datepicker').remove();
  }

  saveInlineEdit(id, field, value, url, callback) {
    if (this.debug) {
      console.log("saveInlineEdit", { id, field, url });
    }
    
    const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
    
    const data = {
      id: id,
      field: field,
      value: value
    };
    
    const fullUrl = url.startsWith('http') || url.startsWith('/') ? url : publicUrl + '/' + url;
    
    if (this.debug) {
      console.log(`Sending AJAX request to ${fullUrl}`);
    }
    
    if (window.fetchWithCsrf) {
      window.fetchWithCsrf(fullUrl, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(data)
      })
      .then(response => response.json())
      .then(response => {
        if (this.debug) {
          console.log("AJAX response:", response);
        }
        
        if (response.success) {
          callback(true);
        } else {
          alert('Error saving changes: ' + (response.message || 'Unknown error'));
          callback(false);
        }
      })
      .catch(error => {
        console.error('Ajax error:', error);
        alert('Error saving changes. Please check the console for details.');
        callback(false);
      });
    } else {
      $.ajax({
        url: fullUrl,
        method: 'POST',
        data: data,
        success: response => {
          if (this.debug) {
            console.log("AJAX response:", response);
          }
          
          if (response.success) {
            callback(true);
          } else {
            alert('Error saving changes: ' + (response.message || 'Unknown error'));
            callback(false);
          }
        },
        error: (xhr, status, error) => {
          console.error('Ajax error:', status, error);
          console.log('XHR object:', xhr);
          alert('Error saving changes. Please check the console for details.');
          callback(false);
        }
      });
    }
  }

  clearEditorEvents(table) {
    const tableNode = table.table().node();
    $(tableNode).off('click.inlineEdit');
    
    this.eventListeners.forEach(eventName => {
      $(document).off(eventName);
    });
    
    this.eventListeners.clear();
    this.activeEdits.clear();
    
    // Remove all date pickers
    $('.dt-datepicker').remove();
  }
  
  /**
   * Clean up resources when destroying
   */
  destroy() {
    this.clearEditorEvents({
      table: () => ({ node: () => document })
    });
    
    // Remove date picker styles
    const styleElement = document.getElementById('dt-datepicker-styles');
    if (styleElement) {
      styleElement.remove();
    }
    
    // Remove editable styles
    const editableStyleElement = document.getElementById('datatables-inline-edit-styles');
    if (editableStyleElement) {
      editableStyleElement.remove();
    }
  }
}