/**
 * DataTables Theme Service - REFACTORED VERSION
 * Preuzeo je SAVU odgovornost za theme i column visibility
 * Uklonjen duplikatni kod iz core.js
 */

export class ThemeService {
  /**
   * Creates a new ThemeService instance
   * @param {Object} options - Configuration options 
   */
  constructor(options = {}) {
    this.currentTheme = document.documentElement.getAttribute('data-theme') || 'dark';
    this.columnVisibilityControls = new Map();
    this.debug = options?.debug || false;
    
    // Default theme configuration
    this.themeConfig = {
      light: {
        primaryColor: 'var(--accent-primary, #4a6cf7)',
        textColor: 'var(--text-primary, #111827)',
        backgroundColor: 'var(--bg-primary, #ffffff)',
        borderColor: 'var(--border, #e5e7eb)'
      },
      dark: {
        primaryColor: 'var(--accent-primary, #4a6cf7)',
        textColor: 'var(--text-primary, #f9fafb)',
        backgroundColor: 'var(--bg-primary, #1f2937)',
        borderColor: 'var(--border, #374151)'
      }
    };
    
    // Set for tracking added event listeners
    this.eventListeners = new Set();
    
    // Bind methods to preserve this context
    this.addColumnVisibilityControl = this.addColumnVisibilityControl.bind(this);
    this.applyThemeStyling = this.applyThemeStyling.bind(this);
    this.applyCompactStyles = this.applyCompactStyles.bind(this);
    this.fixFixedColumnsOnThemeChange = this.fixFixedColumnsOnThemeChange.bind(this);
    this.handleThemeChange = this.handleThemeChange.bind(this);
    this.setupColumnVisibilityHandling = this.setupColumnVisibilityHandling.bind(this);
    this.restoreColumnVisibilityState = this.restoreColumnVisibilityState.bind(this);
    this.saveColumnVisibilityState = this.saveColumnVisibilityState.bind(this);
    
    // Set up listener for theme changes at document level
    this.setupThemeChangeListener();
  }
  
  /**
   * Set up listener for theme changes
   */
  setupThemeChangeListener() {
    // Listen for theme changes through data-theme attribute
    const observer = new MutationObserver(mutations => {
      mutations.forEach(mutation => {
        if (mutation.attributeName === 'data-theme') {
          const newTheme = document.documentElement.getAttribute('data-theme') || 'light';
          if (newTheme !== this.currentTheme) {
            this.currentTheme = newTheme;
            this.handleThemeChange();
          }
        }
      });
    });
    
    observer.observe(document.documentElement, { attributes: true });
    
    // Also listen for global theme change event
    document.addEventListener('themeChange', this.handleThemeChange);
  }
  
  /**
   * Handle theme changes
   */
  handleThemeChange() {
    // Emit theme change event for other components
    const event = new CustomEvent('themechange', { 
      detail: { theme: this.currentTheme }
    });
    document.dispatchEvent(event);
    
    if (this.debug) {
      console.log(`Theme changed to: ${this.currentTheme}`);
    }
  }
  
  /**
   * Apply theme styling to DataTables
   * @param {Object} table - DataTables instance
   */
  applyThemeStyling(table) {
    if (!table) return;
    
    // Get configuration for current theme
    const theme = this.currentTheme;
    const themeConfig = this.themeConfig[theme] || this.themeConfig.light;
    
    // Apply styles to table root element
    const tableContainer = $(table.table().container());
    tableContainer.css('--dt-theme', theme);
    
    // Force redraw of table to apply styles
    table.draw(false);
    
    // Apply special styles for fixed columns
    this.fixFixedColumnsOnThemeChange(table);
  }
  
  /**
   * Apply compact styles for table
   */
  applyCompactStyles() {
    // Add CSS class for compact style if it doesn't exist
    if (!document.getElementById('dt-compact-styles')) {
      const style = document.createElement('style');
      style.id = 'dt-compact-styles';
      style.textContent = `
        .dataTable.compact thead th, 
        .dataTable.compact thead td {
          padding: 2px 4px !important;
          font-size: 12px !important;
        }
        
        .dataTable.compact tbody td {
          padding: 1px 4px !important;
          font-size: 11px !important;
          height: 22px !important;
        }
        
        .dataTable.compact .dt-action button.btn {
          padding: 0px 4px !important;
          font-size: 10px !important;
        }
        
        .dataTable.compact .dt-footer__sum {
          font-size: 11px !important;
        }
      `;
      document.head.appendChild(style);
    }
    
    // Apply compact class to all DataTables tables
    $('.dataTable').addClass('compact');
  }
  
  /**
   * PREUZETO IZ CORE: Restore column visibility state
   * @param {Object} api - DataTables API instance
   */
  restoreColumnVisibilityState(api) {
    // Add safety checks
    if (!api || !api.table || typeof api.table !== 'function') {
      if (this.debug) {
        console.warn('Invalid API object in restoreColumnVisibilityState');
      }
      return;
    }
    
    const table = api.table();
    if (!table || !table.node) {
      if (this.debug) {
        console.warn('Invalid table object in restoreColumnVisibilityState');
      }
      return;
    }
    
    // Wait for columns to be fully initialized
    setTimeout(() => {
      try {
        // Double check that columns are available
        const columnsCount = api.columns().count();
        if (columnsCount === 0) {
          if (this.debug) {
            console.warn('No columns found, skipping column visibility restore');
          }
          return;
        }
        
        // Get table config for route name
        const tableId = table.node().id || 'default';
        const routeName = this.getRouteNameFromStorage(tableId);
        
        if (!routeName) {
          if (this.debug) {
            console.warn('No route name found for column visibility restore');
          }
          return;
        }
        
        const savedState = localStorage.getItem('DataTables_ColVis_' + routeName);
        
        if (savedState) {
          const visibilityState = JSON.parse(savedState);
          
          if (this.debug) {
            console.log('Restoring column visibility state:', visibilityState);
          }
          
          // Validate column indexes before applying
          for (let colIndex in visibilityState) {
            const numIndex = parseInt(colIndex);
            
            // Check if column index is valid
            if (numIndex >= 0 && numIndex < columnsCount) {
              try {
                const column = api.column(numIndex);
                
                // Additional safety check
                if (column && typeof column.visible === 'function') {
                  column.visible(visibilityState[colIndex]);
                } else {
                  if (this.debug) {
                    console.warn(`Column ${numIndex} does not have visible() method`);
                  }
                }
              } catch (colError) {
                if (this.debug) {
                  console.warn(`Error setting visibility for column ${numIndex}:`, colError);
                }
              }
            } else {
              if (this.debug) {
                console.warn(`Invalid column index ${numIndex}, skipping`);
              }
            }
          }
          
          if (this.debug) {
            console.log('Column visibility state restored successfully');
          }
        }
      } catch (e) {
        console.error('Error restoring column visibility:', e);
      }
    }, 100); // Delay to ensure columns are fully initialized
  }
  
  /**
   * PREUZETO IZ CORE: Setup column visibility handling
   * @param {Object} api - DataTables API instance
   * @param {Array} searchColumns - Search column indexes
   * @param {Array} sumColumns - Sum column details
   */
  setupColumnVisibilityHandling(api, searchColumns, sumColumns) {
    api.on('column-visibility.dt', (e, settings, column, state) => {
      if (this.debug) {
        console.log(`Column ${column} visibility changed to: ${state}`);
      }
      
      // Update filters when column visibility changes
      if (window.filterService) {
        setTimeout(() => {
          const $filterRow = $(api.table().header()).find('tr.filters');
          
          if ($filterRow.length) {
            // Hide/show filter cells based on column visibility
            api.columns().every(function(colIdx) {
              const isVisible = this.visible();
              const $filterCell = $filterRow.find('th').eq(colIdx);
              
              if (isVisible) {
                $filterCell.show();
              } else {
                $filterCell.hide().empty();
              }
            });
          }
          
          // Force column width recalculation
          api.columns.adjust();
          
          // Reset scrolling
          const $scrollBody = $(api.table().container()).find('.dataTables_scrollBody');
          if ($scrollBody.length) {
            const currentScrollLeft = $scrollBody.scrollLeft();
            $scrollBody.scrollLeft(0).scrollLeft(currentScrollLeft);
          }
          
          if (this.debug) {
            console.log('Filter row updated and columns adjusted after visibility change');
          }
        }, 100);
      }
      
      // Recalculate sums if column service is available
      if (sumColumns && sumColumns.length > 0 && window.columnService) {
        setTimeout(() => {
          window.columnService.recalculateSums(api, sumColumns);
        }, 150);
      }
      
      // Setup inline editing if editor service is available
      if (window.editorService) {
        setTimeout(() => {
          window.editorService.setupInlineEditing(api, this.getColumnMappings());
        }, 150);
      }
      
      // Final adjustment and draw
      setTimeout(() => {
        api.columns.adjust().draw(false);
        
        // Fix for fixed columns
        if ($(api.table().container()).find('.DTFC_LeftWrapper, .DTFC_RightWrapper').length) {
          $(window).trigger('resize');
        }
      }, 200);
      
      // Save column visibility state
      this.saveColumnVisibilityState(api);
    });
  }
  
  /**
   * PREUZETO IZ CORE: Save column visibility state
   * @param {Object} api - DataTables API instance
   */
  saveColumnVisibilityState(api) {
    const visibilityState = {};
    api.columns().every(function(index) {
      visibilityState[index] = this.visible();
    });
    
    const tableId = api.table().node().id || 'default';
    const routeName = this.getRouteNameFromStorage(tableId);
    
    if (routeName) {
      localStorage.setItem('DataTables_ColVis_' + routeName, JSON.stringify(visibilityState));
      
      if (this.debug) {
        console.log('Column visibility state saved:', visibilityState);
      }
    }
  }
  
  /**
   * Add column visibility control
   * @param {Object} api - DataTables API instance
   * @return {jQuery} - jQuery container object
   */
  addColumnVisibilityControl(api) {
    if (!api) return;
    
    // Generate unique ID for this table if it doesn't have one
    const tableId = api.table().node().id || 'dt-' + Math.random().toString(36).substr(2, 9);
    
    // Check if control already exists for this table
    if (this.columnVisibilityControls.has(tableId)) {
      this.removeColumnVisibilityControl(tableId);
    }
    
    // Create container for button
    const container = $('<div class="dt-button-container"></div>');
    
    // Check where to add control
    if ($('.dt-buttons').length) {
      // Add to beginning of existing button container
      container.prependTo('.dt-buttons');
    } else {
      // Alternatively, add to beginning of wrapper
      container.prependTo($(api.table().container()).find('.dataTables_wrapper'));
    }
    
    // Create column visibility button
    const button = $('<button class="dt-button--colvis" type="button"><i class="fas fa-columns"></i></button>');
    button.attr('title', 'Show/Hide Columns');
    container.append(button);
    
    // Create dropdown
    const dropdown = $('<div class="dt-colvis__dropdown"></div>');
    container.append(dropdown);
    
    // Populate dropdown with column options
    this.populateColumnVisibilityDropdown(api, dropdown);
    
    // Set handler for button click
    button.on('click', e => {
      e.stopPropagation();
      dropdown.toggle();
      
      // Position dropdown below button
      const buttonPos = button.position();
      dropdown.css({
        'top': buttonPos.top + button.outerHeight() + 5,
        'left': buttonPos.left
      });
    });
    
    // Set global handler for closing dropdown
    const clickOutsideHandler = e => {
      if (!dropdown.is(e.target) && dropdown.has(e.target).length === 0 && 
          !button.is(e.target) && button.has(e.target).length === 0) {
        dropdown.hide();
      }
    };
    
    $(document).on('click.dtColvis', clickOutsideHandler);
    this.eventListeners.add('click.dtColvis');
    
    // Save reference to control
    this.columnVisibilityControls.set(tableId, {
      container,
      button,
      dropdown,
      api
    });
    
    // Listen for redraw events to update control
    api.on('column-visibility.dt', () => {
      this.updateColumnVisibilityDropdown(api, dropdown);
    });
    
    return container;
  }
  
  /**
   * Populate dropdown for column visibility
   * @param {Object} api - DataTables API instance
   * @param {jQuery} dropdown - jQuery dropdown object
   */
  populateColumnVisibilityDropdown(api, dropdown) {
    dropdown.empty();
    
    // Add "Select All" control
    const allContainer = $('<div class="dt-colvis__item dt-colvis__item--all"></div>');
    const allCheckbox = $('<input type="checkbox" class="dt-colvis__checkbox-all" />').prop('checked', true);
    const allLabel = $('<label></label>').text(' Show All');
    
    allContainer.append(allCheckbox).append(allLabel);
    dropdown.append(allContainer);
    
    // Add separator
    dropdown.append('<div class="dt-colvis__separator"></div>');
    
    // Counter for visible columns
    let visibleCount = 0;
    let totalColumns = 0;
    
    // Add options for each column (except first if it's control)
    api.columns().every(function(index) {
      if (index !== 0) { // Skip first column if it's control
        const column = this;
        const title = $(column.header()).text().trim();
        
        if (title) {
          totalColumns++;
          
          if (column.visible()) {
            visibleCount++;
          }
          
          const item = $('<div class="dt-colvis__item"></div>');
          const checkbox = $('<input type="checkbox" class="dt-colvis__checkbox" />')
            .prop('checked', column.visible())
            .data('column-index', index);
          
          const label = $('<label></label>').text(' ' + title);
          
          item.append(checkbox).append(label);
          dropdown.append(item);
          
          // Handle checkbox change
          checkbox.on('change', function() {
            const isVisible = $(this).prop('checked');
            column.visible(isVisible);
            
            // Update "select all" checkbox
            if (isVisible) {
              visibleCount++;
            } else {
              visibleCount--;
            }
            
            allCheckbox.prop('checked', visibleCount === totalColumns);
          });
        }
      }
    });
    
    // Handle "select all" checkbox
    allCheckbox.on('change', function() {
      const isChecked = $(this).prop('checked');
      
      // Set all checkboxes to same value
      dropdown.find('.dt-colvis__checkbox').each(function() {
        const columnIndex = $(this).data('column-index');
        if ($(this).prop('checked') !== isChecked) {
          $(this).prop('checked', isChecked);
          api.column(columnIndex).visible(isChecked);
        }
      });
      
      visibleCount = isChecked ? totalColumns : 0;
    });
  }
  
  /**
   * Update dropdown for column visibility
   * @param {Object} api - DataTables API instance
   * @param {jQuery} dropdown - jQuery dropdown object
   */
  updateColumnVisibilityDropdown(api, dropdown) {
    let visibleCount = 0;
    let totalColumns = 0;
    
    dropdown.find('.dt-colvis__checkbox').each(function() {
      const columnIndex = $(this).data('column-index');
      const column = api.column(columnIndex);
      
      if (column) {
        totalColumns++;
        const isVisible = column.visible();
        $(this).prop('checked', isVisible);
        
        if (isVisible) {
          visibleCount++;
        }
      }
    });
    
    // Update "select all" checkbox
    dropdown.find('.dt-colvis__checkbox-all').prop('checked', visibleCount === totalColumns);
  }
  
  /**
   * Remove column visibility control
   * @param {string} tableId - Table ID
   */
  removeColumnVisibilityControl(tableId) {
    const control = this.columnVisibilityControls.get(tableId);
    
    if (control) {
      // Remove event listeners
      control.button.off('click');
      control.dropdown.find('.dt-colvis__checkbox, .dt-colvis__checkbox-all').off('change');
      
      // Remove DOM elements
      control.container.remove();
      
      // Remove from map
      this.columnVisibilityControls.delete(tableId);
    }
  }
  
  /**
   * Fix fixed columns display after theme change
   * @param {Object} table - DataTables instance
   */
  fixFixedColumnsOnThemeChange(table) {
    if (!table) return;
    
    // Delay execution to allow DOM to update
    setTimeout(() => {
      try {
        // Get configuration for current theme
        const themeConfig = this.themeConfig[this.currentTheme] || this.themeConfig.light;
        
        // Apply styles to fixed columns
        
        // 1. Fix fixed column headers
        $('.DTFC_LeftHeadWrapper th, .DTFC_RightHeadWrapper th').css({
          'background-color': themeConfig.backgroundColor,
          'color': themeConfig.primaryColor,
          'border-bottom': `1px solid ${themeConfig.primaryColor}`
        });
        
        // 2. Fix fixed column body - apply to rows based on odd/even
        $('.DTFC_LeftBodyWrapper tr:nth-child(odd) td, .DTFC_RightBodyWrapper tr:nth-child(odd) td')
          .css('background-color', `rgba(${this.hexToRgb(themeConfig.backgroundColor)}, 0.4)`);
        
        $('.DTFC_LeftBodyWrapper tr:nth-child(even) td, .DTFC_RightBodyWrapper tr:nth-child(even) td')
          .css('background-color', themeConfig.backgroundColor);
        
        // 3. Ensure fixed layer shadows have appropriate z-index
        $('.DTFC_LeftWrapper').css('z-index', '200');
        $('.DTFC_RightWrapper').css('z-index', '200');
        
        // 4. Sometimes need to "nudge" the scroll to activate fixed columns
        const $scrollBody = $('.dataTables_scrollBody');
        const currentScrollLeft = $scrollBody.scrollLeft();
        $scrollBody.scrollLeft(currentScrollLeft + 1).scrollLeft(currentScrollLeft);
      } catch (error) {
        console.error('Error fixing fixed columns on theme change:', error);
      }
    }, 50);
  }
  
  /**
   * Convert hex color to RGB format
   * @param {string} hex - Hex color
   * @return {string} - RGB values as string (e.g. "255, 255, 255")
   */
  hexToRgb(hex) {
    // If already CSS variable, return default
    if (hex.startsWith('var(--')) {
      return '255, 255, 255';
    }
    
    // Remove # if present
    hex = hex.replace('#', '');
    
    // Expand shorthand (e.g. #FFF to #FFFFFF)
    if (hex.length === 3) {
      hex = hex[0] + hex[0] + hex[1] + hex[1] + hex[2] + hex[2];
    }
    
    // Convert to RGB
    const r = parseInt(hex.substring(0, 2), 16);
    const g = parseInt(hex.substring(2, 4), 16);
    const b = parseInt(hex.substring(4, 6), 16);
    
    return isNaN(r) || isNaN(g) || isNaN(b) ? '255, 255, 255' : `${r}, ${g}, ${b}`;
  }
  
  /**
   * Set custom theme
   * @param {string} themeName - Theme name
   * @param {Object} themeConfig - Theme configuration
   */
  setCustomTheme(themeName, themeConfig) {
    if (!themeName || !themeConfig) return;
    
    this.themeConfig[themeName] = {
      ...this.themeConfig.light, // Use light as base
      ...themeConfig
    };
    
    if (this.debug) {
      console.log(`Custom theme "${themeName}" registered:`, this.themeConfig[themeName]);
    }
  }
  
  /**
   * UTILITY METHODS
   */
  
  /**
   * Get route name from storage or table config
   * @param {string} tableId - Table ID
   * @return {string} - Route name
   */
  getRouteNameFromStorage(tableId) {
    // Try to get from stored table config
    const configKey = 'DataTables_Config_' + tableId;
    const storedConfig = localStorage.getItem(configKey);
    
    if (storedConfig) {
      try {
        const config = JSON.parse(storedConfig);
        return config.routeName;
      } catch (e) {
        // Ignore parsing errors
      }
    }
    
    // Fallback to table ID
    return tableId;
  }
  
  /**
   * Get column mappings (placeholder - should be injected)
   * @return {Array} - Column mappings
   */
  getColumnMappings() {
    // This should be injected from the main initialization
    // For now, return empty array
    return window.currentColumnMappings || [];
  }
  
  /**
   * Clean up resources when destroying
   */
  destroy() {
    // Remove event listeners
    document.removeEventListener('themeChange', this.handleThemeChange);
    
    this.eventListeners.forEach(eventName => {
      $(document).off(eventName);
    });
    
    // Remove column visibility controls
    this.columnVisibilityControls.forEach((control, tableId) => {
      this.removeColumnVisibilityControl(tableId);
    });
    
    // Reset state
    this.eventListeners.clear();
    this.columnVisibilityControls.clear();
  }
}