<?php
/**
 * Simple Asset Compilation Script
 * 
 * This script is a simplified version for quick asset compilation during development.
 * 
 * Usage:
 *   php assets-compile.php                   - Compile common assets
 *   php assets-compile.php file.css          - Compile a specific file
 *   php assets-compile.php --clear           - Clear asset cache
 *   php assets-compile.php --watch           - Watch for changes (requires inotify extension)
 */

// Define root directory constant (where assets-compile.php is located)
define('ROOT_DIR', __DIR__);

// Define APP_ROOT for configuration paths
define('APP_ROOT', ROOT_DIR);

// Include the autoloader
require_once ROOT_DIR . '/vendor/autoload.php';

// Load Config classes for environment setup
require_once ROOT_DIR . '/app/baseKRIZAN/Bootstrap/initEnvironment.php';

// Load .env
$configPath = ROOT_DIR . '/app/config/';
$dotenv = \Dotenv\Dotenv::createImmutable($configPath);
$dotenv->load();

// Set CLI environment indicator
$_SERVER['CLI_SCRIPT'] = true;

// Load Config
\baseKRIZAN\Config\Config::load([ROOT_DIR . '/app/config/env_config.php']);

// Process command line arguments
$clearCache = false;
$watchMode = false;
$specificFile = null;

// Skip the script name
$args = array_slice($argv, 1);

foreach ($args as $arg) {
    if ($arg === '--clear') {
        $clearCache = true;
    } elseif ($arg === '--watch') {
        $watchMode = true;
    } elseif ($arg[0] !== '-') {
        $specificFile = $arg;
    }
}

// Display help if requested
if (in_array('--help', $argv) || in_array('-h', $argv)) {
    echo "Simple Asset Compilation Script\n";
    echo "Usage: php assets-compile.php [options] [file]\n\n";
    echo "Options:\n";
    echo "  --clear         Clear asset cache before compilation\n";
    echo "  --watch         Watch for changes (requires inotify extension)\n";
    echo "  --help, -h      Display this help message\n";
    echo "\nExamples:\n";
    echo "  php assets-compile.php                   - Compile common assets\n";
    echo "  php assets-compile.php file.css          - Compile a specific file\n";
    exit(0);
}

// Create AssetManager instance
$assetManager = \baseKRIZAN\Assets\AssetManager::getInstance();

// Get path resolver for centralized paths
$pathResolver = \baseKRIZAN\Assets\AssetPathResolver::getInstance();

// Clear cache if requested
if ($clearCache) {
    echo "Clearing asset cache...\n";
    $assetManager->clearCache();
    echo "Asset cache cleared.\n\n";
}

// Display debug path information
echo "DEBUG: Path information:\n";
echo "- ROOT_DIR: " . \baseKRIZAN\Assets\AssetPathResolver::getRootPath() . "\n";
echo "- PUBLIC_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getPublicRootPath() . "\n";
echo "- CORE_SOURCE_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getCoreAssetsSourcePath() . "\n";
echo "- SRC_SOURCE_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getSrcAssetsSourcePath() . "\n";
echo "- MODULE_SOURCE_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getModuleSourceBasePath() . "\n";
echo "- CORE_CACHE_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getCoreAssetsCachePath() . "\n";
echo "- SRC_CACHE_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getSrcAssetsCachePath() . "\n";
echo "- MODULE_CACHE_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getModuleAssetsCachePath() . "\n";
echo "- CORE_PUBLIC_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getCoreAssetsPublicPath() . "\n";
echo "- SRC_PUBLIC_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getSrcAssetsPublicPath() . "\n";
echo "- MODULE_PUBLIC_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getModuleAssetsPublicPath() . "\n";
echo "- BASE_WEB_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getBaseWebPath() . "\n\n";

// Check if we're trying to watch for changes
if ($watchMode) {
    if (!extension_loaded('inotify')) {
        echo "Error: The inotify extension is required for watch mode.\n";
        echo "Install it with: pecl install inotify\n";
        exit(1);
    }
    
    echo "Starting watch mode. Press Ctrl+C to stop.\n";
    $watcher = new \baseKRIZAN\Assets\AssetWatcher(\baseKRIZAN\Assets\AssetPathResolver::getRootPath());
    $watcher->watch();
    exit(0);
}

// Compile a specific file if requested
if ($specificFile !== null) {
    echo "Compiling file: $specificFile\n";
    
    // Determine if this is a module file
    $module = null;
    if (preg_match('/^modules\/([^\/]+)\/(.+)$/', $specificFile, $matches)) {
        $module = $matches[1];
        $specificFile = $matches[2];
    } else if (preg_match('/^srcassets\/(.+)$/', $specificFile, $matches)) {
        // Handle srcassets explicitly
        $specificFile = 'srcassets/' . $matches[1];
        echo "Processing as srcassets file\n";
    } else if (preg_match('/^moduliassets\/([^\/]+)\/(.+)$/', $specificFile, $matches)) {
        // Handle moduliassets explicitly
        $module = $matches[1];
        $specificFile = $matches[2];
        echo "Processing as module file: module=$module, file=$specificFile\n";
    }
    
    // Process the file
    $result = $assetManager->processAssetPath($specificFile, $module);
    
    echo "Compiled to: $result\n";
    exit(0);
}

// Otherwise, compile common assets by pre-loading them
echo "Compiling common assets...\n";

// Load common resource modules to trigger compilation
// This part needs to use ResourceLoader
\baseKRIZAN\Assets\ResourceLoader::init();

// Common modules
$modules = [
    'default',
    'select2',
    'flatpickr',
    'datatable',
    'datetime'
];

// Load each module to trigger processing
foreach ($modules as $module) {
    echo "Loading module: $module\n";
    \baseKRIZAN\Assets\ResourceLoader::loadResource($module);
}

// Generate resources to trigger compilation
ob_start();
\baseKRIZAN\Assets\ResourceLoader::generateHeadResources();
\baseKRIZAN\Assets\ResourceLoader::generateBodyResources();
ob_end_clean();

echo "Common assets compiled successfully.\n";