<?php
/**
 * Asset Deployment Script
 * 
 * This script handles:
 * 1. Compiling all assets
 * 2. Generating manifest for production
 * 3. Configuring .htaccess for proper asset handling
 * 
 * Usage:
 *   php assets-deploy.php                    - Auto-detect environment
 *   php assets-deploy.php --production       - Force production mode
 *   php assets-deploy.php --dev              - Force development mode
 *   php assets-deploy.php --clear            - Clear asset cache
 */

// Define root directory constant (where assets-deploy.php is located)
define('ROOT_DIR', __DIR__);

// Define APP_ROOT for configuration paths
define('APP_ROOT', ROOT_DIR);

// Include the autoloader
require_once ROOT_DIR . '/vendor/autoload.php';

// Load Config classes for environment setup
require_once ROOT_DIR . '/app/baseKRIZAN/Bootstrap/initEnvironment.php';

// Load .env
$configPath = ROOT_DIR . '/app/config/';
$dotenv = \Dotenv\Dotenv::createImmutable($configPath);
$dotenv->load();

// Set CLI environment indicator
$_SERVER['CLI_SCRIPT'] = true;

// Load Config
\baseKRIZAN\Config\Config::load([ROOT_DIR . '/app/config/env_config.php']);

// Process command line arguments
$forceProduction = false;
$forceDevelopment = false;
$clearCache = false;

foreach ($argv as $arg) {
    if ($arg === '--production') {
        $forceProduction = true;
    }
    if ($arg === '--dev') {
        $forceDevelopment = true;
    }
    if ($arg === '--clear') {
        $clearCache = true;
    }
}

// Display help if requested
if (in_array('--help', $argv) || in_array('-h', $argv)) {
    echo "Asset Deployment Script\n";
    echo "Usage: php assets-deploy.php [options]\n\n";
    echo "Options:\n";
    echo "  --production    Force production mode compilation\n";
    echo "  --dev           Force development mode compilation\n";
    echo "  --clear         Clear asset cache before compilation\n";
    echo "  --help, -h      Display this help message\n";
    exit(0);
}

// Create main components
$assetManager = \baseKRIZAN\Assets\AssetManager::getInstance();
$pathResolver = \baseKRIZAN\Assets\AssetPathResolver::getInstance();

// Start deployment process
echo "=== ASSET DEPLOYMENT PROCESS ===\n";

// Get current configuration
$environment = \baseKRIZAN\Config\Config::get('environment');
$bundlingEnabled = \baseKRIZAN\Config\Config::get('asset.bundling_enabled');
$cdnEnabled = \baseKRIZAN\Config\Config::get('asset.cdn_enabled');

// Apply forced settings
if ($forceProduction) {
    $environment = 'production';
    $bundlingEnabled = true;
    echo "Forced production mode\n";
}

if ($forceDevelopment) {
    $environment = 'development';
    echo "Forced development mode\n";
}

echo "Current settings:\n";
echo "- Environment: $environment\n";
echo "- Asset bundling: " . ($bundlingEnabled ? "Enabled" : "Disabled") . "\n";
echo "- CDN: " . ($cdnEnabled ? "Enabled (" . \baseKRIZAN\Config\Config::get('asset.cdn_url') . ")" : "Disabled") . "\n\n";

// Check environment for production deployment
if ($environment !== 'production' && !$forceDevelopment && !$forceProduction) {
    echo "NOTICE: Environment is not set to 'production'.\n";
    echo "Running in development mode. Use --production to force production mode.\n\n";
}

// Clear cache if requested
if ($clearCache) {
    echo "Clearing asset cache...\n";
    $assetManager->clearCache();
    echo "Asset cache cleared.\n\n";
}

// Ensure asset directories exist
echo "Checking asset directories...\n";
$dirs = [
    \baseKRIZAN\Assets\AssetPathResolver::getCoreAssetsCachePath(),
    \baseKRIZAN\Assets\AssetPathResolver::getSrcAssetsCachePath(),
    \baseKRIZAN\Assets\AssetPathResolver::getModuleAssetsCachePath()
];

foreach ($dirs as $dir) {
    if (!is_dir($dir)) {
        echo "Creating directory: $dir\n";
        $pathResolver->ensureDirectoryExists($dir);
    } else {
        echo "Directory exists: $dir\n";
    }
}
echo "\n";

// Set up .htaccess for assets
echo "Setting up .htaccess for assets...\n";
setupHtaccess($pathResolver);
echo "\n";

// Create assets loader script
echo "Setting up assets loader script...\n";
setupAssetLoader($pathResolver);
echo "\n";

// Debug info - show paths
echo "DEBUG: Path information:\n";
echo "- ROOT_DIR: " . \baseKRIZAN\Assets\AssetPathResolver::getRootPath() . "\n";
echo "- PUBLIC_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getPublicRootPath() . "\n";
echo "- CORE_SOURCE_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getCoreAssetsSourcePath() . "\n";
echo "- SRC_SOURCE_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getSrcAssetsSourcePath() . "\n";
echo "- MODULE_SOURCE_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getModuleSourceBasePath() . "\n";
echo "- CORE_CACHE_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getCoreAssetsCachePath() . "\n";
echo "- SRC_CACHE_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getSrcAssetsCachePath() . "\n";
echo "- MODULE_CACHE_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getModuleAssetsCachePath() . "\n";
echo "- CORE_PUBLIC_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getCoreAssetsPublicPath() . "\n";
echo "- SRC_PUBLIC_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getSrcAssetsPublicPath() . "\n";
echo "- MODULE_PUBLIC_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getModuleAssetsPublicPath() . "\n";
echo "- BASE_WEB_PATH: " . \baseKRIZAN\Assets\AssetPathResolver::getBaseWebPath() . "\n\n";

// Compile assets
echo "Compiling assets...\n";
$startTime = microtime(true);
$result = $assetManager->compileAll($forceProduction);
$duration = round(microtime(true) - $startTime, 2);
echo "Asset compilation completed in {$duration}s\n\n";

// Final checks
echo "Verifying setup...\n";
verifySetup($pathResolver, $environment, $forceProduction);

/**
 * Setup .htaccess for assets directory
 */
function setupHtaccess(\baseKRIZAN\Assets\AssetPathResolver $pathResolver): void
{
    $htaccessContent = <<<'HTACCESS'
# .htaccess file for assets directory

# Disable directory browsing
Options -Indexes

# Enable rewrite engine
<IfModule mod_rewrite.c>
    RewriteEngine On
    
    # Don't rewrite files that exist
    RewriteCond %{REQUEST_FILENAME} -f
    RewriteRule ^ - [L]
    
    # For development mode, redirect missing assets to loader
    RewriteRule ^appassets/(.*)$ ../assets-loader.php?path=appassets/$1 [L,QSA]
    RewriteRule ^srcassets/(.*)$ ../assets-loader.php?path=srcassets/$1 [L,QSA]
    RewriteRule ^moduliassets/(.*)$ ../assets-loader.php?path=moduliassets/$1 [L,QSA]
    
    # Don't rewrite these paths
    RewriteRule ^(css|js|images|fonts)/(.*)$ - [L]
    
    # Rewrite other asset requests to loader
    RewriteRule ^(.*)$ ../assets-loader.php?path=$1 [L,QSA]
</IfModule>

# Cache control
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType text/css "access plus 1 year"
    ExpiresByType application/javascript "access plus 1 year"
    ExpiresByType image/svg+xml "access plus 1 year"
    ExpiresByType image/jpeg "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType image/gif "access plus 1 year"
    ExpiresByType image/x-icon "access plus 1 year"
    ExpiresByType font/ttf "access plus 1 year"
    ExpiresByType font/woff "access plus 1 year"
    ExpiresByType font/woff2 "access plus 1 year"
    ExpiresByType application/vnd.ms-fontobject "access plus 1 year"
    ExpiresByType font/otf "access plus 1 year"
</IfModule>

# Compression
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/css
    AddOutputFilterByType DEFLATE application/javascript
    AddOutputFilterByType DEFLATE image/svg+xml
    AddOutputFilterByType DEFLATE font/ttf
    AddOutputFilterByType DEFLATE font/woff
    AddOutputFilterByType DEFLATE application/vnd.ms-fontobject
    AddOutputFilterByType DEFLATE font/otf
</IfModule>

# Security headers
<IfModule mod_headers.c>
    Header set X-Content-Type-Options "nosniff"
    Header set Cache-Control "public, max-age=31536000"
</IfModule>
HTACCESS;

    // Write .htaccess file to public assets directory
    $publicAssetsPath = \baseKRIZAN\Assets\AssetPathResolver::getCoreAssetsCachePath();
    $htaccessPath = $publicAssetsPath . '/.htaccess';
    
    if (!is_dir(dirname($htaccessPath))) {
        mkdir(dirname($htaccessPath), 0755, true);
    }
    
    file_put_contents($htaccessPath, $htaccessContent);
    echo "Created .htaccess at: $htaccessPath\n";
}

/**
 * Create the asset loader script
 */
function setupAssetLoader(\baseKRIZAN\Assets\AssetPathResolver $pathResolver): void
{
    $loaderContent = <<<LOADER
<?php
/**
 * Asset Loader Script
 * 
 * This script handles on-demand asset compilation in development mode
 * and serves pre-compiled assets in production mode.
 */

// Define root directory constant
define('ROOT_DIR', __DIR__);

// Load dependencies
require_once ROOT_DIR . '/vendor/autoload.php';

// Load environment functions
require_once ROOT_DIR . '/app/baseKRIZAN/Bootstrap/initEnvironment.php';

// Load .env
\$configPath = ROOT_DIR . '/app/config/';
\$dotenv = \\Dotenv\\Dotenv::createImmutable(\$configPath);
\$dotenv->load();

// Load Config
\\baseKRIZAN\\Services\\Config::load([ROOT_DIR . '/app/config/env_config.php']);

// Get path resolver for paths
\$pathResolver = \\baseKRIZAN\\Assets\\AssetPathResolver::getInstance();

// Get asset manager
\$assetManager = \\baseKRIZAN\\Assets\\AssetManager::getInstance();

// Get requested path
\$requestPath = \$_GET['path'] ?? '';
if (empty(\$requestPath)) {
    \$requestPath = \$_SERVER['REQUEST_URI'] ?? '';
    \$basePath = \\baseKRIZAN\\Assets\\AssetPathResolver::getBaseWebPath();
    
    // Strip base path and assets/ prefix
    if (!empty(\$basePath) && strpos(\$requestPath, \$basePath) === 0) {
        \$requestPath = substr(\$requestPath, strlen(\$basePath));
    }
    
    \$requestPath = ltrim(\$requestPath, '/');
}

// Detect asset type based on path prefix
\$assetType = 'core';
\$assetPath = \$requestPath;
\$module = null;

if (strpos(\$requestPath, 'appassets/') === 0) {
    \$assetType = 'core';
    \$assetPath = substr(\$requestPath, 10); // Length of 'appassets/'
} elseif (strpos(\$requestPath, 'srcassets/') === 0) {
    \$assetType = 'src';
    \$assetPath = substr(\$requestPath, 10); // Length of 'srcassets/'
} elseif (strpos(\$requestPath, 'moduliassets/') === 0) {
    \$assetType = 'module';
    \$assetPath = substr(\$requestPath, 13); // Length of 'moduliassets/'
    
    // Extract module name
    \$parts = explode('/', \$assetPath);
    if (count(\$parts) >= 1) {
        \$module = \$parts[0];
        \$assetPath = implode('/', array_slice(\$parts, 1));
    }
}

// Get file extension
\$extension = pathinfo(\$requestPath, PATHINFO_EXTENSION);

// Set content type
\$contentTypes = [
    'css' => 'text/css',
    'js' => 'application/javascript',
    'svg' => 'image/svg+xml',
    'png' => 'image/png',
    'jpg' => 'image/jpeg',
    'jpeg' => 'image/jpeg',
    'gif' => 'image/gif',
    'ico' => 'image/x-icon',
    'ttf' => 'font/ttf',
    'woff' => 'font/woff',
    'woff2' => 'font/woff2',
    'eot' => 'application/vnd.ms-fontobject',
    'otf' => 'font/otf'
];

\$contentType = \$contentTypes[\$extension] ?? 'application/octet-stream';

// Development mode - determine paths
\$developmentMode = \\baseKRIZAN\\Services\\Config::get('environment') === 'development';

// Determine the cache path based on asset type
\$cachePath = '';

// Check for hash in the filename to determine if this is already a processed asset
\$isHashedFile = preg_match('/(.+)\\.([a-f0-9]{8})\\.([a-z]+)\$/', \$assetPath);

if (\$assetType === 'module') {
    if (\$isHashedFile) {
        // This is a hashed filename
        \$cachePath = \\baseKRIZAN\\Assets\\AssetPathResolver::getModuleAssetsCachePath() . "/\$module/" . \$assetPath;
    } else {
        // Process the file on-demand
        \$assetManager->processAssetPath(\$assetPath, \$module);
        \$paths = \$pathResolver->determineAssetPaths(\$assetPath, \$module);
        \$cachePath = \$paths['cache'];
    }
} elseif (\$assetType === 'src') {
    if (\$isHashedFile) {
        // This is a hashed filename
        \$cachePath = \\baseKRIZAN\\Assets\\AssetPathResolver::getSrcAssetsCachePath() . '/' . \$assetPath;
    } else {
        // Process the file on-demand using the srcassets prefix
        \$srcPath = 'srcassets/' . \$assetPath;
        \$assetManager->processAssetPath(\$srcPath);
        \$paths = \$pathResolver->determineAssetPaths(\$srcPath);
        \$cachePath = \$paths['cache'];
    }
} else {
    // Core asset (appassets)
    if (\$isHashedFile) {
        // This is a hashed filename
        \$cachePath = \\baseKRIZAN\\Assets\\AssetPathResolver::getCoreAssetsCachePath() . '/' . \$assetPath;
    } else {
        // Process the file on-demand
        \$assetManager->processAssetPath(\$assetPath);
        \$paths = \$pathResolver->determineAssetPaths(\$assetPath);
        \$cachePath = \$paths['cache'];
    }
}

// Serve the file if it exists
if (!empty(\$cachePath) && file_exists(\$cachePath)) {
    header("Content-Type: \$contentType");
    header("Cache-Control: public, max-age=31536000");
    header("Expires: " . gmdate("D, d M Y H:i:s", time() + 31536000) . " GMT");
    header("Last-Modified: " . gmdate("D, d M Y H:i:s", filemtime(\$cachePath)) . " GMT");
    
    readfile(\$cachePath);
    exit;
}

// If not found and we're in development mode, try to compile on-demand
if (\$developmentMode) {
    // Try to determine the original asset path
    \$originalPath = preg_replace('/\\.[a-f0-9]{8}\\.(css|js|svg|jpg|jpeg|png|gif|ico|ttf|woff|woff2|eot|otf)\$/', '.\$1', \$assetPath);
    
    if (\$assetType === 'module' && \$module) {
        // Process the module asset
        \$newPath = \$assetManager->processAssetPath(\$originalPath, \$module);
        
        if (\$newPath) {
            header("Location: \$newPath");
            exit;
        }
    } elseif (\$assetType === 'src') {
        // Process the src asset with 'srcassets/' prefix
        \$srcPath = 'srcassets/' . \$originalPath;
        \$newPath = \$assetManager->processAssetPath(\$srcPath);
        
        if (\$newPath) {
            header("Location: \$newPath");
            exit;
        }
    } else {
        // Process the core asset
        \$newPath = \$assetManager->processAssetPath(\$originalPath);
        
        if (\$newPath) {
            header("Location: \$newPath");
            exit;
        }
    }
}

// File not found
header("HTTP/1.0 404 Not Found");
echo "404 - Asset not found: " . htmlspecialchars(\$requestPath);
exit;
LOADER;

    // Write asset loader script
    $loaderPath = \baseKRIZAN\Assets\AssetPathResolver::getPublicRootPath() . '/assets-loader.php';
    file_put_contents($loaderPath, $loaderContent);
    echo "Created asset loader at: $loaderPath\n";
}

/**
 * Verify production setup
 */
function verifySetup(\baseKRIZAN\Assets\AssetPathResolver $pathResolver, string $environment, bool $forceProduction = false): void
{
    // Check if manifest exists for production
    if ($environment === 'production' || $forceProduction) {
        $manifestPath = \baseKRIZAN\Assets\AssetPathResolver::getCoreAssetsCachePath() . '/manifest.json';
        if (file_exists($manifestPath)) {
            $manifest = json_decode(file_get_contents($manifestPath), true);
            $entryCount = 0;
            
            foreach ($manifest as $section => $types) {
                if ($section === 'bundles') {
                    $entryCount += count($types['css'] ?? []) + count($types['js'] ?? []);
                } elseif ($section !== 'meta') {
                    foreach ($types as $type => $entries) {
                        $entryCount += count($entries);
                    }
                }
            }
            
            echo "Manifest file created with $entryCount entries\n";
        } else {
            echo "WARNING: Manifest file not created\n";
        }
    }
    
    echo "\n=== ASSET DEPLOYMENT COMPLETE ===\n";
    echo "Assets are ready for " . ($environment === 'production' || $forceProduction ? "production" : "development") . " mode\n";
}