<?php

namespace financije\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Controllers\BaseController;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Database\DatabaseConnection;
use financije\Models\DatabaseTableNaplata;
use financije\Models\DatabaseTableRashodi;

class Financije extends BaseController 
{
    protected DatabaseTableNaplata $naplataTable;
    protected ?\financije\Models\DatabaseTableRashodi $rashodiTable;
    protected DatabaseConnection $dbConnection;

    public function __construct(
        Authentication $authentication,
        DatabaseTableNaplata $naplataTable,
        DatabaseConnection $dbConnection,
        Logger $logger,
        ?SessionManager $sessionManager = null,
        ?\financije\Models\DatabaseTableRashodi $rashodiTable = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->naplataTable = $naplataTable;
        $this->rashodiTable = $rashodiTable;
        $this->dbConnection = $dbConnection;
    }
    
    public function admin(): Response 
    {
        try {
            $naplataStats = [
                'totalAmount' => $this->naplataTable->getTotalAmount(),
                'paidAmount' => $this->naplataTable->getPaidAmount(),
                'unpaidAmount' => $this->naplataTable->getUnpaidAmount(),
                'byCompany' => $this->naplataTable->getAmountByCompany(),
                'byStatus' => $this->naplataTable->getAmountByStatus(),
                'byMonth' => $this->naplataTable->getAmountByMonth(),
                'recentItems' => $this->naplataTable->getRecentItems()
            ];

            $rashodiStats = [];
            if ($this->rashodiTable) {
                $rashodiStats = [
                    'totalAmount' => $this->rashodiTable->getTotalAmount(),
                    'byType' => $this->rashodiTable->getAmountByType(),
                    'byMonth' => $this->rashodiTable->getAmountByMonth(),
                    'recentItems' => $this->rashodiTable->getRecentItems()
                ];
            }

            $combinedMonthlyData = $this->combineMonthlyData(
                $naplataStats['byMonth'], 
                $rashodiStats['byMonth'] ?? []
            );

            $profitabilityData = $this->calculateProfitability($combinedMonthlyData);
            
            return $this->response()->render(
                'financije/resources/views/financije/admin.html.php',
                [
                    'title' => 'Naplata & Rashodi Admin',
                    'naplataStats' => $naplataStats,
                    'rashodiStats' => $rashodiStats,
                    'combinedData' => $combinedMonthlyData,
                    'profitabilityData' => $profitabilityData
                ],
                'Naplata & Rashodi Admin'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading combined admin', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load admin panel');
        }
    }
    
    public function getAdminStats(): Response
    {
        try {
            $naplataStats = [
                'totalAmount' => $this->naplataTable->getTotalAmount(),
                'paidAmount' => $this->naplataTable->getPaidAmount(), 
                'unpaidAmount' => $this->naplataTable->getUnpaidAmount(),
                'byCompany' => $this->naplataTable->getAmountByCompany(),
                'byStatus' => $this->naplataTable->getAmountByStatus(),
                'byMonth' => $this->naplataTable->getAmountByMonth()
            ];

            $rashodiStats = [];
            if ($this->rashodiTable) {
                $rashodiStats = [
                    'totalAmount' => $this->rashodiTable->getTotalAmount(),
                    'byType' => $this->rashodiTable->getAmountByType(),
                    'byMonth' => $this->rashodiTable->getAmountByMonth()
                ];
            }

            $combinedMonthlyData = $this->combineMonthlyData(
                $naplataStats['byMonth'], 
                $rashodiStats['byMonth'] ?? []
            );

            $profitabilityData = $this->calculateProfitability($combinedMonthlyData);
            
            $statistics = [
                'naplata' => $naplataStats,
                'rashodi' => $rashodiStats,
                'combined' => $combinedMonthlyData,
                'profitability' => $profitabilityData
            ];
            
            return Response::json($statistics);
        } catch (\Throwable $e) {
            $this->logger->error('Error fetching admin stats', [
                'message' => $e->getMessage()
            ]);
            return Response::json(['error' => 'Failed to fetch admin stats'], 500);
        }
    }

    // HELPER METHODS
    private function combineMonthlyData(array $naplataMonthly, array $rashodiMonthly): array
    {
        $combined = [];
        
        foreach ($naplataMonthly as $item) {
            $key = $item['year'] . '-' . $item['month'];
            $combined[$key] = [
                'month' => (int)$item['month'],
                'year' => (int)$item['year'],
                'naplata' => (float)$item['amount'],
                'rashodi' => 0.0
            ];
        }
        
        foreach ($rashodiMonthly as $item) {
            $key = $item['year'] . '-' . $item['month'];
            if (isset($combined[$key])) {
                $combined[$key]['rashodi'] = (float)$item['amount'];
            } else {
                $combined[$key] = [
                    'month' => (int)$item['month'],
                    'year' => (int)$item['year'],
                    'naplata' => 0.0,
                    'rashodi' => (float)$item['amount']
                ];
            }
        }
        
        ksort($combined);
        return array_values($combined);
    }

    private function calculateProfitability(array $combinedData): array
    {
        $profitability = [];
        
        foreach ($combinedData as $item) {
            $profit = $item['naplata'] - $item['rashodi'];
            $margin = $item['naplata'] > 0 ? ($profit / $item['naplata']) * 100 : 0;
            
            $profitability[] = [
                'month' => $item['month'],
                'year' => $item['year'],
                'naplata' => $item['naplata'],
                'rashodi' => $item['rashodi'],
                'profit' => $profit,
                'margin' => $margin
            ];
        }
        
        return $profitability;
    }
}