<?php

namespace financije\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Controllers\BaseController;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Validation\Validator;
use baseKRIZAN\Database\DatabaseConnection;
use financije\Models\DatabaseTableNaplata;
use Models\DatatablesSSP;
use Models\Entity\Korisnik;

class Naplata extends BaseController 
{
    private const MAX_EXPORT_ROWS = 1000;
    private const YES = 'yes';
    private const NO = 'no';

    protected DatabaseTable $korisniciTable;
    protected DatabaseTableNaplata $naplataTable;
    protected Validator $validator;
    protected DatabaseConnection $dbConnection;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableNaplata $naplataTable,
        Validator $validator,
        DatabaseConnection $dbConnection,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->naplataTable = $naplataTable;
        $this->validator = $validator;
        $this->dbConnection = $dbConnection;
    }

    /**
     * DataTables mapping konfiguracija za naplate
     */
    private function getDataTablesMapping(): array
    {
        $tableConfig = [
            'table_name' => 'financije_naplata',
            'primary_key' => 'ID',
            'fixed_columns' => 1,
            'route_name' => 'naplatalist'
        ];

        $columns = [
            // ID i akcije - uvijek fiksno
            [
                'type' => 'number',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'ID',
                'sqlcolumnname' => 'ID',
                'width' => 75,
                'render' => 'actions'
            ],
            // Tvrtka
            [
                'type' => 'text',
                'searchbox' => self::NO,
                'dropdown' => self::YES,
                'calculateinfoot' => self::NO,
                'headername' => 'Tvrtka',
                'sqlcolumnname' => 'Tvrtka',
                'width' => 150
            ],
            // Period rada - prikaz iz Period_rada kolone
            [
                'type' => 'text',
                'searchbox' => self::NO,
                'dropdown' => self::YES,
                'calculateinfoot' => self::NO,
                'headername' => 'Period rada',
                'sqlcolumnname' => 'Period_rada',
                'width' => 100,
                'render' => 'period'
            ],
            // Cijena bez PDV-a
            [
                'type' => 'currency',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::YES,
                'headername' => 'Cijena bez PDV (€)',
                'sqlcolumnname' => 'Cijena_bez_pdv',
                'width' => 120,
                'render' => 'currency'
            ],
            // PDV
            [
                'type' => 'currency',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::YES,
                'headername' => 'PDV (€)',
                'sqlcolumnname' => 'PDV',
                'width' => 100,
                'render' => 'currency'
            ],
            // Ukupan iznos
            [
                'type' => 'currency',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::YES,
                'headername' => 'Ukupan iznos (€)',
                'sqlcolumnname' => 'Ukupan_iznos',
                'width' => 130,
                'render' => 'currency'
            ],
            // Valuta plaćanja
            [
                'type' => 'date',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Valuta plaćanja',
                'sqlcolumnname' => 'Valuta_placanja',
                'width' => 120
            ],
            // Plaćeno
            [
                'type' => 'currency',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::YES,
                'headername' => 'Plaćeno (€)',
                'sqlcolumnname' => 'Placeno',
                'width' => 110,
                'render' => 'currency'
            ],
            // Nenaplaćeno - NOVO POLJE
            [
                'type' => 'currency',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::YES,
                'headername' => 'Nenaplaćeno (€)',
                'sqlcolumnname' => 'Nenaplaceno',
                'width' => 120,
                'render' => 'currency'
            ],
            // Status
            [
                'type' => 'text',
                'searchbox' => self::NO,
                'dropdown' => self::YES,
                'calculateinfoot' => self::NO,
                'headername' => 'Status',
                'sqlcolumnname' => 'Status',
                'width' => 150
            ]
        ];

        return array_merge($tableConfig, ['columns' => $columns]);
    }

    public function edit(Request $request): Response 
    {
        try {
            $korisnik = $this->authentication->getUser();
            
            $financija = null;
            $editToken = null;
            
            $id = $request->getParam('id');
            if ($id) {
                $financija = $this->naplataTable->findById($id);
                if ($financija) {
                    $editToken = $this->createEditToken($financija->ID, 'naplata');
                    
                    // Format period for display
                    if (!empty($financija->Period_rada)) {
                        $financija->formatted_period = $financija->Period_rada;
                    } elseif (isset($financija->mjesec) && isset($financija->godina)) {
                        $financija->formatted_period = DatabaseTableNaplata::formatPeriod(
                            $financija->mjesec, 
                            $financija->godina
                        );
                    }
                }
            }

            return $this->response()->render(
                'financije/resources/views/naplata/edit.html.php',
                [
                    'financija' => $financija,
                    'edit_token' => $editToken,
                    'user' => $korisnik
                ],
                'Uredi financijsku stavku'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading naplata edit form', [
                'message' => $e->getMessage(),
                'id' => $id ?? null
            ]);
            return $this->response()->renderError('Failed to load naplata edit form');
        }
    }

    public function saveEdit(Request $request): Response 
    {
        try {
            // 1. DOHVAĆANJE OSNOVNIH PODATAKA
            $editToken = $request->getPost('edit_token');
            $financija = $request->getPost('financija');
            $financijaId = null;
            $validToken = null;
            
            // 2. VALIDACIJA TOKENA (AKO POSTOJI)
            if ($editToken) {
                $tokenResult = $this->validateEditTokenForValidation($request, 'naplata');
                if (isset($tokenResult['error'])) {
                    return $this->renderNaplataForm(
                        $financija, 
                        $editToken, 
                        $tokenResult
                    );
                }
                $financijaId = $tokenResult['id'];
                $validToken = $tokenResult['token'];
                $financija['ID'] = $financijaId;
            }
            
            // 3. VALIDACIJA PODATAKA FORME
            $rules = [
                'financija[Tvrtka]' => ['required', 'plain_text'],
                'financija[period_input]' => ['required', 'regex:/^\d{1,2}-\d{4}$/'], // Format MM-YYYY
                'financija[Cijena_bez_pdv]' => ['required', 'currency'],
                'financija[PDV]' => ['optional', 'currency'],
                'financija[Ukupan_iznos]' => ['required', 'currency'],
                'financija[Valuta_placanja]' => ['required', 'date'],
                'financija[Placeno]' => ['required', 'currency'],
                'financija[Status]' => ['required', 'plain_text'],
            ];

            $validation = $request->validate($rules);
            if ($validation->fails()) {
                $newEditToken = null;
                if ($financijaId && $validToken) {
                    $newEditToken = $this->refreshEditToken($validToken, 'naplata');
                }
                
                return $this->renderNaplataForm(
                    $financija, 
                    $newEditToken, 
                    $validation->errors()
                );
            }

            // 4. USPJEŠNA VALIDACIJA - troši token
            if ($validToken) {
                $this->consumeEditToken($validToken);
            }

            // 5. PROCESIRANJE PODATAKA
            $this->processDateFields($financija);
            $this->processCurrencyFields($financija);
            $this->processPeriodField($financija);
            
            // 6. SPREMANJE PODATAKA
            $this->naplataTable->save($financija);

            // 7. PREUSMJERAVANJE
            return Response::redirect('naplata/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error saving financija', [
                'message' => $e->getMessage(),
                'financija' => $financija ?? null
            ]);
            return $this->response()->renderError('Failed to save financija');
        }
    }

    public function list(): Response 
    {
        try {
            return $this->response()->render(
                'financije/resources/views/naplata/list.html.php',
                [
                    'title' => 'Sve naplate'
                ],
                'Sve naplate'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading naplata list', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load naplata list');
        }
    }

    public function fetchnaplata(): Response 
    {
        try {
            $requestData = $_POST;
            
            // 1. KORISTI INTERNI MAPPING
            $naplataMapping = $this->getDataTablesMapping();
            
            // 2. PRIPREMI KOLUMNE ZA DatatablesSSP
            $columns = [];
            foreach ($naplataMapping['columns'] as $index => $column) {
                $columns[] = [
                    'db' => $column['sqlcolumnname'],
                    'dt' => $index
                ];
            }

            // 3. POZOVI DatatablesSSP DIREKTNO (naplata ne trebaju WHERE uvjete za uloge)
            $data = DatatablesSSP::complexWithUniques(
                $requestData,
                $this->dbConnection,
                $naplataMapping['table_name'],
                $naplataMapping['primary_key'],
                $columns,
                $naplataMapping['columns'],
                null // Nema WHERE uvjeta za naplata
            );

            // 4. DODAJ KONFIGURACIJU NA PRVI ZAHTJEV
            if (isset($requestData['init']) && $requestData['init'] === 'true') {
                $data['config'] = [
                    'columns' => $naplataMapping['columns'],
                    'fixedColumns' => $naplataMapping['fixed_columns'],
                    'routeName' => $naplataMapping['route_name']
                ];
                
                if (isset($naplataMapping['rowStyling'])) {
                    $data['config']['rowStyling'] = $naplataMapping['rowStyling'];
                }
            }
            
            echo json_encode($data);
            exit;
        } catch (\Throwable $e) {
            $this->logger->error('Error fetching naplata data', [
                'message' => $e->getMessage()
            ]);
            return Response::json(['error' => 'Failed to fetch naplata data'], 500);
        }
    }

    public function export(Request $request): Response 
    {
        try {
            $naplata = $this->naplataTable->findAll();
            
            $columns = [
                'ID', 'Tvrtka', 'mjesec', 'godina', 'Cijena_bez_pdv', 
                'PDV', 'Ukupan_iznos', 'Valuta_placanja', 
                'Placeno', 'Nenaplaceno', 'Status'
            ];
            
            $exportData = $this->exportNaplata($naplata, $columns, "sve financijske stavke");
            
            return Response::download(
                $exportData['content'],
                $exportData['filename'],
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Export failed', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id', []),
                'ip' => $request->getIp()
            ]);
            return $this->response()->renderError('Failed to export naplata');
        }
    }

    public function delete(Request $request): Response 
    {
        try {
            $korisnik = $this->authentication->getUser();
            $financijaId = $request->getPost('id');
            $financija = $this->naplataTable->findById($financijaId);
            
            if (!$financija) {
                $this->logger->security('Attempt to delete non-existent naplata record', [
                    'id' => $financijaId,
                    'user_id' => $korisnik->id,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Record not found');
            }

            if (!$korisnik->hasPermission(\Models\Entity\Korisnik::KORISNIK_MASTER)) {
                $this->logger->security('Unauthorized naplata deletion attempt', [
                    'id' => $financijaId,
                    'user_id' => $korisnik->id,
                    'ip' => $request->getIp()
                ]);
                return $this->renderNaplataForm(
                    [], 
                    null, 
                    ['error' => ['Nemate potrebna prava za ovu akciju']]
                );
            }

            $this->naplataTable->delete($financija->ID);
            
            $this->logger->security('Naplata record deleted', [
                'id' => $financijaId,
                'deleted_by' => $korisnik->id,
                'ip' => $request->getIp()
            ]);

            return Response::redirect('naplata/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error deleting naplata record', [
                'message' => $e->getMessage(),
                'id' => $financijaId ?? null
            ]);
            return $this->response()->renderError('Failed to delete naplata record');
        }
    }

    // HELPER METHODS
    private function processDateFields(array &$financija): void 
    {
        require_once APP_ROOT . '/moduli/financije/Helpers/convertDateToEnglishFormat.php';
        
        if (isset($financija['Valuta_placanja'])) {
            $financija['Valuta_placanja'] = convertDateToEnglishFormat($financija['Valuta_placanja']);
        }
    }

    private function processCurrencyFields(array &$financija): void 
    {
        $currencyFields = ['Cijena_bez_pdv', 'PDV', 'Ukupan_iznos', 'Placeno', 'Nenaplaceno'];
        foreach ($currencyFields as $field) {
            if (isset($financija[$field]) && is_string($financija[$field])) {
                $financija[$field] = str_replace(',', '.', $financija[$field]);
            }
        }
        
        // Automatski izračunaj nenaplaćeni iznos (može biti negativan)
        if (isset($financija['Ukupan_iznos']) && isset($financija['Placeno'])) {
            $ukupanIznos = (float)$financija['Ukupan_iznos'];
            $placeno = (float)$financija['Placeno'];
            $financija['Nenaplaceno'] = $ukupanIznos - $placeno;
        }
    }

    private function processPeriodField(array &$financija): void 
    {
        if (isset($financija['period_input']) && !empty($financija['period_input'])) {
            $periodData = DatabaseTableNaplata::parsePeriod($financija['period_input']);
            $financija['mjesec'] = $periodData['mjesec'];
            $financija['godina'] = $periodData['godina'];
            $financija['Period_rada'] = $financija['period_input'];
            unset($financija['period_input']);
        }
    }

    private function renderNaplataForm(array $financija, ?string $editToken, array $errors): Response 
    {
        if (is_array($financija)) {
            if (!empty($financija['Period_rada'])) {
                $financija['formatted_period'] = $financija['Period_rada'];
            } elseif (isset($financija['mjesec']) && isset($financija['godina'])) {
                $financija['formatted_period'] = DatabaseTableNaplata::formatPeriod(
                    $financija['mjesec'], 
                    $financija['godina']
                );
            }
        }

        return $this->response()->render(
            'financije/resources/views/naplata/edit.html.php',
            [
                'errors' => $errors,
                'financija' => is_array($financija) ? (object)$financija : $financija,
                'edit_token' => $editToken,
                'user' => $this->authentication->getUser()
            ],
            'Uredi financijsku stavku'
        );
    }

    private function exportNaplata(array $naplata, array $columns, string $filename): array 
    {
        try {
            $currentUser = $this->authentication->getUser();
            if (!$currentUser || $currentUser->user_permissions < Korisnik::KORISNIK_MASTER) {
                throw new \RuntimeException('Unauthorized to export naplata');
            }
    
            if (count($naplata) > self::MAX_EXPORT_ROWS) {
                throw new \RuntimeException(
                    sprintf('Export limit exceeded: %d rows', self::MAX_EXPORT_ROWS)
                );
            }
    
            require_once APP_ROOT . '/app/Helpers/excelexport.php';
            $savefileas = sprintf(
                "%s - %s tjedan %s",
                $filename,
                date('d-m-Y'),
                date("W", strtotime(date('d-m-Y')))
            );
    
            return excelexport('naplata', $columns, $savefileas, $naplata);
        } catch (\Throwable $e) {
            $this->logger->error('Export failed', [
                'message' => $e->getMessage(),
                'rows_count' => count($naplata)
            ]);
            throw $e;
        }
    }
}