<?php

namespace financije\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Controllers\BaseController;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Validation\Validator;
use baseKRIZAN\Database\DatabaseConnection;
use financije\Models\DatabaseTableRashodi;
use Models\DatatablesSSP;
use Models\Entity\Korisnik;

class Rashodi extends BaseController 
{
    private const MAX_EXPORT_ROWS = 1000;
    private const YES = 'yes';
    private const NO = 'no';

    protected DatabaseTable $korisniciTable;
    protected DatabaseTableRashodi $rashodiTable;
    protected Validator $validator;
    protected DatabaseConnection $dbConnection;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableRashodi $rashodiTable,
        Validator $validator,
        DatabaseConnection $dbConnection,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->rashodiTable = $rashodiTable;
        $this->validator = $validator;
        $this->dbConnection = $dbConnection;
    }

    /**
     * DataTables mapping konfiguracija za rashode
     */
    private function getDataTablesMapping(): array
    {
        $tableConfig = [
            'table_name' => 'financije_rashodi',
            'primary_key' => 'ID',
            'fixed_columns' => 1,
            'route_name' => 'rashodilist'
        ];

        $columns = [
            // ID i akcije
            [
                'type' => 'number',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'ID',
                'sqlcolumnname' => 'ID',
                'width' => 75,
                'render' => 'actions'
            ],
            // Period rada
            [
                'type' => 'text',
                'searchbox' => self::NO,
                'dropdown' => self::YES,
                'calculateinfoot' => self::NO,
                'headername' => 'Period',
                'sqlcolumnname' => 'Period_rada',
                'width' => 100
            ],
            // Vrsta rashoda
            [
                'type' => 'text',
                'searchbox' => self::NO,
                'dropdown' => self::YES,
                'calculateinfoot' => self::NO,
                'headername' => 'Vrsta rashoda',
                'sqlcolumnname' => 'vrsta_rashoda',
                'width' => 200
            ],
            // Cijena
            [
                'type' => 'currency',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::YES,
                'headername' => 'Cijena (€)',
                'sqlcolumnname' => 'cijena',
                'width' => 120,
                'render' => 'currency'
            ]
        ];

        return array_merge($tableConfig, ['columns' => $columns]);
    }

    public function edit(Request $request): Response 
    {
        try {
            $korisnik = $this->authentication->getUser();
            
            $rashod = null;
            $editToken = null;
            
            $id = $request->getParam('id');
            if ($id) {
                $rashod = $this->rashodiTable->findById($id);
                if ($rashod) {
                    $editToken = $this->createEditToken($rashod->ID, 'rashodi');
                    
                    // Format period for display
                    if (!empty($rashod->Period_rada)) {
                        $rashod->formatted_period = $rashod->Period_rada;
                    } elseif (isset($rashod->mjesec) && isset($rashod->godina)) {
                        $rashod->formatted_period = DatabaseTableRashodi::formatPeriod(
                            $rashod->mjesec, 
                            $rashod->godina
                        );
                    }
                }
            }

            $expenseTypes = $this->rashodiTable->getExpenseTypes();

            return $this->response()->render(
                'financije/resources/views/rashodi/edit.html.php',
                [
                    'rashod' => $rashod,
                    'edit_token' => $editToken,
                    'user' => $korisnik,
                    'expense_types' => $expenseTypes
                ],
                'Uredi rashod'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading rashodi edit form', [
                'message' => $e->getMessage(),
                'id' => $id ?? null
            ]);
            return $this->response()->renderError('Failed to load rashodi edit form');
        }
    }

    public function saveEdit(Request $request): Response 
    {
        try {
            // 1. DOHVAĆANJE OSNOVNIH PODATAKA
            $editToken = $request->getPost('edit_token');
            $rashod = $request->getPost('rashod');
            $rashodId = null;
            $validToken = null;
            
            // 2. VALIDACIJA TOKENA (AKO POSTOJI)
            if ($editToken) {
                $tokenResult = $this->validateEditTokenForValidation($request, 'rashodi');
                if (isset($tokenResult['error'])) {
                    return $this->renderRashodForm(
                        $rashod, 
                        $editToken, 
                        $tokenResult
                    );
                }
                $rashodId = $tokenResult['id'];
                $validToken = $tokenResult['token'];
                $rashod['ID'] = $rashodId;
            }
            
            // 3. VALIDACIJA PODATAKA FORME
            $rules = [
                'rashod[period_input]' => ['required', 'regex:/^\d{1,2}-\d{4}$/'], // Format MM-YYYY
                'rashod[vrsta_rashoda]' => ['required', 'plain_text'],
                'rashod[cijena]' => ['required', 'currency'],
            ];

            $validation = $request->validate($rules);
            if ($validation->fails()) {
                $newEditToken = null;
                if ($rashodId && $validToken) {
                    $newEditToken = $this->refreshEditToken($validToken, 'rashodi');
                }
                
                return $this->renderRashodForm(
                    $rashod, 
                    $newEditToken, 
                    $validation->errors()
                );
            }

            // 4. USPJEŠNA VALIDACIJA - troši token
            if ($validToken) {
                $this->consumeEditToken($validToken);
            }

            // 5. PROCESIRANJE PODATAKA
            $this->processPeriodField($rashod);
            $this->processCurrencyFields($rashod);
            
            // 6. SPREMANJE PODATAKA
            $this->rashodiTable->save($rashod);

            // 7. PREUSMJERAVANJE
            return Response::redirect('rashodi/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error saving rashod', [
                'message' => $e->getMessage(),
                'rashod' => $rashod ?? null
            ]);
            return $this->response()->renderError('Failed to save rashod');
        }
    }

    public function list(): Response 
    {
        try {
            return $this->response()->render(
                'financije/resources/views/rashodi/list.html.php',
                [
                    'title' => 'Svi rashodi'
                ],
                'Svi rashodi'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading rashodi list', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load rashodi list');
        }
    }

    public function fetchRashodi(): Response 
    {
        try {
            $requestData = $_POST;
            
            // 1. KORISTI INTERNI MAPPING
            $rashodiMapping = $this->getDataTablesMapping();
            
            // 2. PRIPREMI KOLUMNE ZA DatatablesSSP
            $columns = [];
            foreach ($rashodiMapping['columns'] as $index => $column) {
                $columns[] = [
                    'db' => $column['sqlcolumnname'],
                    'dt' => $index
                ];
            }

            // 3. POZOVI DatatablesSSP DIREKTNO (rashodi ne trebaju WHERE uvjete za uloge)
            $data = DatatablesSSP::complexWithUniques(
                $requestData,
                $this->dbConnection,
                $rashodiMapping['table_name'],
                $rashodiMapping['primary_key'],
                $columns,
                $rashodiMapping['columns'],
                null // Nema WHERE uvjeta za rashodi
            );

            // 4. DODAJ KONFIGURACIJU NA PRVI ZAHTJEV
            if (isset($requestData['init']) && $requestData['init'] === 'true') {
                $data['config'] = [
                    'columns' => $rashodiMapping['columns'],
                    'fixedColumns' => $rashodiMapping['fixed_columns'],
                    'routeName' => $rashodiMapping['route_name']
                ];
                
                if (isset($rashodiMapping['rowStyling'])) {
                    $data['config']['rowStyling'] = $rashodiMapping['rowStyling'];
                }
            }
            
            echo json_encode($data);
            exit;
        } catch (\Throwable $e) {
            $this->logger->error('Error fetching rashodi data', [
                'message' => $e->getMessage()
            ]);
            return Response::json(['error' => 'Failed to fetch rashodi data'], 500);
        }
    }

    public function export(Request $request): Response 
    {
        try {
            $rashodi = $this->rashodiTable->findAll();
            
            $columns = [
                'ID', 'Period_rada', 'vrsta_rashoda', 'cijena'
            ];
            
            $exportData = $this->exportRashodiData($rashodi, $columns, "svi rashodi");
            
            return Response::download(
                $exportData['content'],
                $exportData['filename'],
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Rashodi export failed', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id', []),
                'ip' => $request->getIp()
            ]);
            return $this->response()->renderError('Failed to export rashodi');
        }
    }

    public function delete(Request $request): Response 
    {
        try {
            $korisnik = $this->authentication->getUser();
            $rashodId = $request->getPost('id');
            $rashod = $this->rashodiTable->findById($rashodId);
            
            if (!$rashod) {
                $this->logger->security('Attempt to delete non-existent rashod record', [
                    'id' => $rashodId,
                    'user_id' => $korisnik->id,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Record not found');
            }

            if (!$korisnik->hasPermission(\Models\Entity\Korisnik::KORISNIK_MASTER)) {
                $this->logger->security('Unauthorized rashod deletion attempt', [
                    'id' => $rashodId,
                    'user_id' => $korisnik->id,
                    'ip' => $request->getIp()
                ]);
                return $this->renderRashodForm(
                    [], 
                    null, 
                    ['error' => ['Nemate potrebna prava za ovu akciju']]
                );
            }

            $this->rashodiTable->delete($rashod->ID);
            
            $this->logger->security('Rashod record deleted', [
                'id' => $rashodId,
                'deleted_by' => $korisnik->id,
                'ip' => $request->getIp()
            ]);

            return Response::redirect('rashodi/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error deleting rashod record', [
                'message' => $e->getMessage(),
                'id' => $rashodId ?? null
            ]);
            return $this->response()->renderError('Failed to delete rashod record');
        }
    }

    // HELPER METHODS
    private function processPeriodField(array &$rashod): void 
    {
        if (isset($rashod['period_input']) && !empty($rashod['period_input'])) {
            $periodData = DatabaseTableRashodi::parsePeriod($rashod['period_input']);
            $rashod['mjesec'] = $periodData['mjesec'];
            $rashod['godina'] = $periodData['godina'];
            $rashod['Period_rada'] = $rashod['period_input'];
            unset($rashod['period_input']);
        }
    }

    private function processCurrencyFields(array &$rashod): void 
    {
        if (isset($rashod['cijena']) && is_string($rashod['cijena'])) {
            $rashod['cijena'] = str_replace(',', '.', $rashod['cijena']);
        }
    }

    private function renderRashodForm(array $rashod, ?string $editToken, array $errors): Response 
    {
        if (is_array($rashod)) {
            if (!empty($rashod['Period_rada'])) {
                $rashod['formatted_period'] = $rashod['Period_rada'];
            } elseif (isset($rashod['mjesec']) && isset($rashod['godina'])) {
                $rashod['formatted_period'] = DatabaseTableRashodi::formatPeriod(
                    $rashod['mjesec'], 
                    $rashod['godina']
                );
            }
        }

        $expenseTypes = $this->rashodiTable->getExpenseTypes();

        return $this->response()->render(
            'financije/resources/views/rashodi/edit.html.php',
            [
                'errors' => $errors,
                'rashod' => is_array($rashod) ? (object)$rashod : $rashod,
                'edit_token' => $editToken,
                'user' => $this->authentication->getUser(),
                'expense_types' => $expenseTypes
            ],
            'Uredi rashod'
        );
    }

    private function exportRashodiData(array $rashodi, array $columns, string $filename): array 
    {
        try {
            $currentUser = $this->authentication->getUser();
            if (!$currentUser || $currentUser->user_permissions < Korisnik::KORISNIK_MASTER) {
                throw new \RuntimeException('Unauthorized to export rashodi');
            }
    
            if (count($rashodi) > self::MAX_EXPORT_ROWS) {
                throw new \RuntimeException(
                    sprintf('Export limit exceeded: %d rows', self::MAX_EXPORT_ROWS)
                );
            }
    
            require_once APP_ROOT . '/app/Helpers/excelexport.php';
            $savefileas = sprintf(
                "%s - %s tjedan %s",
                $filename,
                date('d-m-Y'),
                date("W", strtotime(date('d-m-Y')))
            );
    
            return excelexport('rashodi', $columns, $savefileas, $rashodi);
        } catch (\Throwable $e) {
            $this->logger->error('Rashodi export failed', [
                'message' => $e->getMessage(),
                'rows_count' => count($rashodi)
            ]);
            throw $e;
        }
    }
}