<?php

namespace financije\Models;

use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

class DatabaseTableNaplata extends DatabaseTable
{
    /**
     * @var \baseKRIZAN\Database\DatabaseConnection
     */
    protected $dbConnection;
    
    /**
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Constructor with logger support
     * 
     * @param DatabaseConnection $dbConnection Database connection
     * @param string $table Table name
     * @param string $primaryKey Primary key
     * @param string $className Class name
     * @param array $constructorArgs Constructor arguments
     * @param Logger|null $logger Logger instance
     */
    public function __construct(
        DatabaseConnection $dbConnection, 
        string $table, 
        string $primaryKey, 
        string $className = 'stdClass', 
        array $constructorArgs = [],
        ?Logger $logger = null
    ) {
        parent::__construct($dbConnection, $table, $primaryKey, $className, $constructorArgs);
        $this->dbConnection = $dbConnection;
        $this->logger = $logger;
        
        // Ensure all required tables exist
        $this->ensureTablesExist();
        
        if ($this->logger) {
            $this->logger->modules('DatabaseTableNaplata initialized', [
                'table' => $table
            ]);
        }
    }
    
    /**
     * Ensure all required tables exist
     * 
     * @return void
     */
    private function ensureTablesExist(): void
    {
        $this->ensureNaplataTableExists();
    }
    
    /**
     * Ensure naplata table exists with new structure
     * 
     * @return void
     */
    private function ensureNaplataTableExists(): void
    {
        try {
            // Check if table exists
            $sql = "SELECT 1 FROM information_schema.tables WHERE table_name = 'financije_naplata' AND table_schema = DATABASE()";
            $tableExists = $this->dbConnection->querySingleValue($sql);
            
            if (!$tableExists) {
                // Create table with new structure
                $sql = "CREATE TABLE `financije_naplata` (
                    `ID` int(11) NOT NULL AUTO_INCREMENT,
                    `Tvrtka` varchar(50) COLLATE utf8mb4_general_ci NOT NULL,
                    `Period_rada` varchar(20) COLLATE utf8mb4_general_ci DEFAULT NULL COMMENT 'Legacy field',
                    `mjesec` TINYINT(2) NOT NULL DEFAULT 1 COMMENT 'Mjesec (1-12)',
                    `godina` YEAR NOT NULL DEFAULT 2025 COMMENT 'Godina',
                    `Cijena_bez_pdv` decimal(10,2) NOT NULL,
                    `PDV` decimal(10,2) DEFAULT NULL,
                    `Ukupan_iznos` decimal(10,2) NOT NULL,
                    `Valuta_placanja` date NOT NULL,
                    `Placeno` decimal(10,2) NOT NULL,
                    `Status` varchar(30) COLLATE utf8mb4_general_ci NOT NULL,
                    PRIMARY KEY (`ID`),
                    INDEX `idx_mjesec_godina` (`mjesec`, `godina`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created financije_naplata table with new structure');
                }
            } else {
                // Check if new columns exist and add them if they don't
                $this->ensureNewColumnsExist();
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableNaplata', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureNaplataTableExists'
                ]);
            }
            throw $e;
        }
    }
    
    /**
     * Ensure new columns exist for existing tables
     * 
     * @return void
     */
    private function ensureNewColumnsExist(): void
    {
        try {
            // Check if mjesec column exists
            $sql = "SELECT 1 FROM information_schema.columns 
                   WHERE table_name = 'financije_naplata' 
                   AND column_name = 'mjesec' 
                   AND table_schema = DATABASE()";
            $mjesecExists = $this->dbConnection->querySingleValue($sql);
            
            if (!$mjesecExists) {
                $sql = "ALTER TABLE `financije_naplata` 
                       ADD COLUMN `mjesec` TINYINT(2) NOT NULL DEFAULT 1 COMMENT 'Mjesec (1-12)' AFTER `Period_rada`,
                       ADD COLUMN `godina` YEAR NOT NULL DEFAULT 2025 COMMENT 'Godina' AFTER `mjesec`,
                       ADD INDEX `idx_mjesec_godina` (`mjesec`, `godina`)";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Added new columns mjesec and godina to financije_naplata table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error adding new columns', [
                    'error' => $e->getMessage()
                ]);
            }
            // Don't throw error if columns already exist
        }
    }
    
    // Metode za financijske statistike - ažurirane za nove kolone
    /**
     * Get total amount from all invoices
     * 
     * @return float Total amount
     */
    public function getTotalAmount(): float 
    {
        try {
            $query = "SELECT SUM(Ukupan_iznos) as total FROM {$this->table}";
            $result = $this->dbConnection->querySingleValue($query);
            return (float)($result ?? 0);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting total amount', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0.0;
        }
    }
    
    /**
     * Get paid amount from all fully paid invoices
     * 
     * @return float Paid amount
     */
    public function getPaidAmount(): float 
    {
        try {
            $query = "SELECT SUM(Placeno) as total FROM {$this->table} 
                    WHERE Status LIKE '%naplaćeno u cijelosti%'";
            $result = $this->dbConnection->querySingleValue($query);
            return (float)($result ?? 0);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting paid amount', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0.0;
        }
    }
    
    /**
     * Get unpaid amount from all invoices that are not fully paid
     * 
     * @return float Unpaid amount
     */
    public function getUnpaidAmount(): float 
    {
        try {
            $query = "SELECT SUM(Ukupan_iznos - Placeno) as total FROM {$this->table} 
                    WHERE Status NOT LIKE '%naplaćeno u cijelosti%'";
            $result = $this->dbConnection->querySingleValue($query);
            return (float)($result ?? 0);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting unpaid amount', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0.0;
        }
    }
    
    /**
     * Get amount by company
     * 
     * @return array Amount by company
     */
    public function getAmountByCompany(): array 
    {
        try {
            $query = "SELECT Tvrtka as company, SUM(Ukupan_iznos) as amount 
                    FROM {$this->table} 
                    GROUP BY Tvrtka 
                    ORDER BY amount DESC";
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting amount by company', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }
    
    /**
     * Get amount by status
     * 
     * @return array Amount by status
     */
    public function getAmountByStatus(): array 
    {
        try {
            $query = "SELECT Status, SUM(Ukupan_iznos) as amount, COUNT(*) as count
                    FROM {$this->table} 
                    GROUP BY Status 
                    ORDER BY amount DESC";
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting amount by status', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }
    
    /**
     * Get amount by month using new format (mjesec/godina)
     * 
     * @return array Amount by month
     */
    public function getAmountByMonth(): array 
    {
        try {
            $query = "SELECT 
                    mjesec as month, 
                    godina as year,
                    SUM(Ukupan_iznos) as amount 
                    FROM {$this->table} 
                    GROUP BY godina, mjesec 
                    ORDER BY godina ASC, mjesec ASC";
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting amount by month', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }
    
    /**
     * Get recent items
     * 
     * @param int $limit Maximum number of items to return
     * @return array Recent items
     */
    public function getRecentItems(int $limit = 5): array 
    {
        try {
            $query = "SELECT *
                     FROM {$this->table} 
                     ORDER BY godina DESC, mjesec DESC, ID DESC
                     LIMIT " . (int)$limit;
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting recent items', [
                    'error' => $e->getMessage(),
                    'limit' => $limit
                ]);
            }
            return [];
        }
    }
    
    /**
     * Get overdue invoices (past due date but not fully paid)
     * 
     * @return array Overdue invoices
     */
    public function getOverdueInvoices(): array
    {
        try {
            $query = "SELECT *
                     FROM {$this->table} 
                     WHERE Valuta_placanja < CURDATE() 
                     AND Status NOT LIKE '%naplaćeno u cijelosti%' 
                     ORDER BY Valuta_placanja ASC";
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting overdue invoices', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }
    
    /**
     * Get upcoming invoices due within the next N days
     * 
     * @param int $days Number of days to look ahead
     * @return array Upcoming invoices
     */
    public function getUpcomingInvoices(int $days = 30): array
    {
        try {
            $query = "SELECT *
                     FROM {$this->table} 
                     WHERE Valuta_placanja BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL :days DAY) 
                     AND Status NOT LIKE '%naplaćeno u cijelosti%' 
                     ORDER BY Valuta_placanja ASC";
            $params = ['days' => $days];
            return $this->dbConnection->queryAndFetchAllAssoc($query, $params);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting upcoming invoices', [
                    'error' => $e->getMessage(),
                    'days' => $days
                ]);
            }
            return [];
        }
    }
    
    /**
     * Get quarterly summary using new format
     * 
     * @param int|null $year Year to analyze or null for current year
     * @return array Quarterly summary
     */
    public function getQuarterlySummary(?int $year = null): array
    {
        if ($year === null) {
            $year = (int)date('Y');
        }
        
        try {
            $query = "SELECT 
                    CASE 
                        WHEN mjesec IN (1,2,3) THEN 1
                        WHEN mjesec IN (4,5,6) THEN 2
                        WHEN mjesec IN (7,8,9) THEN 3
                        WHEN mjesec IN (10,11,12) THEN 4
                    END as quarter,
                    SUM(Ukupan_iznos) as total_amount,
                    SUM(Placeno) as paid_amount,
                    COUNT(*) as invoice_count
                    FROM {$this->table}
                    WHERE godina = :year
                    GROUP BY quarter
                    ORDER BY quarter ASC";
            
            $params = ['year' => $year];
            return $this->dbConnection->queryAndFetchAllAssoc($query, $params);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting quarterly summary', [
                    'error' => $e->getMessage(),
                    'year' => $year
                ]);
            }
            return [];
        }
    }
    
    /**
     * Calculate payment efficiency (percentage of invoices paid on time)
     * 
     * @param string|null $startDate Start date (Y-m-d format) or null for all time
     * @param string|null $endDate End date (Y-m-d format) or null for current date
     * @return float Payment efficiency percentage
     */
    public function getPaymentEfficiency(?string $startDate = null, ?string $endDate = null): float
    {
        try {
            $params = [];
            $whereClause = "";
            
            if ($startDate) {
                $whereClause = " WHERE Valuta_placanja >= :startDate";
                $params['startDate'] = $startDate;
                
                if ($endDate) {
                    $whereClause .= " AND Valuta_placanja <= :endDate";
                    $params['endDate'] = $endDate;
                }
            }
            
            // Count total invoices in the period
            $totalQuery = "SELECT COUNT(*) as total FROM {$this->table}" . $whereClause;
            $total = (int)$this->dbConnection->querySingleValue($totalQuery, $params);
            
            if ($total === 0) {
                return 0.0;
            }
            
            // Add condition for paid on time
            $paidOnTimeWhere = $whereClause;
            if ($paidOnTimeWhere) {
                $paidOnTimeWhere .= " AND Status LIKE '%naplaćeno u cijelosti%'";
            } else {
                $paidOnTimeWhere = " WHERE Status LIKE '%naplaćeno u cijelosti%'";
            }
            
            // Count invoices paid on time
            $paidOnTimeQuery = "SELECT COUNT(*) as paid_on_time FROM {$this->table}" . $paidOnTimeWhere;
            $paidOnTime = (int)$this->dbConnection->querySingleValue($paidOnTimeQuery, $params);
            
            // Calculate efficiency
            return ($paidOnTime / $total) * 100;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error calculating payment efficiency', [
                    'error' => $e->getMessage(),
                    'startDate' => $startDate,
                    'endDate' => $endDate
                ]);
            }
            return 0.0;
        }
    }

    /**
     * Helper method to format month-year for display
     * 
     * @param int $mjesec Month (1-12)
     * @param int $godina Year
     * @return string Formatted string (MM-YYYY)
     */
    public static function formatPeriod(int $mjesec, int $godina): string
    {
        return sprintf('%02d-%04d', $mjesec, $godina);
    }

    /**
     * Helper method to parse period string to month and year
     * 
     * @param string $period Period in format "MM-YYYY" or "M-YYYY"
     * @return array Array with 'mjesec' and 'godina' keys
     */
    public static function parsePeriod(string $period): array
    {
        if (preg_match('/^(\d{1,2})-(\d{4})$/', $period, $matches)) {
            return [
                'mjesec' => (int)$matches[1],
                'godina' => (int)$matches[2]
            ];
        }
        
        // Default to current month/year if invalid format
        return [
            'mjesec' => (int)date('n'),
            'godina' => (int)date('Y')
        ];
    }
}