<?php

namespace financije\Models;

use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

class DatabaseTableRashodi extends DatabaseTable
{
    /**
     * @var \baseKRIZAN\Database\DatabaseConnection
     */
    protected $dbConnection;
    
    /**
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Constructor with logger support
     * 
     * @param DatabaseConnection $dbConnection Database connection
     * @param string $table Table name
     * @param string $primaryKey Primary key
     * @param string $className Class name
     * @param array $constructorArgs Constructor arguments
     * @param Logger|null $logger Logger instance
     */
    public function __construct(
        DatabaseConnection $dbConnection, 
        string $table, 
        string $primaryKey, 
        string $className = 'stdClass', 
        array $constructorArgs = [],
        ?Logger $logger = null
    ) {
        parent::__construct($dbConnection, $table, $primaryKey, $className, $constructorArgs);
        $this->dbConnection = $dbConnection;
        $this->logger = $logger;
        
        // Ensure all required tables exist
        $this->ensureTablesExist();
        
        if ($this->logger) {
            $this->logger->modules('DatabaseTableRashodi initialized', [
                'table' => $table
            ]);
        }
    }
    
    /**
     * Ensure all required tables exist
     * 
     * @return void
     */
    private function ensureTablesExist(): void
    {
        $this->ensureRashodiTableExists();
    }
    
    /**
     * Ensure rashodi table exists
     * 
     * @return void
     */
    private function ensureRashodiTableExists(): void
    {
        try {
            // Check if table exists
            $sql = "SELECT 1 FROM information_schema.tables WHERE table_name = 'financije_rashodi' AND table_schema = DATABASE()";
            $tableExists = $this->dbConnection->querySingleValue($sql);
            
            if (!$tableExists) {
                // Create table
                $sql = "CREATE TABLE `financije_rashodi` (
                    `ID` int(11) NOT NULL AUTO_INCREMENT,
                    `Period_rada` varchar(20) COLLATE utf8mb4_general_ci NOT NULL COMMENT 'Format MM-YYYY',
                    `mjesec` TINYINT(2) NOT NULL COMMENT 'Mjesec (1-12)',
                    `godina` YEAR NOT NULL COMMENT 'Godina',
                    `vrsta_rashoda` varchar(100) COLLATE utf8mb4_general_ci NOT NULL COMMENT 'Vrsta rashoda',
                    `cijena` decimal(10,2) NOT NULL COMMENT 'Cijena u eurima',
                    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    PRIMARY KEY (`ID`),
                    INDEX `idx_mjesec_godina` (`mjesec`, `godina`),
                    INDEX `idx_vrsta_rashoda` (`vrsta_rashoda`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created financije_rashodi table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableRashodi', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureRashodiTableExists'
                ]);
            }
            throw $e;
        }
    }
    
    /**
     * Get total expense amount
     * 
     * @return float Total amount
     */
    public function getTotalAmount(): float 
    {
        try {
            $query = "SELECT SUM(cijena) as total FROM {$this->table}";
            $result = $this->dbConnection->querySingleValue($query);
            return (float)($result ?? 0);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting total financije_rashodi amount', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0.0;
        }
    }
    
    /**
     * Get amount by expense type
     * 
     * @return array Amount by expense type
     */
    public function getAmountByType(): array 
    {
        try {
            $query = "SELECT vrsta_rashoda as type, SUM(cijena) as amount 
                    FROM {$this->table} 
                    GROUP BY vrsta_rashoda 
                    ORDER BY amount DESC";
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting amount by expense type', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }
    
    /**
     * Get amount by month
     * 
     * @return array Amount by month
     */
    public function getAmountByMonth(): array 
    {
        try {
            $query = "SELECT 
                    mjesec as month, 
                    godina as year,
                    SUM(cijena) as amount 
                    FROM {$this->table} 
                    GROUP BY godina, mjesec 
                    ORDER BY godina ASC, mjesec ASC";
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting financije_rashodi amount by month', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }
    
    /**
     * Get recent expense items
     * 
     * @param int $limit Maximum number of items to return
     * @return array Recent items
     */
    public function getRecentItems(int $limit = 5): array 
    {
        try {
            $query = "SELECT *
                     FROM {$this->table} 
                     ORDER BY godina DESC, mjesec DESC, ID DESC
                     LIMIT " . (int)$limit;
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting recent financije_rashodi items', [
                    'error' => $e->getMessage(),
                    'limit' => $limit
                ]);
            }
            return [];
        }
    }
    
    /**
     * Get quarterly summary
     * 
     * @param int|null $year Year to analyze or null for current year
     * @return array Quarterly summary
     */
    public function getQuarterlySummary(?int $year = null): array
    {
        if ($year === null) {
            $year = (int)date('Y');
        }
        
        try {
            $query = "SELECT 
                    CASE 
                        WHEN mjesec IN (1,2,3) THEN 1
                        WHEN mjesec IN (4,5,6) THEN 2
                        WHEN mjesec IN (7,8,9) THEN 3
                        WHEN mjesec IN (10,11,12) THEN 4
                    END as quarter,
                    SUM(cijena) as total_amount,
                    COUNT(*) as expense_count
                    FROM {$this->table}
                    WHERE godina = :year
                    GROUP BY quarter
                    ORDER BY quarter ASC";
            
            $params = ['year' => $year];
            return $this->dbConnection->queryAndFetchAllAssoc($query, $params);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting quarterly summary for financije_rashodi', [
                    'error' => $e->getMessage(),
                    'year' => $year
                ]);
            }
            return [];
        }
    }
    
    /**
     * Get expense types for dropdown
     * 
     * @return array Available expense types
     */
    public function getExpenseTypes(): array
    {
        return [
            'Troškovi platnog prometa',
            'Troškovi sitnog inventara',
            'Troškovi telefona',
            'Trošak zakupa skladišta',
            'Troškovi smještaja radnika - stanarina',
            'Troškovi servisa automobila',
            'Ostali troškovi',
            'Gorivo i cestarine',
            'Trošak plaća'
        ];
    }
    
    /**
     * Helper method to format month-year for display
     * 
     * @param int $mjesec Month (1-12)
     * @param int $godina Year
     * @return string Formatted string (MM-YYYY)
     */
    public static function formatPeriod(int $mjesec, int $godina): string
    {
        return sprintf('%02d-%04d', $mjesec, $godina);
    }

    /**
     * Helper method to parse period string to month and year
     * 
     * @param string $period Period in format "MM-YYYY" or "M-YYYY"
     * @return array Array with 'mjesec' and 'godina' keys
     */
    public static function parsePeriod(string $period): array
    {
        if (preg_match('/^(\d{1,2})-(\d{4})$/', $period, $matches)) {
            return [
                'mjesec' => (int)$matches[1],
                'godina' => (int)$matches[2]
            ];
        }
        
        // Default to current month/year if invalid format
        return [
            'mjesec' => (int)date('n'),
            'godina' => (int)date('Y')
        ];
    }
}