// Enhanced financijeadmin.js - KOMPLETNI FAJL
document.addEventListener('DOMContentLoaded', function() {
  // Initialize all dashboard functionality
  initializeCounterAnimations();
  initializeInteractiveElements();
  initializeSidebar();
  initializeProgressBars();
  
  // Set up periodic refresh for real-time data
  if (typeof refreshFinancijeInterval === 'undefined') {
    let refreshFinancijeInterval = setInterval(refreshStats, 300000); // Every 5 minutes
  }
});

/**
 * Initialize counter animations for KPI values
 */
function initializeCounterAnimations() {
  // Animate dashboard cards with currency values
  const cardValues = document.querySelectorAll('.card-value');
  cardValues.forEach((valueElement, index) => {
    // Stagger the animations for better visual effect
    setTimeout(() => {
      animateCounter(valueElement, 1500);
    }, index * 200);
  });
}

/**
 * Initialize progress bars with animation
 */
function initializeProgressBars() {
  const progressBars = document.querySelectorAll('.progress-fill');
  progressBars.forEach((bar, index) => {
    const targetWidth = bar.style.width;
    bar.style.width = '0%';
    
    setTimeout(() => {
      bar.style.width = targetWidth;
    }, 500 + (index * 300));
  });
}

/**
 * Animate a counter element from 0 to its final value
 */
function animateCounter(element, duration) {
  const text = element.textContent.trim();
  let finalValue, suffix = '', isPercentage = false, isCurrency = false, isNegative = false;
  
  // Check if value is negative
  if (text.startsWith('-')) {
    isNegative = true;
  }
  
  // Handle percentages
  if (text.includes('%')) {
    finalValue = parseFloat(text.replace('%', '').replace('-', ''));
    suffix = '%';
    isPercentage = true;
  }
  // Handle currency values (with comma as decimal separator)
  else if (text.includes(',')) {
    // Convert from Croatian format (123.456,78) to standard format
    const cleanText = text.replace('-', '').replace(/\./g, '').replace(',', '.');
    finalValue = parseFloat(cleanText);
    isCurrency = true;
  }
  // Handle pure numbers
  else if (/^-?\d+$/.test(text.replace('-', ''))) {
    finalValue = parseInt(text.replace('-', ''));
  }
  // Skip non-numeric values
  else {
    return;
  }
  
  if (isNaN(finalValue)) return;
  
  // Apply negative sign if needed
  if (isNegative) {
    finalValue = -finalValue;
  }
  
  // Store original formatted value
  const originalValue = text;
  element.setAttribute('data-original', originalValue);
  
  let startTimestamp = null;
  const startValue = 0;
  
  const step = (timestamp) => {
    if (!startTimestamp) startTimestamp = timestamp;
    const progress = Math.min((timestamp - startTimestamp) / duration, 1);
    
    // Use easing function for smoother animation
    const easedProgress = easeOutCubic(progress);
    const currentValue = easedProgress * (finalValue - startValue) + startValue;
    
    if (isPercentage) {
      element.textContent = currentValue.toFixed(1) + suffix;
    } else if (isCurrency) {
      element.textContent = formatCurrency(currentValue);
    } else {
      element.textContent = Math.round(currentValue) + suffix;
    }
    
    if (progress < 1) {
      window.requestAnimationFrame(step);
    } else {
      // Ensure final value is exact
      element.textContent = originalValue;
    }
  };
  
  window.requestAnimationFrame(step);
}

/**
 * Easing function for smoother animations
 */
function easeOutCubic(t) {
  return 1 - Math.pow(1 - t, 3);
}

/**
 * Initialize interactive elements
 */
function initializeInteractiveElements() {
  // Add hover effects to recent items
  const recentItems = document.querySelectorAll('.recent-item');
  recentItems.forEach(item => {
    item.addEventListener('mouseenter', function() {
      this.style.transform = 'translateX(5px)';
      this.style.boxShadow = '0 6px 12px var(--shadow)';
    });
    
    item.addEventListener('mouseleave', function() {
      this.style.transform = 'translateX(0)';
      this.style.boxShadow = '';
    });
  });
  
  // Add hover effects to dashboard cards
  const dashboardCards = document.querySelectorAll('.dashboard-card');
  dashboardCards.forEach(card => {
    card.addEventListener('mouseenter', function() {
      this.style.transform = 'translateY(-8px) scale(1.02)';
      this.style.boxShadow = '0 12px 24px var(--shadow)';
    });
    
    card.addEventListener('mouseleave', function() {
      this.style.transform = 'translateY(0) scale(1)';
      this.style.boxShadow = '0 4px 12px var(--shadow)';
    });
  });
  
  // Add click handlers to dashboard cards for potential drill-down
  dashboardCards.forEach((card, index) => {
    card.addEventListener('click', function() {
      const label = this.querySelector('.card-label').textContent;
      console.log('Clicked on KPI:', label);
      
      // Add visual feedback
      this.style.transform = 'scale(0.95)';
      setTimeout(() => {
        this.style.transform = '';
      }, 150);
      
      // Potential navigation based on card type
      switch(index) {
        case 0: // Ukupna naplata
          console.log('Navigate to total revenue details');
          break;
        case 1: // Plaćeno
          console.log('Navigate to paid amount details');
          break;
        case 2: // Nenaplaćeno
          console.log('Navigate to unpaid amount details');
          break;
        case 3: // Rashodi
          console.log('Navigate to expenses details');
          break;
        case 4: // Trenutna dobit
          console.log('Navigate to current profit details');
          break;
        case 5: // Očekivana dobit
          console.log('Navigate to expected profit details');
          break;
        case 6: // Trenutna marža
          console.log('Navigate to current margin details');
          break;
        case 7: // Očekivana marža
          console.log('Navigate to expected margin details');
          break;
      }
    });
  });
  
  // Add click handlers to export options
  const exportOptions = document.querySelectorAll('.export-option');
  exportOptions.forEach(option => {
    option.addEventListener('click', function(e) {
      if (!e.target.closest('.export-button')) {
        // Clicked on option but not button - highlight the button
        const button = this.querySelector('.export-button');
        if (button) {
          button.style.transform = 'scale(1.15) rotate(360deg)';
          button.style.transition = 'transform 0.3s ease';
          setTimeout(() => {
            button.style.transform = '';
          }, 300);
        }
      }
    });
  });

  // Add progress bar click handlers for interactivity
  const progressItems = document.querySelectorAll('.progress-item');
  progressItems.forEach(item => {
    item.addEventListener('click', function() {
      const progressBar = this.querySelector('.progress-fill');
      if (progressBar) {
        // Add pulse effect
        progressBar.style.animation = 'none';
        setTimeout(() => {
          progressBar.style.animation = 'progressPulse 0.6s ease-out';
        }, 10);
      }
    });
  });
}

/**
 * Initialize sidebar functionality
 */
function initializeSidebar() {
  const mobileToggle = document.getElementById('mobile-toggle');
  const sidebar = document.querySelector('.sidebar');
  
  // Mobile sidebar toggle
  if (mobileToggle) {
    mobileToggle.addEventListener('click', function() {
      sidebar.classList.toggle('open');
      
      // Change icon based on state
      const icon = mobileToggle.querySelector('i');
      if (sidebar.classList.contains('open')) {
        icon.classList.remove('fa-bars');
        icon.classList.add('fa-times');
        mobileToggle.style.transform = 'rotate(90deg)';
      } else {
        icon.classList.remove('fa-times');
        icon.classList.add('fa-bars');
        mobileToggle.style.transform = 'rotate(0deg)';
      }
    });
  }
  
  // Initialize sidebar items
  const sidebarItems = document.querySelectorAll('.sidebar-item');
  sidebarItems.forEach(item => {
    const link = item.querySelector('.sidebar-link');
    if (link) {
      link.addEventListener('click', function(e) {
        e.preventDefault();
        
        // Toggle active state on the clicked item
        const wasActive = item.classList.contains('active');
        
        // If we're on mobile, don't close the sidebar when toggling items
        const isMobile = window.innerWidth <= 768;
        
        // First, check if we should close other items
        if (!e.ctrlKey) { // Allow keeping multiple items open with Ctrl key
          sidebarItems.forEach(i => {
            if (i !== item) { // Don't touch the clicked item yet
              i.classList.remove('active');
            }
          });
        }
        
        // Toggle the clicked item with animation
        if (wasActive) {
          item.classList.remove('active');
        } else {
          item.classList.add('active');
          // Add subtle animation
          item.style.animation = 'slideInLeft 0.3s ease-out';
          setTimeout(() => {
            item.style.animation = '';
          }, 300);
        }
        
        // Prevent closing sidebar on mobile when clicking items
        e.stopPropagation();
      });
    }
  });
  
  // Handle clicks on submenu links
  const submenuLinks = document.querySelectorAll('.sidebar-submenu li a');
  submenuLinks.forEach(link => {
    link.addEventListener('click', function(e) {
      // Add visual feedback
      this.style.transform = 'translateX(5px)';
      setTimeout(() => {
        this.style.transform = '';
      }, 200);
      
      // Allow normal navigation for these links
      // If we're on mobile, close the sidebar after clicking a submenu link
      if (window.innerWidth <= 768) {
        setTimeout(() => {
          sidebar.classList.remove('open');
          if (mobileToggle) {
            const icon = mobileToggle.querySelector('i');
            icon.classList.remove('fa-times');
            icon.classList.add('fa-bars');
            mobileToggle.style.transform = 'rotate(0deg)';
          }
        }, 300);
      }
    });
  });
}

/**
 * Function to refresh stats via AJAX
 */
function refreshStats() {
  fetch('/financije/api/admin-stats', {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    }
  })
    .then(response => {
      if (!response.ok) {
        throw new Error('Network response was not ok');
      }
      return response.json();
    })
    .then(data => {
      updateDashboardValues(data);
      showUpdateIndicator();
    })
    .catch(error => {
      console.error('Error refreshing stats:', error);
      showErrorIndicator();
    });
}

/**
 * Update dashboard values with new data
 */
function updateDashboardValues(data) {
  // Update main dashboard cards with the new structure
  if (data.naplata) {
    updateElementValue('.total-revenue-card .card-value', data.naplata.totalAmount, 'currency');
    updateElementValue('.paid-card .card-value', data.naplata.paidAmount, 'currency');
    updateElementValue('.unpaid-card .card-value', data.naplata.unpaidAmount, 'currency');
  }
  
  if (data.rashodi) {
    updateElementValue('.expenses-card .card-value', data.rashodi.totalAmount, 'currency');
  }
  
  // Calculate and update profit values
  if (data.naplata && data.rashodi) {
    const currentProfit = data.naplata.paidAmount - data.rashodi.totalAmount;
    const expectedProfit = data.naplata.totalAmount - data.rashodi.totalAmount;
    const currentMargin = data.naplata.paidAmount > 0 ? (currentProfit / data.naplata.paidAmount) * 100 : 0;
    const expectedMargin = data.naplata.totalAmount > 0 ? (expectedProfit / data.naplata.totalAmount) * 100 : 0;
    
    updateElementValue('.current-profit-card .card-value', currentProfit, 'currency');
    updateElementValue('.expected-profit-card .card-value', expectedProfit, 'currency');
    updateElementValue('.current-margin-card .card-value', currentMargin, '%');
    updateElementValue('.expected-margin-card .card-value', expectedMargin, '%');
  }
  
  // Update progress bars
  updateProgressBars(data);
}

/**
 * Update progress bars with new data
 */
function updateProgressBars(data) {
  if (data.naplata) {
    const paidPercentage = data.naplata.totalAmount > 0 ? (data.naplata.paidAmount / data.naplata.totalAmount) * 100 : 0;
    const progressBar = document.querySelector('.naplata-progress');
    if (progressBar) {
      progressBar.style.width = Math.min(paidPercentage, 100) + '%';
    }
    
    // Update progress value text
    const progressValue = document.querySelector('.progress-item:first-child .progress-value');
    if (progressValue) {
      progressValue.textContent = paidPercentage.toFixed(1) + '%';
    }
  }
}

/**
 * Update a specific element's value with animation
 */
function updateElementValue(selector, newValue, type = '') {
  const element = document.querySelector(selector);
  if (!element || newValue === undefined) return;
  
  let currentValue, targetValue;
  
  if (type === 'currency') {
    // Parse Croatian currency format
    const currentText = element.textContent.replace('-', '').replace(/\./g, '').replace(',', '.');
    currentValue = parseFloat(currentText) || 0;
    if (element.textContent.includes('-')) currentValue = -currentValue;
    targetValue = parseFloat(newValue) || 0;
  } else if (type === '%') {
    currentValue = parseFloat(element.textContent.replace('%', '').replace('-', '')) || 0;
    if (element.textContent.includes('-')) currentValue = -currentValue;
    targetValue = parseFloat(newValue) || 0;
  } else {
    currentValue = parseFloat(element.textContent.replace(/[^\d.-]/g, '')) || 0;
    targetValue = parseFloat(newValue) || 0;
  }
  
  // Only animate if value has changed significantly
  if (Math.abs(currentValue - targetValue) > 0.01) {
    // Add update flash effect
    element.classList.add('value-updated');
    
    // Animate to new value
    animateValueChange(element, currentValue, targetValue, type, 1000);
    
    // Remove flash effect after animation
    setTimeout(() => {
      element.classList.remove('value-updated');
    }, 1200);
  }
}

/**
 * Animate value change from current to new value
 */
function animateValueChange(element, startValue, endValue, type, duration) {
  let startTimestamp = null;
  
  const step = (timestamp) => {
    if (!startTimestamp) startTimestamp = timestamp;
    const progress = Math.min((timestamp - startTimestamp) / duration, 1);
    
    // Use easing for smooth animation
    const easedProgress = easeOutCubic(progress);
    const currentValue = startValue + (endValue - startValue) * easedProgress;
    
    if (type === '%') {
      element.textContent = currentValue.toFixed(1) + '%';
    } else if (type === 'currency') {
      element.textContent = formatCurrency(currentValue);
    } else {
      element.textContent = Math.round(currentValue);
    }
    
    if (progress < 1) {
      window.requestAnimationFrame(step);
    } else {
      // Ensure final value is exact
      if (type === '%') {
        element.textContent = endValue.toFixed(1) + '%';
      } else if (type === 'currency') {
        element.textContent = formatCurrency(endValue);
      } else {
        element.textContent = Math.round(endValue);
      }
    }
  };
  
  window.requestAnimationFrame(step);
}

/**
 * Format currency value (Croatian format)
 */
function formatCurrency(value) {
  const absValue = Math.abs(value);
  const formatted = Number(absValue).toLocaleString('hr-HR', {
    minimumFractionDigits: 2,
    maximumFractionDigits: 2
  });
  
  return value < 0 ? '-' + formatted : formatted;
}

/**
 * Show update indicator
 */
function showUpdateIndicator() {
  // Create or update a small indicator showing last update time
  let indicator = document.querySelector('.update-indicator');
  if (!indicator) {
    indicator = document.createElement('div');
    indicator.className = 'update-indicator';
    document.querySelector('.content-header').appendChild(indicator);
  }
  
  const now = new Date();
  indicator.innerHTML = `
    <i class="fas fa-sync-alt"></i>
    <span>Ažurirano: ${now.toLocaleTimeString('hr-HR')}</span>
  `;
  
  // Add pulse animation
  indicator.classList.add('updated');
  setTimeout(() => {
    indicator.classList.remove('updated');
  }, 1000);
}

/**
 * Show error indicator
 */
function showErrorIndicator() {
  let indicator = document.querySelector('.update-indicator');
  if (!indicator) {
    indicator = document.createElement('div');
    indicator.className = 'update-indicator error';
    document.querySelector('.content-header').appendChild(indicator);
  }
  
  indicator.innerHTML = `
    <i class="fas fa-exclamation-triangle"></i>
    <span>Greška pri ažuriranju</span>
  `;
  
  indicator.classList.add('error');
  setTimeout(() => {
    indicator.classList.remove('error');
  }, 3000);
}

/**
 * Show loading indicator
 */
function showLoadingIndicator() {
  let indicator = document.querySelector('.update-indicator');
  if (!indicator) {
    indicator = document.createElement('div');
    indicator.className = 'update-indicator';
    document.querySelector('.content-header').appendChild(indicator);
  }
  
  indicator.innerHTML = `
    <i class="fas fa-spinner fa-spin"></i>
    <span>Ažuriranje...</span>
  `;
}

/**
 * Manual refresh function that can be called by user
 */
function manualRefresh() {
  showLoadingIndicator();
  refreshStats();
}

/**
 * Add keyboard shortcuts for power users
 */
document.addEventListener('keydown', function(e) {
  // Ctrl + R for manual refresh
  if (e.ctrlKey && e.key === 'r') {
    e.preventDefault();
    manualRefresh();
  }
  
  // Ctrl + E for export (focus on first export option)
  if (e.ctrlKey && e.key === 'e') {
    e.preventDefault();
    const firstExportOption = document.querySelector('.export-option');
    if (firstExportOption) {
      firstExportOption.scrollIntoView({ behavior: 'smooth' });
      firstExportOption.style.animation = 'pulse 0.6s ease-out';
      setTimeout(() => {
        firstExportOption.style.animation = '';
      }, 600);
    }
  }
});

/**
 * Add CSS animations dynamically
 */
function addDynamicAnimations() {
  const style = document.createElement('style');
  style.textContent = `
    @keyframes progressPulse {
      0% { transform: scaleY(1); }
      50% { transform: scaleY(1.1); }
      100% { transform: scaleY(1); }
    }
    
    @keyframes slideInLeft {
      from { transform: translateX(-10px); opacity: 0.7; }
      to { transform: translateX(0); opacity: 1; }
    }
    
    @keyframes pulse {
      0% { transform: scale(1); }
      50% { transform: scale(1.05); }
      100% { transform: scale(1); }
    }
  `;
  document.head.appendChild(style);
}

// Initialize dynamic animations when DOM is ready
document.addEventListener('DOMContentLoaded', addDynamicAnimations);