<?php

namespace pm\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Controllers\BaseController;
use Models\DatabaseTable;
use pm\Models\DatabaseTablePM;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Validation\Validator;
use baseKRIZAN\Services\NotificationEnhancerService;
use baseKRIZAN\Services\NotificationService;
use baseKRIZAN\Services\NotificationTypeRegistry;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use Models\Entity\Korisnik;

class PM extends BaseController
{
    private $korisniciTable;
    private $pmTable;
    private Validator $validator;
    private NotificationEnhancerService $notificationEnhancer;
    private NotificationService $notificationService;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTablePM $pmTable, 
        Validator $validator,
        Logger $logger,
        NotificationEnhancerService $notificationEnhancer,
        NotificationService $notificationService,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->pmTable = $pmTable;
        $this->validator = $validator;
        $this->notificationEnhancer = $notificationEnhancer;
        $this->notificationService = $notificationService;
        
        // Register PM notification types directly in the controller
        $this->registerPMNotificationTypes();
    }
    
    /**
     * Register PM notification types with the registry
     */
    protected function registerPMNotificationTypes(): void
    {
        // Register the pm_zadatak notification type with basic information
        NotificationTypeRegistry::register(
            'pm_zadatak',
            'Novi PM zadatak',
            'pm/dashboard'
        );

        // Register formatter for PM task notifications
        NotificationTypeRegistry::registerFormatter(
            'pm_zadatak',
            function (array $data, string $senderName) {
                return sprintf(
                    'Zadatak "%s" kreiran od strane %s',
                    $data['title'] ?? 'Novi zadatak',
                    $senderName
                );
            }
        );
    }

    public function dashboard() 
    {
        return $this->response()->render(
            'pm/resources/views/pm/dashboard.html.php',
            ['title' => 'Project Management Dashboard'],
            'Dashboard'
        );
    }

    public function gettasks() 
    {
        if ($_SERVER['REQUEST_METHOD'] === 'GET') {
            // Log successful API call
            $this->logger->modules('PM: Get tasks API called', [
                'user' => $this->authentication->getUser()->id ?? 'unknown',
                'remote_addr' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
            ]);
            
            return Response::json($this->pmTable->getAllTasks());
        }
        
        // Return error for non-GET requests
        return Response::json(['error' => 'Method not allowed'], 405);
    }

    /**
     * Add a new task
     * 
     * Required fields: title, description, status
     */
    public function addtask(Request $request)
    {
        try {
            if ($request->getMethod() !== 'POST') {
                return Response::json(['error' => 'Method not allowed'], 405);
            }
            
            $data = json_decode(file_get_contents("php://input"), true);
            
            // Validate required fields
            if (empty($data['title']) || empty($data['description']) || empty($data['status'])) {
                $missingFields = [];
                if (empty($data['title'])) $missingFields[] = 'title';
                if (empty($data['description'])) $missingFields[] = 'description';
                if (empty($data['status'])) $missingFields[] = 'status';
                
                $this->logger->modules('PM: Missing required fields for addtask', [
                    'missing_fields' => $missingFields,
                    'data' => $data
                ]);
                
                return Response::json([
                    'success' => false,
                    'message' => 'Missing required fields: ' . implode(', ', $missingFields)
                ], 400);
            }

            // Add task
            $taskId = $this->pmTable->addTask(
                $data['title'],
                $data['description'],
                $data['status']
            );

            if ($taskId) {
                // Get current user (task creator)
                $currentUser = $this->authentication->getUser();
                
                // Emit event for task creation
                $this->emitTaskCreatedEvent($taskId, $currentUser->id);
                
                // Prepare notification data
                $notificationData = [
                    'parent_id' => 0,
                    'from_id' => $currentUser->id,
                    'title' => $data['title'],
                    'message' => $data['description'],
                    'type' => 'pm_zadatak',
                    'related_id' => $taskId,
                    'route' => 'pm/dashboard',
                    'send_push' => true // Explicitly enable push notifications for this notification
                ];
                
                // Get admin users as recipients
                $allUsers = $this->korisniciTable->findAll();
                $recipients = [];
                
                foreach ($allUsers as $user) {
                    if ($user->id !== $currentUser->id && 
                        ($user->hasPermission(Korisnik::KORISNIK_ADMIN) || 
                        $user->hasPermission(Korisnik::KORISNIK_MASTER))) {
                        $recipients[] = $user->id;
                    }
                }
                
                // Send enhanced notifications for PM tasks
                // This will use FCM and Realtime DB if enabled in config
                $this->notificationEnhancer->createNotification(
                    $notificationData, 
                    $recipients
                );
                
                $this->logger->modules('PM: Task added successfully', [
                    'task_id' => $taskId,
                    'user' => $currentUser->id,
                    'title' => $data['title'],
                    'fcm_enabled' => $this->notificationEnhancer->isFcmEnabled(),
                    'realtime_db_enabled' => $this->notificationEnhancer->isRealtimeDbEnabled()
                ]);

                return Response::json([
                    'success' => true,
                    'message' => 'Task added successfully',
                    'taskId' => $taskId
                ]);
            }

            $this->logger->error('PM: Failed to add task', [
                'data' => $data
            ]);
            
            return Response::json([
                'success' => false,
                'message' => 'Error adding task'
            ], 500);

        } catch (\Exception $e) {
            $this->logger->error('PM: Exception adding task', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'data' => $data ?? null
            ]);

            return Response::json([
                'success' => false,
                'message' => 'Internal server error: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update task status
     * 
     * Required fields: id, status
     */
    public function updatetask(Request $request) 
    {
        try {
            if ($request->getMethod() !== 'POST') {
                return Response::json(['error' => 'Method not allowed'], 405);
            }
            
            $data = json_decode(file_get_contents("php://input"), true);
            
            // Validate required fields
            if (!isset($data['id']) || !isset($data['status'])) {
                $missingFields = [];
                if (!isset($data['id'])) $missingFields[] = 'id';
                if (!isset($data['status'])) $missingFields[] = 'status';
                
                $this->logger->modules('PM: Missing required fields for updatetask', [
                    'missing_fields' => $missingFields,
                    'data' => $data
                ]);
                
                return Response::json([
                    'success' => false,
                    'message' => 'Missing required fields: ' . implode(', ', $missingFields)
                ], 400);
            }

            $success = $this->pmTable->updateTaskStatus(
                (int)$data['id'],
                $data['status']
            );
            
            if ($success) {
                // Get current user
                $currentUser = $this->authentication->getUser();
                
                // Emit event for project update
                $this->emitProjectUpdatedEvent((int)$data['id'], $currentUser->id);
                
                $this->logger->modules('PM: Task status updated successfully', [
                    'task_id' => (int)$data['id'],
                    'status' => $data['status'],
                    'user' => $currentUser->id
                ]);
            } else {
                $this->logger->modules('PM: Failed to update task status', [
                    'task_id' => (int)$data['id'],
                    'status' => $data['status']
                ]);
            }

            return Response::json([
                'success' => $success,
                'message' => $success ? 'Task updated successfully' : 'Error updating task'
            ]);
        } catch (\Exception $e) {
            $this->logger->error('PM: Exception updating task status', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'data' => $data ?? null
            ]);

            return Response::json([
                'success' => false,
                'message' => 'Internal server error: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Edit task details
     * 
     * Required fields: id, title, description
     */
    public function edittask(Request $request)
    {
        try {
            if ($request->getMethod() !== 'POST') {
                return Response::json(['error' => 'Method not allowed'], 405);
            }
            
            $data = json_decode(file_get_contents("php://input"), true);
            
            // Validate required fields
            if (!isset($data['id']) || !isset($data['title']) || !isset($data['description'])) {
                $missingFields = [];
                if (!isset($data['id'])) $missingFields[] = 'id';
                if (!isset($data['title'])) $missingFields[] = 'title';
                if (!isset($data['description'])) $missingFields[] = 'description';
                
                $this->logger->modules('PM: Missing required fields for edittask', [
                    'missing_fields' => $missingFields,
                    'data' => $data
                ]);
                
                return Response::json([
                    'success' => false,
                    'message' => 'Missing required fields: ' . implode(', ', $missingFields)
                ], 400);
            }

            $success = $this->pmTable->updateTask(
                (int)$data['id'],
                $data['title'],
                $data['description']
            );
            
            if ($success) {
                // Get current user
                $currentUser = $this->authentication->getUser();
                
                // Emit event for project update
                $this->emitProjectUpdatedEvent((int)$data['id'], $currentUser->id);
                
                $this->logger->modules('PM: Task updated successfully', [
                    'task_id' => (int)$data['id'],
                    'user' => $currentUser->id
                ]);
            } else {
                $this->logger->modules('PM: Failed to update task', [
                    'task_id' => (int)$data['id']
                ]);
            }

            return Response::json([
                'success' => $success,
                'message' => $success ? 'Task updated successfully' : 'Error updating task'
            ]);
        } catch (\Exception $e) {
            $this->logger->error('PM: Exception editing task', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'data' => $data ?? null
            ]);

            return Response::json([
                'success' => false,
                'message' => 'Internal server error: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete task
     * 
     * Required fields: id
     */
    public function deletetask(Request $request)
    {
        try {
            if ($request->getMethod() !== 'POST') {
                return Response::json(['error' => 'Method not allowed'], 405);
            }
            
            $data = json_decode(file_get_contents("php://input"), true);
            
            // Validate required fields
            if (!isset($data['id'])) {
                $this->logger->modules('PM: Missing task ID for deletetask', [
                    'data' => $data
                ]);
                
                return Response::json([
                    'success' => false,
                    'message' => 'Missing task ID'
                ], 400);
            }

            $success = $this->pmTable->deleteTask((int)$data['id']);
            
            if ($success) {
                $this->logger->modules('PM: Task deleted successfully', [
                    'task_id' => (int)$data['id'],
                    'user' => $this->authentication->getUser()->id ?? 'unknown'
                ]);
            } else {
                $this->logger->modules('PM: Failed to delete task', [
                    'task_id' => (int)$data['id']
                ]);
            }

            return Response::json([
                'success' => $success,
                'message' => $success ? 'Task deleted successfully' : 'Error deleting task'
            ]);
        } catch (\Exception $e) {
            $this->logger->error('PM: Exception deleting task', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'data' => $data ?? null
            ]);

            return Response::json([
                'success' => false,
                'message' => 'Internal server error: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Emitira događaj za kreiranje novog zadatka
     * 
     * @param int $taskId ID zadatka
     * @param int $userId ID korisnika
     * @return void
     */
    private function emitTaskCreatedEvent(int $taskId, int $userId): void
    {
        // Provjeri postoji li event dispatcher
        if ($this->container && $this->container->has('eventDispatcher')) {
            $eventDispatcher = $this->container->get('eventDispatcher');
            
            $eventDispatcher->dispatch('pm.task_created', [
                'task_id' => $taskId,
                'user_id' => $userId,
                'timestamp' => time()
            ]);
            
            $this->logger->modules('PM task created event emitted', [
                'task_id' => $taskId,
                'user_id' => $userId
            ]);
        }
    }
    
    /**
     * Emitira događaj za ažuriranje projekta
     * 
     * @param int $projectId ID projekta
     * @param int $userId ID korisnika
     * @return void
     */
    private function emitProjectUpdatedEvent(int $projectId, int $userId): void
    {
        // Provjeri postoji li event dispatcher
        if ($this->container && $this->container->has('eventDispatcher')) {
            $eventDispatcher = $this->container->get('eventDispatcher');
            
            $eventDispatcher->dispatch('pm.project_updated', [
                'project_id' => $projectId,
                'user_id' => $userId,
                'timestamp' => time()
            ]);
            
            $this->logger->modules('Project updated event emitted', [
                'project_id' => $projectId,
                'user_id' => $userId
            ]);
        }
    }
}