<?php

namespace pm\Models;

use Models\DatabaseTable;
use baseKRIZAN\Database\DatabaseConnection;
use baseKRIZAN\Error\Logger;

class DatabaseTablePM extends DatabaseTable
{
    /**
     * @var \baseKRIZAN\Database\DatabaseConnection
     */
    protected $dbConnection;
    
    /**
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Constructor with logger support
     * 
     * @param DatabaseConnection $dbConnection Database connection
     * @param string $table Table name
     * @param string $primaryKey Primary key
     * @param string $className Class name
     * @param array $constructorArgs Constructor arguments
     * @param Logger|null $logger Logger instance
     */
    public function __construct(
        DatabaseConnection $dbConnection, 
        string $table, 
        string $primaryKey, 
        string $className = 'stdClass', 
        array $constructorArgs = [],
        ?Logger $logger = null
    ) {
        parent::__construct($dbConnection, $table, $primaryKey, $className, $constructorArgs);
        $this->dbConnection = $dbConnection;
        $this->logger = $logger;
        
        // Ensure all required tables exist
        $this->ensureTablesExist();
        
        if ($this->logger) {
            $this->logger->modules('DatabaseTablePM initialized', [
                'table' => $table
            ]);
        }
    }
    
    /**
     * Ensure all required tables exist
     * 
     * @return void
     */
    private function ensureTablesExist(): void
    {
        $this->ensureProjectManagementTableExists();
    }
    
    /**
     * Ensure project_management table exists
     * 
     * @return void
     */
    private function ensureProjectManagementTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'project_management' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                // Create table
                $sql = "CREATE TABLE `project_management` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `title` varchar(255) COLLATE utf8mb4_general_ci NOT NULL,
                    `description` text COLLATE utf8mb4_general_ci DEFAULT NULL,
                    `pm_status` enum('todo','in_progress','done') COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'todo',
                    `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
                    `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created project_management table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTablePM', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureProjectManagementTableExists'
                ]);
            }
            throw $e;
        }
    }
    
    /**
     * Get all tasks
     * 
     * @return array All tasks
     */
    public function getAllTasks(): array
    {
        try {
            $query = "SELECT * FROM project_management ORDER BY created_at DESC";
            // Use parent's query method
            $result = $this->dbConnection->queryAndFetchAllAssoc($query);
            return $result;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting all tasks', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Add a new task
     * 
     * @param string $title Task title
     * @param string $description Task description
     * @param string $status Task status
     * @return int New task ID or 0 on failure
     */
    public function addTask(string $title, string $description, string $status): int
    {
        $query = "INSERT INTO project_management (title, description, pm_status) VALUES (?, ?, ?)";
        try {
            // Use the execute method from DatabaseConnection
            $this->dbConnection->execute($query, [$title, $description, $status]);
            
            // Get the last insert ID directly from dbConnection
            $taskId = (int)$this->dbConnection->lastInsertId();
            
            if ($this->logger) {
                $this->logger->modules('Added new task', [
                    'task_id' => $taskId,
                    'title' => $title
                ]);
            }
            
            return $taskId;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Failed to add task', [
                    'error' => $e->getMessage(),
                    'title' => $title
                ]);
            }
            throw new \Exception("Failed to add task: " . $e->getMessage());
        }
    }

    /**
     * Update task status
     * 
     * @param int $taskId Task ID
     * @param string $status New status
     * @return bool Success status
     */
    public function updateTaskStatus(int $taskId, string $status): bool
    {
        try {
            $query = "UPDATE project_management SET pm_status = ? WHERE id = ?";
            $rowsAffected = $this->dbConnection->execute($query, [$status, $taskId]);
            
            $success = $rowsAffected > 0;
            
            if ($success && $this->logger) {
                $this->logger->modules('Updated task status', [
                    'task_id' => $taskId,
                    'new_status' => $status
                ]);
            }
            
            return $success;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Failed to update task status', [
                    'error' => $e->getMessage(),
                    'task_id' => $taskId
                ]);
            }
            return false;
        }
    }

    /**
     * Update task details
     * 
     * @param int $taskId Task ID
     * @param string $title New title
     * @param string $description New description
     * @return bool Success status
     */
    public function updateTask(int $taskId, string $title, string $description): bool
    {
        try {
            $query = "UPDATE project_management SET title = ?, description = ? WHERE id = ?";
            $rowsAffected = $this->dbConnection->execute($query, [$title, $description, $taskId]);
            
            $success = $rowsAffected > 0;
            
            if ($success && $this->logger) {
                $this->logger->modules('Updated task details', [
                    'task_id' => $taskId,
                    'title' => $title
                ]);
            }
            
            return $success;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Failed to update task details', [
                    'error' => $e->getMessage(),
                    'task_id' => $taskId
                ]);
            }
            return false;
        }
    }

    /**
     * Delete a task
     * 
     * @param int $taskId Task ID
     * @return bool Success status
     */
    public function deleteTask(int $taskId): bool
    {
        try {
            $query = "DELETE FROM project_management WHERE id = ?";
            $rowsAffected = $this->dbConnection->execute($query, [$taskId]);
            
            $success = $rowsAffected > 0;
            
            if ($success && $this->logger) {
                $this->logger->modules('Deleted task', [
                    'task_id' => $taskId
                ]);
            }
            
            return $success;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Failed to delete task', [
                    'error' => $e->getMessage(),
                    'task_id' => $taskId
                ]);
            }
            return false;
        }
    }
    
    /**
     * Get tasks by status
     * 
     * @param string $status Status to filter by
     * @return array Tasks with the specified status
     */
    public function getTasksByStatus(string $status): array
    {
        try {
            $query = "SELECT * FROM project_management WHERE pm_status = ? ORDER BY created_at DESC";
            return $this->dbConnection->queryAndFetchAllAssoc($query, [$status]);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting tasks by status', [
                    'error' => $e->getMessage(),
                    'status' => $status
                ]);
            }
            return [];
        }
    }
    
    /**
     * Search tasks by title or description
     * 
     * @param string $searchTerm Search term
     * @return array Matching tasks
     */
    public function searchTasks(string $searchTerm): array
    {
        try {
            $searchTerm = '%' . $searchTerm . '%';
            $query = "SELECT * FROM project_management 
                     WHERE title LIKE ? OR description LIKE ? 
                     ORDER BY created_at DESC";
            return $this->dbConnection->queryAndFetchAllAssoc($query, [$searchTerm, $searchTerm]);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error searching tasks', [
                    'error' => $e->getMessage(),
                    'search_term' => $searchTerm
                ]);
            }
            return [];
        }
    }
    
    /**
     * Get task status distribution
     * 
     * @return array Status distribution
     */
    public function getStatusDistribution(): array
    {
        try {
            $query = "SELECT pm_status, COUNT(*) as count FROM project_management GROUP BY pm_status";
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting status distribution', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }
    
    /**
     * Get recent tasks
     * 
     * @param int $limit Maximum number of tasks to return
     * @return array Recent tasks
     */
    public function getRecentTasks(int $limit = 5): array
    {
        try {
            $query = "SELECT * FROM project_management ORDER BY created_at DESC LIMIT " . (int)$limit;
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting recent tasks', [
                    'error' => $e->getMessage(),
                    'limit' => $limit
                ]);
            }
            return [];
        }
    }
    
    /**
     * Get recently updated tasks
     * 
     * @param int $limit Maximum number of tasks to return
     * @return array Recently updated tasks
     */
    public function getRecentlyUpdatedTasks(int $limit = 5): array
    {
        try {
            $query = "SELECT * FROM project_management ORDER BY updated_at DESC LIMIT " . (int)$limit;
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting recently updated tasks', [
                    'error' => $e->getMessage(),
                    'limit' => $limit
                ]);
            }
            return [];
        }
    }
}