<?php
// moduli/pm/pm_events.php

/**
 * Definicija slušatelja događaja za modul pm
 * 
 * @return array Lista slušatelja događaja
 */
 
return [
    // Događaji iz sustava modula
    'module.activated' => [
        function($event) {
            $moduleId = $event->get('id');
            
            // Dohvatimo container iz Bootstrap instance
            $container = \baseKRIZAN\Bootstrap\Bootstrap::getInstance()->getContainer();
            
            // Ako je aktiviran modul PM ili neki od ovisnih modula
            if ($moduleId === 'pm' || in_array($moduleId, ['tasks', 'sales'])) {
                $logger = $container->get('logger');
                $logger->modules('PM module detected activation of related module', [
                    'module' => $moduleId
                ]);
                
                // Ovdje možete implementirati dodatnu logiku za povezivanje s drugim modulima
            }
            
            return null;
        }
    ],
    
    // Vlastiti događaji specifični za modul PM
    'pm.task_created' => [
        function($event) {
            $taskId = $event->get('task_id');
            $userId = $event->get('user_id');
            
            // Dohvatimo container iz Bootstrap instance
            $container = \baseKRIZAN\Bootstrap\Bootstrap::getInstance()->getContainer();
            
            $logger = $container->get('logger');
            $logger->modules('New project task created', [
                'task_id' => $taskId,
                'user_id' => $userId
            ]);
            
            // Provjeri postoji li notifikacijski servis
            if ($container->has('notificationService')) {
                $notificationService = $container->get('notificationService');
                
                // Pošalji notifikaciju o novom zadatku
                $notificationService->sendNotification(
                    $userId,
                    'New project task created',
                    'You have created a new project task with ID ' . $taskId,
                    ['task_id' => $taskId]
                );
                
                $logger->modules('Project task creation notification sent', [
                    'task_id' => $taskId,
                    'user_id' => $userId
                ]);
            }
            
            return null;
        }
    ],
    
    'pm.project_updated' => [
        function($event) {
            $projectId = $event->get('project_id');
            $userId = $event->get('user_id');
            
            // Dohvatimo container iz Bootstrap instance
            $container = \baseKRIZAN\Bootstrap\Bootstrap::getInstance()->getContainer();
            
            $logger = $container->get('logger');
            $logger->modules('Project updated', [
                'project_id' => $projectId,
                'user_id' => $userId
            ]);
            
            // Provjeri postoji li notifikacijski servis
            if ($container->has('notificationService')) {
                $notificationService = $container->get('notificationService');
                
                // Dohvati korisnike koji bi trebali biti obaviješteni
                if ($container->has('korisniciTable')) {
                    $korisniciTable = $container->get('korisniciTable');
                    $adminKorisnici = $korisniciTable->findByPermission(\Models\Entity\Korisnik::KORISNIK_ADMIN);
                    
                    foreach ($adminKorisnici as $korisnik) {
                        if ($korisnik->id != $userId) {
                            // Pošalji notifikaciju drugim administratorima
                            $notificationService->sendNotification(
                                $korisnik->id,
                                'Project Updated',
                                'Project with ID ' . $projectId . ' has been updated',
                                ['project_id' => $projectId]
                            );
                        }
                    }
                }
                
                $logger->modules('Project update notifications sent', [
                    'project_id' => $projectId,
                    'user_id' => $userId
                ]);
            }
            
            // Provjeri postoji li tasks modul i ažuriraj povezane zadatke
            if ($container->has('activatedModules')) {
                $activatedModules = $container->get('activatedModules');
                
                if (in_array('tasks', $activatedModules) && $container->has('eventDispatcher')) {
                    $eventDispatcher = $container->get('eventDispatcher');
                    
                    // Obavijesti tasks modul da je projekt ažuriran
                    $eventDispatcher->dispatch('tasks.project_updated', [
                        'project_id' => $projectId,
                        'timestamp' => time()
                    ]);
                    
                    $logger->modules('Notified tasks module about project update', [
                        'project_id' => $projectId
                    ]);
                }
            }
            
            return null;
        }
    ],
    
    // Slušanje događaja iz drugih modula
    'tasks.created' => [
        function($event) {
            $taskId = $event->get('task_id');
            
            // Dohvatimo container iz Bootstrap instance
            $container = \baseKRIZAN\Bootstrap\Bootstrap::getInstance()->getContainer();
            
            // Ažuriraj projekte povezane sa zadatkom
            if ($container->has('pmTable')) {
                $pmTable = $container->get('pmTable');
                
                // Implementacija logike za povezivanje zadataka sa projektima
                // ...
                
                $container->get('logger')->modules('Processed task for project management', [
                    'task_id' => $taskId
                ]);
            }
            
            return null;
        }
    ]
];