<?php

namespace sales\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Models\DatabaseTable;
use Controllers\BaseController;
use sales\Models\DatabaseTableAgentBonus;     // Novi AgentBonus model
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Error\Logger;

class AgentBonus extends BaseController
{
    private DatabaseTable $korisniciTable;
    private DatabaseTableAgentBonus $agentBonusTable;    // Specializirani model

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableAgentBonus $agentBonusTable,        // Novi model umjesto salesTable
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->agentBonusTable = $agentBonusTable;        // Specializirani model
    }

    /**
     * Agent bonus dashboard page
     * 
     * @param Request $request The request object
     * @return Response The response object
     */
    public function index(Request $request): Response 
    {
        try {
            // Koristi agentBonusTable za sve AgentBonus analytics
            $statistics = [
                'monthlyActivesByAgent' => $this->agentBonusTable->getMonthlyActivesByAgent(),
                'activeSalesByAgent' => $this->agentBonusTable->getActiveSalesByAgent(),
                'totalActivesByMonth' => $this->agentBonusTable->getTotalActivesByMonth(),
                'agentServiceBreakdown' => $this->agentBonusTable->getAgentServiceBreakdown(),
                'currentMonthActives' => $this->agentBonusTable->getCurrentMonthActiveCount(),
                'lastMonthActives' => $this->agentBonusTable->getLastMonthActiveCount(),
                'totalActiveAgents' => $this->agentBonusTable->getTotalActiveAgents(),
                'topPerformingAgents' => $this->agentBonusTable->getTopPerformingAgents(),
                'monthlyAgentComparison' => $this->agentBonusTable->getMonthlyAgentComparison(),
                'agentGrowthTrends' => $this->agentBonusTable->getAgentGrowthTrends()
            ];
            
            return $this->response()->render(
                'sales/resources/views/agentbonus/index.html.php',
                [
                    'title' => 'Agent Bonus Dashboard',
                    'statistics' => $statistics,
                    'user' => $this->authentication->getUser()
                ],
                'Agent Bonus Dashboard'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading agent bonus dashboard', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load agent bonus dashboard');
        }
    }

    /**
     * Get agent bonus statistics via AJAX
     */
    public function getStats(Request $request): Response
    {
        try {
            $statistics = [
                'monthlyActivesByAgent' => $this->agentBonusTable->getMonthlyActivesByAgent(),
                'activeSalesByAgent' => $this->agentBonusTable->getActiveSalesByAgent(),
                'totalActivesByMonth' => $this->agentBonusTable->getTotalActivesByMonth(),
                'agentServiceBreakdown' => $this->agentBonusTable->getAgentServiceBreakdown(),
                'currentMonthActives' => $this->agentBonusTable->getCurrentMonthActiveCount(),
                'lastMonthActives' => $this->agentBonusTable->getLastMonthActiveCount(),
                'totalActiveAgents' => $this->agentBonusTable->getTotalActiveAgents(),
                'topPerformingAgents' => $this->agentBonusTable->getTopPerformingAgents(),
                'monthlyAgentComparison' => $this->agentBonusTable->getMonthlyAgentComparison(),
                'agentGrowthTrends' => $this->agentBonusTable->getAgentGrowthTrends()
            ];
            
            return Response::json($statistics);
        } catch (\Throwable $e) {
            $this->logger->error('Error fetching agent bonus stats', [
                'message' => $e->getMessage()
            ]);
            return Response::json(['error' => 'Failed to fetch agent bonus stats'], 500);
        }
    }

    /**
     * Export agent bonus report to Excel
     * 
     * @param Request $request The request object
     * @return Response The response object
     */
    public function export(Request $request): Response 
    {
        try {
            $currentUser = $this->authentication->getUser();
            if (!$currentUser || !$currentUser->hasPermission(\Models\Entity\Korisnik::KORISNIK_sales_supervisor)) {
                throw new \RuntimeException('Unauthorized to export agent bonus data');
            }
            
            // FIXED: koristi agentBonusTable umjesto salesTable
            $monthlyData = $this->agentBonusTable->getMonthlyActivesByAgent();
            $agentData = $this->agentBonusTable->getActiveSalesByAgent();
            
            // Prepare export data
            $exportData = [];
            foreach ($monthlyData as $record) {
                $exportData[] = [
                    'mjesec' => $record['month'],
                    'godina' => $record['year'],
                    'djelatnik' => $record['agent_name'],
                    'broj_aktivnih' => $record['active_count']
                ];
            }
            
            $columns = ['mjesec', 'godina', 'djelatnik', 'broj_aktivnih'];
            
            require_once APP_ROOT . '/app/Helpers/excelexport.php';
            $savefileas = sprintf(
                "agent_bonus_report_%s",
                date('Y-m-d')
            );
            
            $export = excelexport('agent_bonus', $columns, $savefileas, $exportData);
            
            return Response::download(
                $export['content'],
                $export['filename'],
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Agent bonus export failed', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id', []),
                'ip' => $request->getIp()
            ]);
            return $this->response()->renderError('Failed to export agent bonus data');
        }
    }

    /**
     * Generate detailed report for specific agent and time period
     * 
     * @param Request $request The request object
     * @return Response The response object
     */
    public function detailReport(Request $request): Response 
    {
        try {
            $agentName = $request->getParam('agent', '');
            $month = $request->getParam('month', date('m'));
            $year = $request->getParam('year', date('Y'));
            
            if (empty($agentName)) {
                return $this->response()->renderError('Agent name is required for detailed report');
            }
            
            // FIXED: koristi agentBonusTable umjesto salesTable
            $detailData = $this->agentBonusTable->getDetailedActiveSalesForAgent($agentName, $month, $year);
            $summary = $this->agentBonusTable->getAgentSummaryForPeriod($agentName, $month, $year);
            $serviceBreakdown = $this->agentBonusTable->getAgentServiceBreakdownForPeriod($agentName, $month, $year);
            
            return $this->response()->render(
                'sales/resources/views/agentbonus/detail_report.html.php',
                [
                    'title' => 'Detaljni izvještaj - ' . $agentName,
                    'agentName' => $agentName,
                    'month' => $month,
                    'year' => $year,
                    'detailData' => $detailData,
                    'summary' => $summary,
                    'serviceBreakdown' => $serviceBreakdown,
                    'user' => $this->authentication->getUser()
                ],
                'Detaljni izvještaj agent bonusa'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Failed to generate detailed agent bonus report', [
                'message' => $e->getMessage(),
                'agent' => $agentName ?? 'unknown',
                'month' => $month ?? 'unknown',
                'year' => $year ?? 'unknown'
            ]);
            return $this->response()->renderError('Failed to generate detailed report');
        }
    }

    /**
     * Print report view
     * 
     * @param Request $request The request object
     * @return Response The response object
     */
    public function print(Request $request): Response 
    {
        try {
            $currentUser = $this->authentication->getUser();
            
            $month = $request->getParam('month', date('m'));
            $year = $request->getParam('year', date('Y'));
            
            // FIXED: koristi agentBonusTable umjesto salesTable
            $monthlyData = $this->agentBonusTable->getMonthlyActivesByAgent($month, $year);
            $agentData = $this->agentBonusTable->getActiveSalesByAgent();
            
            $title = "Agent Bonus izvještaj za {$month}/{$year}";
            
            return $this->response()->render(
                'sales/resources/views/agentbonus/print.html.php',
                [
                    'monthlyData' => $monthlyData,
                    'agentData' => $agentData,
                    'title' => $title,
                    'month' => $month,
                    'year' => $year,
                    'user' => $currentUser,
                    'print_date' => date('d.m.Y H:i:s')
                ],
                $title
            );
        } catch (\Throwable $e) {
            $this->logger->error('Agent bonus print report failed', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id', []),
                'ip' => $request->getIp()
            ]);
            return $this->response()->renderError('Failed to generate print report');
        }
    }
}