<?php

namespace sales\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Models\DatabaseTable;
use Controllers\BaseController;
use sales\Models\DatabaseTableAgentBonusBodovanje;
use sales\Models\DatabaseTableFeedback;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Error\Logger;

class AgentBonusBodovanje extends BaseController
{
    private DatabaseTable $korisniciTable;
    private DatabaseTableAgentBonusBodovanje $agentBonusBodovanjeTable;
    private DatabaseTableFeedback $feedbackTable;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableAgentBonusBodovanje $agentBonusBodovanjeTable,
        DatabaseTableFeedback $feedbackTable,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->agentBonusBodovanjeTable = $agentBonusBodovanjeTable;
        $this->feedbackTable = $feedbackTable;
    }

    /**
     * Agent bonus bodovanje dashboard page
     */
    public function index(Request $request): Response 
    {
        try {
            // Get filter parameters
            $month = $request->getQuery('month') ?? 
                    $request->getParam('month') ?? 
                    $request->input('month') ??
                    $_GET['month'] ?? 
                    date('m');
                    
            $year = $request->getQuery('year') ?? 
                   $request->getParam('year') ?? 
                   $request->input('year') ??
                   $_GET['year'] ?? 
                   date('Y');
            
            // Debug log
            if ($this->logger) {
                $this->logger->info('AgentBonusBodovanje filter request', [
                    'month' => $month,
                    'year' => $year,
                    'query_params' => $request->getQuery(),
                    'get_params' => $_GET,
                    'url' => $_SERVER['REQUEST_URI'] ?? 'unknown'
                ]);
            }
            
            // Get all statistics
            $statistics = [
                // Main statistics by agent groups
                'mnpAgents' => $this->agentBonusBodovanjeTable->getMNPAgentsStatistics($month, $year),
                'optikaAgents' => $this->agentBonusBodovanjeTable->getOptikaAgentsStatistics($month, $year),
                'p2pAgents' => $this->agentBonusBodovanjeTable->getP2PAgentsStatistics($month, $year),
                
                // Summary statistics
                'totalPoints' => $this->agentBonusBodovanjeTable->getTotalPointsAllGroups($month, $year),
                'totalBonus' => $this->agentBonusBodovanjeTable->getTotalBonusAllGroups($month, $year),
                'activeAgents' => $this->agentBonusBodovanjeTable->getActiveAgentsCount($month, $year),
                
                // Top performers
                'topPerformers' => $this->agentBonusBodovanjeTable->getTopPerformers($month, $year, 10),
                
                // Monthly trend for charts
                'monthlyTrend' => $this->agentBonusBodovanjeTable->getMonthlyPointsTrend(6),
                
                // Group distribution for charts
                'groupDistribution' => $this->agentBonusBodovanjeTable->getGroupPointsDistribution($month, $year),
                
                // Rankings data
                'rankingsData' => $this->agentBonusBodovanjeTable->getAllAgentsWithRankings($month, $year),
                
                // Current filter values
                'currentMonth' => $month,
                'currentYear' => $year,
                'selectedPeriod' => "$year-$month"
            ];
            
            return $this->response()->render(
                'sales/resources/views/agentbonusbodovanje/index.html.php',
                [
                    'title' => 'Agent Bonus Bodovanje Dashboard',
                    'statistics' => $statistics,
                    'user' => $this->authentication->getUser()
                ],
                'Agent Bonus Bodovanje Dashboard'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading agent bonus bodovanje dashboard', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load agent bonus bodovanje dashboard');
        }
    }

    /**
     * Get agent bonus bodovanje statistics via AJAX
     */
    public function getStats(Request $request): Response
    {
        try {
            $month = $request->getPost('month') ?? 
                    $request->getQuery('month') ?? 
                    $request->getParam('month') ?? 
                    $request->input('month') ??
                    $_GET['month'] ?? 
                    date('m');
                    
            $year = $request->getPost('year') ?? 
                   $request->getQuery('year') ?? 
                   $request->getParam('year') ?? 
                   $request->input('year') ??
                   $_GET['year'] ?? 
                   date('Y');
            
            $statistics = [
                'mnpAgents' => $this->agentBonusBodovanjeTable->getMNPAgentsStatistics($month, $year),
                'optikaAgents' => $this->agentBonusBodovanjeTable->getOptikaAgentsStatistics($month, $year),
                'p2pAgents' => $this->agentBonusBodovanjeTable->getP2PAgentsStatistics($month, $year),
                'totalPoints' => $this->agentBonusBodovanjeTable->getTotalPointsAllGroups($month, $year),
                'totalBonus' => $this->agentBonusBodovanjeTable->getTotalBonusAllGroups($month, $year),
                'activeAgents' => $this->agentBonusBodovanjeTable->getActiveAgentsCount($month, $year),
                'topPerformers' => $this->agentBonusBodovanjeTable->getTopPerformers($month, $year, 10),
                'monthlyTrend' => $this->agentBonusBodovanjeTable->getMonthlyPointsTrend(6),
                'groupDistribution' => $this->agentBonusBodovanjeTable->getGroupPointsDistribution($month, $year),
                'selectedPeriod' => "$year-$month"
            ];
            
            return Response::json($statistics);
        } catch (\Throwable $e) {
            $this->logger->error('Error fetching agent bonus bodovanje stats', [
                'message' => $e->getMessage()
            ]);
            return Response::json(['error' => 'Failed to fetch agent bonus bodovanje stats'], 500);
        }
    }

    /**
     * Export agent bonus bodovanje report to Excel
     */
    public function export(Request $request): Response 
    {
        try {
            $currentUser = $this->authentication->getUser();
            if (!$currentUser || !$currentUser->hasPermission(\Models\Entity\Korisnik::KORISNIK_sales_supervisor)) {
                throw new \RuntimeException('Unauthorized to export agent bonus bodovanje data');
            }
            
            $month = $request->getParam('month', date('m'));
            $year = $request->getParam('year', date('Y'));
            
            $rankingsData = $this->agentBonusBodovanjeTable->getAllAgentsWithRankings($month, $year);
            
            // Prepare export data
            $exportData = [];
            
            foreach ($rankingsData as $agentData) {
                $exportData[] = [
                    'djelatnik' => $agentData['agent_name'],
                    'grupa' => $agentData['agent_group'],
                    'ukupni_bodovi' => $agentData['total_points'],
                    'otkljucani_razred' => $agentData['current_rank'],
                    'bonus_eur' => $agentData['bonus_amount'],
                    'prodanih_usluga' => $agentData['services_sold'],
                    'mjesec' => $month,
                    'godina' => $year
                ];
            }
            
            $columns = ['djelatnik', 'grupa', 'ukupni_bodovi', 'otkljucani_razred', 'bonus_eur', 'prodanih_usluga', 'mjesec', 'godina'];
            
            require_once APP_ROOT . '/app/Helpers/excelexport.php';
            $savefileas = sprintf(
                "agent_bonus_bodovanje_report_%s_%s",
                $year,
                str_pad($month, 2, '0', STR_PAD_LEFT)
            );
            
            $export = excelexport('agent_bonus_bodovanje', $columns, $savefileas, $exportData);
            
            return Response::download(
                $export['content'],
                $export['filename'],
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Agent bonus bodovanje export failed', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id', []),
                'ip' => $request->getIp()
            ]);
            return $this->response()->renderError('Failed to export agent bonus bodovanje data');
        }
    }

    /**
     * Generate detailed report for specific agent group and time period
     */
    public function detailReport(Request $request): Response 
    {
        try {
            $agentGroup = $request->getQuery('group') ?? 
                         $request->getParam('group') ?? 
                         $request->input('group') ??
                         $_GET['group'] ?? 
                         '';
                        
            $month = $request->getQuery('month') ?? 
                    $request->getParam('month') ?? 
                    $request->input('month') ??
                    $_GET['month'] ?? 
                    date('m');
                    
            $year = $request->getQuery('year') ?? 
                   $request->getParam('year') ?? 
                   $request->input('year') ??
                   $_GET['year'] ?? 
                   date('Y');
            
            if (empty($agentGroup)) {
                return $this->response()->renderError('Agent group is required for detailed report');
            }
            
            $detailData = [];
            switch($agentGroup) {
                case 'mnp':
                    $detailData = $this->agentBonusBodovanjeTable->getMNPAgentsStatistics($month, $year);
                    break;
                case 'optika':
                    $detailData = $this->agentBonusBodovanjeTable->getOptikaAgentsStatistics($month, $year);
                    break;
                case 'p2p':
                    $detailData = $this->agentBonusBodovanjeTable->getP2PAgentsStatistics($month, $year);
                    break;
                default:
                    return $this->response()->renderError('Invalid agent group specified');
            }
            
            return $this->response()->render(
                'sales/resources/views/agentbonusbodovanje/detail_report.html.php',
                [
                    'title' => 'Detaljni izvještaj - ' . strtoupper($agentGroup),
                    'agentGroup' => $agentGroup,
                    'month' => $month,
                    'year' => $year,
                    'detailData' => $detailData,
                    'user' => $this->authentication->getUser()
                ],
                'Detaljni izvještaj agent bonus bodovanja'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Failed to generate detailed agent bonus bodovanje report', [
                'message' => $e->getMessage(),
                'group' => $agentGroup ?? 'unknown',
                'month' => $month ?? 'unknown',
                'year' => $year ?? 'unknown'
            ]);
            return $this->response()->renderError('Failed to generate detailed report');
        }
    }

    /**
     * Print report view
     */
    public function print(Request $request): Response 
    {
        try {
            $currentUser = $this->authentication->getUser();
            
            $month = $request->getQuery('month') ?? 
                    $request->getParam('month') ?? 
                    $request->input('month') ??
                    $_GET['month'] ?? 
                    date('m');
                    
            $year = $request->getQuery('year') ?? 
                   $request->getParam('year') ?? 
                   $request->input('year') ??
                   $_GET['year'] ?? 
                   date('Y');
            
            $rankingsData = $this->agentBonusBodovanjeTable->getAllAgentsWithRankings($month, $year);
            $totalPoints = $this->agentBonusBodovanjeTable->getTotalPointsAllGroups($month, $year);
            $totalBonus = $this->agentBonusBodovanjeTable->getTotalBonusAllGroups($month, $year);
            
            $title = "Agent Bonus Bodovanje izvještaj za {$month}/{$year}";
            
            return $this->response()->render(
                'sales/resources/views/agentbonusbodovanje/print.html.php',
                [
                    'rankingsData' => $rankingsData,
                    'totalPoints' => $totalPoints,
                    'totalBonus' => $totalBonus,
                    'title' => $title,
                    'month' => $month,
                    'year' => $year,
                    'user' => $currentUser,
                    'print_date' => date('d.m.Y H:i:s')
                ],
                $title
            );
        } catch (\Throwable $e) {
            $this->logger->error('Agent bonus bodovanje print report failed', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id', []),
                'ip' => $request->getIp()
            ]);
            return $this->response()->renderError('Failed to generate print report');
        }
    }

    /**
     * My bonus page for individual agents - ENHANCED WITH FEEDBACK
     */
    public function myBonus(Request $request): Response 
    {
        try {
            $currentUser = $this->authentication->getUser();
            if (!$currentUser) {
                return $this->response()->redirect(route('login'));
            }
            
            // Get agent name from user data
            $agentName = $currentUser->user_firstandlastname ?? '';
        
            if (empty($agentName)) {
                return $this->response()->renderError('Ime agenta nije pronađeno u korisničkim podacima');
            }
            
            // Get filter parameters
            $month = $request->getQuery('month') ?? 
                    $request->getParam('month') ?? 
                    $request->input('month') ??
                    $_GET['month'] ?? 
                    date('m');
                    
            $year = $request->getQuery('year') ?? 
                   $request->getParam('year') ?? 
                   $request->input('year') ??
                   $_GET['year'] ?? 
                   date('Y');
            
            // Get agent's organizational unit to determine group
            $agentGroup = $this->agentBonusBodovanjeTable->getAgentGroup($agentName);
            
            if (!$agentGroup) {
                return $this->response()->renderError('Agent nije pronađen u sistemu prodaje');
            }
            
            // Get agent's statistics
            $agentStats = $this->agentBonusBodovanjeTable->getAgentDetailedStats($agentName, $month, $year, $agentGroup);
            
            // Get agent's historical performance (last 6 months)
            $historicalPerformance = $this->agentBonusBodovanjeTable->getAgentHistoricalPerformance($agentName, 6);
            
            // Get agent's ranking among peers in same group
            $groupRanking = $this->agentBonusBodovanjeTable->getAgentGroupRanking($agentName, $month, $year, $agentGroup);
            
            // Get next rank requirements
            $nextRankRequirements = $this->agentBonusBodovanjeTable->getNextRankRequirements($agentStats['total_points'], $agentGroup);
            
            // Get ranking system for this group
            $rankingSystem = $this->agentBonusBodovanjeTable->getRankingSystem($agentGroup);
            
            // =============================================
            // FEEDBACK INTEGRATION
            // =============================================
            
            // Get agent's feedback data
            $agentFeedbacks = $this->feedbackTable->getAllFeedbacks($agentName);
            $latestFeedback = $this->feedbackTable->getLatestFeedback($agentName);
            $feedbackStats = $this->feedbackTable->getFeedbackStats($agentName);
            
            $statistics = [
                'agentStats' => $agentStats,
                'historicalPerformance' => $historicalPerformance,
                'groupRanking' => $groupRanking,
                'nextRankRequirements' => $nextRankRequirements,
                'rankingSystem' => $rankingSystem,
                'currentMonth' => $month,
                'currentYear' => $year,
                'selectedPeriod' => "$year-$month",
                'agentName' => $agentName,
                'agentGroup' => $agentGroup,
                
                // FEEDBACK DATA
                'agentFeedbacks' => $agentFeedbacks,
                'latestFeedback' => $latestFeedback,
                'feedbackStats' => $feedbackStats
            ];
            
            return $this->response()->render(
                'sales/resources/views/agentbonusbodovanje/my_bonus.html.php',
                [
                    'title' => 'Moj Bonus Dashboard',
                    'statistics' => $statistics,
                    'user' => $currentUser
                ],
                'Moj Bonus Dashboard'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading my bonus dashboard', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id', [])
            ]);
            return $this->response()->renderError('Greška pri učitavanju bonus dashboard-a');
        }
    }

    /**
     * Get my feedback data via AJAX
     */
    public function getMyFeedback(Request $request): Response
    {
        try {
            $currentUser = $this->authentication->getUser();
            if (!$currentUser) {
                return Response::json(['error' => 'Unauthorized'], 401);
            }
            
            $agentName = $currentUser->user_firstandlastname ?? '';
            
            if (empty($agentName)) {
                return Response::json(['error' => 'Agent name not found'], 400);
            }
            
            $allFeedbacks = $this->feedbackTable->getAllFeedbacks($agentName);
            $latestFeedback = $this->feedbackTable->getLatestFeedback($agentName);
            $feedbackStats = $this->feedbackTable->getFeedbackStats($agentName);
            
            return Response::json([
                'feedbacks' => $allFeedbacks,
                'latest_feedback' => $latestFeedback,
                'stats' => $feedbackStats
            ]);
        } catch (\Throwable $e) {
            $this->logger->error('Error getting my feedback', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id', [])
            ]);
            return Response::json(['error' => 'Failed to get feedback'], 500);
        }
    }
}