<?php

namespace sales\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Controllers\BaseController;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use sales\Models\DatabaseTableSales;
use sales\Models\DatabaseTableBonus;
use baseKRIZAN\Validation\Validator;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;

class Bonus extends BaseController 
{
    private DatabaseTable $korisniciTable;
    private DatabaseTableSales $salesTable;
    private DatabaseTableBonus $bonusPrijenosTable;
    private DatabaseTableBonus $bonusP2pTable;
    private DatabaseTableBonus $bonusTomatoTable;
    private Validator $validator;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableSales $salesTable,
        DatabaseTableBonus $bonusPrijenosTable,
        DatabaseTableBonus $bonusP2pTable,
        DatabaseTableBonus $bonusTomatoTable,
        Validator $validator,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->salesTable = $salesTable;
        $this->bonusPrijenosTable = $bonusPrijenosTable;
        $this->bonusP2pTable = $bonusP2pTable;
        $this->bonusTomatoTable = $bonusTomatoTable;
        $this->validator = $validator;
    }

    /**
     * List all bonus schemes by type
     */
    public function list(Request $request): Response 
    {
        try {
            // Pokušaj dohvatiti type iz različitih izvora
            $type = $request->getQuery('type') ?? 
                   $request->getParam('type') ?? 
                   $request->input('type') ??
                   $_GET['type'] ?? 
                   'prijenos';
            
            // Validate type - AŽURIRANO za novu strukturu
            $validTypes = ['prijenos', 'p2p', 'tomato'];
            if (!in_array($type, $validTypes)) {
                $type = 'prijenos';
            }
            
            $bonuses = $this->getBonusesByType($type);
            $this->sessionManager->set('valid_bonus_ids_' . $type, array_map(fn($bonus) => $bonus->id, $bonuses));

            return $this->response()->render(
                'sales/resources/views/bonus/list.html.php',
                [
                    'user' => $this->authentication->getUser(),
                    'bonuses' => $bonuses,
                    'current_type' => $type,
                    'type_title' => $this->getTypeTitle($type),
                    'types' => [
                        'prijenos' => 'Prijenos',
                        'p2p' => 'P2P',
                        'tomato' => 'Tomato'
                    ]
                ],
                'Bonus sheme - ' . $this->getTypeTitle($type)
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading bonus list', [
                'message' => $e->getMessage(),
                'type' => $type ?? null
            ]);
            return $this->response()->renderError('Failed to load bonus list');
        }
    }

    /**
     * Edit bonus form
     */
    public function edit(Request $request): Response 
    {
        try {
            // Dohvaćamo type iz query parametara ili default
            $type = $request->getQuery('type') ?? 
                   $request->getParam('type') ?? 
                   $_GET['type'] ?? 
                   'prijenos';
            
            // Dohvaćamo ID iz različitih izvora
            $id = $request->getQuery('id') ?? 
                  $request->getParam('id') ?? 
                  $_GET['id'] ?? 
                  null;
            
            // Convert to integer if present
            if ($id !== null) {
                $id = (int)$id;
            }
            
            $bonus = null;
            $editToken = null;

            // Debug log
            if ($this->logger) {
                $this->logger->info('Bonus edit request', [
                    'type' => $type,
                    'id' => $id,
                    'query_params' => $request->getQuery(),
                    'get_params' => $_GET,
                    'url' => $_SERVER['REQUEST_URI'] ?? 'unknown'
                ]);
            }

            if ($id && $id > 0) {
                $bonus = $this->getBonusByTypeAndId($type, $id);
                if ($bonus) {
                    $editToken = $this->createEditToken($bonus->id, 'bonus_' . $type);
                    
                    if ($this->logger) {
                        $this->logger->info('Bonus found for editing', [
                            'bonus_id' => $bonus->id,
                            'bonus_name' => $bonus->vrsta_tarifa ?? 'unknown',
                            'type' => $type
                        ]);
                    }
                } else {
                    if ($this->logger) {
                        $this->logger->warning('Bonus not found', [
                            'id' => $id,
                            'type' => $type
                        ]);
                    }
                }
            }

            return $this->response()->render(
                'sales/resources/views/bonus/edit.html.php',
                [
                    'bonus' => $bonus,
                    'edit_token' => $editToken,
                    'type' => $type,
                    'type_title' => $this->getTypeTitle($type)
                ],
                ($id ? 'Uredi' : 'Dodaj') . ' bonus - ' . $this->getTypeTitle($type)
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading bonus edit form', [
                'message' => $e->getMessage(),
                'type' => $type ?? null,
                'id' => $id ?? null
            ]);
            return $this->response()->renderError('Failed to load bonus edit form');
        }
    }

    /**
     * Save bonus
     */
    public function saveEdit(Request $request): Response 
    {
        try {
            // 1. DOHVAĆANJE OSNOVNIH PODATAKA
            $type = $request->getPost('type') ?? $request->getParam('type');
            $editToken = $request->getPost('edit_token');
            $bonus = $request->getPost('bonus');
            $bonusId = null;
            $validToken = null;
            
            // Validate type - AŽURIRANO
            $validTypes = ['prijenos', 'p2p', 'tomato'];
            if (!in_array($type, $validTypes)) {
                throw new \InvalidArgumentException('Invalid bonus type');
            }
            
            // 2. VALIDACIJA TOKENA (AKO POSTOJI) - ovo NE troši token
            if ($editToken) {
                $tokenResult = $this->validateEditTokenForValidation($request, 'bonus_' . $type);
                if (isset($tokenResult['error'])) {
                    return $this->renderBonusForm($bonus, $editToken, $type, $tokenResult);
                }
                // Token je valjan
                $bonusId = $tokenResult['id'];
                $validToken = $tokenResult['token'];
                $bonus['id'] = $bonusId;
            }
            
            // 3. VALIDACIJA PODATAKA FORME
            $rules = $this->getValidationRules($type);
            $validation = $request->validate($rules);
            
            if ($validation->fails()) {
                // Stvori novi token za ponovni pokušaj
                $newEditToken = null;
                if ($bonusId && $validToken) {
                    $newEditToken = $this->refreshEditToken($validToken, 'bonus_' . $type);
                }
                
                return $this->renderBonusForm($bonus, $newEditToken, $type, $validation->errors());
            }

            // 4. USPJEŠNA VALIDACIJA - SADA troši token (jednokratno)
            if ($validToken) {
                $this->consumeEditToken($validToken);
            }

            // 5. SPREMANJE PODATAKA
            $this->saveBonusByType($type, $bonus);
            
            // 6. PREUSMJERAVANJE
            return Response::redirect('salebonus/list?type=' . $type);
        } catch (\Throwable $e) {
            $this->logger->error('Error saving bonus', [
                'message' => $e->getMessage(),
                'type' => $type ?? null,
                'bonus' => $bonus ?? null
            ]);
            return $this->response()->renderError('Failed to save bonus');
        }
    }

    /**
     * Delete bonus
     */
    public function delete(Request $request): Response 
    {
        try {
            $type = $request->getPost('type');
            $bonusId = $request->getPost('id');
            
            // Security check
            $validBonusIds = $this->sessionManager->get('valid_bonus_ids_' . $type, []);
            if (!in_array($bonusId, $validBonusIds ?? [])) {
                $this->logger->security('Unauthorized bonus deletion attempt', [
                    'id' => $bonusId,
                    'type' => $type,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Unauthorized action');
            }

            // Delete from appropriate table
            $this->deleteBonusByType($type, $bonusId);
            $this->removeFromSession($type, $bonusId);

            return Response::redirect('salebonus/list?type=' . $type);
        } catch (\Throwable $e) {
            $this->logger->error('Error deleting bonus', [
                'message' => $e->getMessage(),
                'type' => $type ?? null,
                'id' => $bonusId ?? null
            ]);
            return $this->response()->renderError('Failed to delete bonus');
        }
    }

    /**
     * Get bonuses by type - AŽURIRANO
     */
    private function getBonusesByType(string $type): array
    {
        switch ($type) {
            case 'prijenos':
                return $this->bonusPrijenosTable->findAllByType($type);
            case 'p2p':
                return $this->bonusP2pTable->findAllByType($type);
            case 'tomato':
                return $this->bonusTomatoTable->findAllByType($type);
            default:
                return [];
        }
    }

    /**
     * Get bonus by type and ID - AŽURIRANO
     */
    private function getBonusByTypeAndId(string $type, int $id): ?object
    {
        switch ($type) {
            case 'prijenos':
                return $this->bonusPrijenosTable->findByIdAndType($type, $id);
            case 'p2p':
                return $this->bonusP2pTable->findByIdAndType($type, $id);
            case 'tomato':
                return $this->bonusTomatoTable->findByIdAndType($type, $id);
            default:
                return null;
        }
    }

    /**
     * Save bonus by type - AŽURIRANO
     */
    private function saveBonusByType(string $type, array $bonus): void
    {
        switch ($type) {
            case 'prijenos':
                $this->bonusPrijenosTable->saveByType($type, $bonus);
                break;
            case 'p2p':
                $this->bonusP2pTable->saveByType($type, $bonus);
                break;
            case 'tomato':
                $this->bonusTomatoTable->saveByType($type, $bonus);
                break;
        }
    }

    /**
     * Delete bonus by type - AŽURIRANO
     */
    private function deleteBonusByType(string $type, int $id): void
    {
        switch ($type) {
            case 'prijenos':
                $this->bonusPrijenosTable->deleteByType($type, $id);
                break;
            case 'p2p':
                $this->bonusP2pTable->deleteByType($type, $id);
                break;
            case 'tomato':
                $this->bonusTomatoTable->deleteByType($type, $id);
                break;
        }
    }

    /**
     * Get validation rules based on type - AŽURIRANO
     */
    private function getValidationRules(string $type): array
    {
        $baseRules = [
            'bonus[vrsta_tarifa]' => ['required', 'string', 'max:100'],
            'bonus[grupa]' => ['required', 'string', 'max:50'],
            'bonus[osnovna_naknada]' => ['required', 'numeric'],
            'bonus[rang_70_80]' => ['required', 'numeric'],
            'bonus[rang_80_90]' => ['required', 'numeric'],
            'bonus[rang_90_120]' => ['required', 'numeric']
        ];

        switch ($type) {
            case 'prijenos':
                return array_merge($baseRules, [
                    'bonus[rang_249]' => ['required', 'numeric'],
                    'bonus[rang_250_449]' => ['required', 'numeric'],
                    'bonus[rang_450_749]' => ['required', 'numeric'],
                    'bonus[rang_750_1049]' => ['required', 'numeric']
                ]);
                
            case 'p2p':
                return array_merge($baseRules, [
                    'bonus[rang_150]' => ['required', 'numeric'],
                    'bonus[rang_150_400]' => ['required', 'numeric'],
                    'bonus[rang_400_650]' => ['required', 'numeric'],
                    'bonus[rang_650_plus]' => ['required', 'numeric']
                ]);
                
            case 'tomato':
                return array_merge($baseRules, [
                    'bonus[rang_299]' => ['nullable', 'numeric'],
                    'bonus[rang_300_549]' => ['nullable', 'numeric'],
                    'bonus[rang_550_779]' => ['nullable', 'numeric'],
                    'bonus[rang_800_1099]' => ['nullable', 'numeric']
                ]);
                
            default:
                return $baseRules;
        }
    }

    /**
     * Get type title - AŽURIRANO
     */
    private function getTypeTitle(string $type): string
    {
        $titles = [
            'prijenos' => 'Prijenos',
            'p2p' => 'P2P',
            'tomato' => 'Tomato'
        ];
        
        return $titles[$type] ?? $type;
    }

    /**
     * Remove bonus ID from session
     */
    private function removeFromSession(string $type, int $bonusId): void 
    {
        $validIds = $this->sessionManager->get('valid_bonus_ids_' . $type, []);
        $validIds = array_filter($validIds, fn($id) => $id != $bonusId);
        $this->sessionManager->set('valid_bonus_ids_' . $type, $validIds);
    }
    
    /**
     * Helper method to render bonus form with errors
     */
    private function renderBonusForm(array $bonus, ?string $editToken, string $type, array $errors): Response
    {
        return $this->response()->render(
            'sales/resources/views/bonus/edit.html.php',
            [
                'bonus' => (object)$bonus,
                'edit_token' => $editToken,
                'type' => $type,
                'type_title' => $this->getTypeTitle($type),
                'errors' => $errors
            ],
            'Edit Bonus - ' . $this->getTypeTitle($type)
        );
    }
}