<?php

namespace sales\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Controllers\BaseController;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use sales\Models\DatabaseTableSales;
use sales\Models\DatabaseTableFeedback;
use baseKRIZAN\Validation\Validator;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;

class Osoba extends BaseController 
{
    private DatabaseTable $korisniciTable;
    private DatabaseTableSales $osobe2Table;
    private DatabaseTableFeedback $feedbackTable;
    private Validator $validator;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableSales $osobe2Table,
        DatabaseTableFeedback $feedbackTable,
        Validator $validator,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->osobe2Table = $osobe2Table;
        $this->feedbackTable = $feedbackTable;
        $this->validator = $validator;
    }

    public function edit(Request $request): Response 
    {
        try {
            $osoba = null;
            $editToken = null;
            $latestFeedback = null;
            $allFeedbacks = [];
            $feedbackStats = [];
            $canManageFeedback = false;

            if ($id = $request->getParam('id')) {
                $osoba = $this->osobe2Table->findById($id);
                if ($osoba) {
                    $editToken = $this->createEditToken($osoba->id, 'osoba');
                    
                    // Get feedback data
                    $latestFeedback = $this->feedbackTable->getLatestFeedback($osoba->name);
                    $allFeedbacks = $this->feedbackTable->getAllFeedbacks($osoba->name);
                    $feedbackStats = $this->feedbackTable->getFeedbackStats($osoba->name);
                    
                    // Check permissions
                    $currentUser = $this->authentication->getUser();
                    $canManageFeedback = $this->feedbackTable->canManageFeedback($currentUser);
                }
            }

            return $this->response()->render(
                'sales/resources/views/osoba/edit.html.php',
                [
                    'osoba' => $osoba,
                    'edit_token' => $editToken,
                    'latest_feedback' => $latestFeedback,
                    'all_feedbacks' => $allFeedbacks,
                    'feedback_stats' => $feedbackStats,
                    'can_manage_feedback' => $canManageFeedback,
                    'user' => $this->authentication->getUser()
                ],
                'Edit Osoba'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading person edit form', [
                'message' => $e->getMessage(),
                'id' => $id ?? null
            ]);
            return $this->response()->renderError('Failed to load person edit form');
        }
    }

    public function saveEdit(Request $request): Response 
    {
        try {
            // 1. DOHVAĆANJE OSNOVNIH PODATAKA
            $editToken = $request->getPost('edit_token');
            $osoba = $request->getPost('osoba');
            $osobaId = null;
            $validToken = null;
            
            // 2. VALIDACIJA TOKENA (AKO POSTOJI) - ovo NE troši token
            if ($editToken) {
                $tokenResult = $this->validateEditTokenForValidation($request, 'osoba');
                if (isset($tokenResult['error'])) {
                    return $this->renderOsobaForm(
                        $osoba, 
                        $editToken, 
                        $tokenResult
                    );
                }
                // Token je valjan
                $osobaId = $tokenResult['id'];
                $validToken = $tokenResult['token'];
                $osoba['id'] = $osobaId;
            }
            
            // 3. VALIDACIJA PODATAKA FORME
            $rules = [
                'osoba[name]' => ['required', 'alpha_spaces', ['max', 255]],
                'osoba[radnomjesto]' => ['required', 'plain_text', ['max', 255]],
                'osoba[area]' => ['required', 'plain_text', ['max', 255]],
                'osoba[orgjedinica]' => ['required', 'plain_text', ['max', 255]],
                'osoba[vidljivost]' => ['required', 'numeric']
            ];

            $validation = $request->validate($rules);
            if ($validation->fails()) {
                // Stvori novi token za ponovni pokušaj
                $newEditToken = null;
                if ($osobaId && $validToken) {
                    $newEditToken = $this->refreshEditToken($validToken, 'osoba');
                }
                
                return $this->renderOsobaForm(
                    $osoba, 
                    $newEditToken, 
                    $validation->errors()
                );
            }

            // 4. USPJEŠNA VALIDACIJA - SADA troši token (jednokratno)
            if ($validToken) {
                $this->consumeEditToken($validToken);
            }

            // 5. SPREMANJE PODATAKA
            $this->osobe2Table->save($osoba);
            
            // 6. PREUSMJERAVANJE
            return Response::redirect('saleosoba/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error saving person', [
                'message' => $e->getMessage(),
                'osoba' => $osoba ?? null
            ]);
            return $this->response()->renderError('Failed to save person');
        }
    }

    public function list(): Response 
    {
        try {
            // Get all people with their latest feedback
            $osobe = $this->feedbackTable->getAllPeopleWithLatestFeedback();
            $this->sessionManager->set('valid_osobe_ids', array_map(fn($osoba) => $osoba['id'], $osobe));

            // Check permissions
            $currentUser = $this->authentication->getUser();
            $canManageFeedback = $this->feedbackTable->canManageFeedback($currentUser);

            return $this->response()->render(
                'sales/resources/views/osoba/list.html.php',
                [
                    'user' => $currentUser,
                    'osobe' => $osobe,
                    'can_manage_feedback' => $canManageFeedback
                ],
                'Sve Osobe'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading persons list', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load persons list');
        }
    }

    public function delete(Request $request): Response 
    {
        try {
            $osobaId = $request->getPost('id');
            
            // Get person name for validation
            $osoba = $this->osobe2Table->findById($osobaId);
            if (!$osoba) {
                return $this->response()->renderError('Person not found');
            }

            // Sigurnosna provjera da li je ID na listi valjanih
            $validOsobeIds = $this->sessionManager->get('valid_osobe_ids', []);
            if (!in_array($osobaId, $validOsobeIds ?? [])) {
                $this->logger->security('Unauthorized person deletion attempt', [
                    'id' => $osobaId,
                    'name' => $osoba->name,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Unauthorized action');
            }

            // Check permissions
            $currentUser = $this->authentication->getUser();
            if (!$this->feedbackTable->canManageFeedback($currentUser)) {
                return $this->response()->renderError('Insufficient permissions');
            }

            // Brisanje osobe
            $this->osobe2Table->delete($osobaId);
            $this->removeFromSession($osobaId);

            return Response::redirect('saleosoba/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error deleting person', [
                'message' => $e->getMessage(),
                'id' => $osobaId ?? null
            ]);
            return $this->response()->renderError('Failed to delete person');
        }
    }

    /**
     * Add new feedback via AJAX
     */
    public function addFeedback(Request $request): Response
    {
        try {
            $currentUser = $this->authentication->getUser();
            if (!$this->feedbackTable->canManageFeedback($currentUser)) {
                return Response::json(['error' => 'Insufficient permissions'], 403);
            }

            $personName = $request->getPost('person_name');
            $feedback = $request->getPost('feedback');
            $ocjena = $request->getPost('ocjena');

            // Validation
            if (empty($personName)) {
                return Response::json(['error' => 'Person name is required'], 400);
            }

            if (empty($feedback) && (empty($ocjena) || $ocjena <= 0)) {
                return Response::json(['error' => 'Either feedback or rating must be provided'], 400);
            }

            // Convert ocjena to float
            $ocjenaFloat = !empty($ocjena) ? (float)$ocjena : null;
            
            // Validate ocjena range
            if ($ocjenaFloat !== null && ($ocjenaFloat < 0 || $ocjenaFloat > 5)) {
                return Response::json(['error' => 'Rating must be between 0 and 5'], 400);
            }

            $success = $this->feedbackTable->addFeedback(
                $personName, 
                $feedback, 
                $ocjenaFloat, 
                $currentUser->user_firstandlastname ?? 'Unknown'
            );

            if ($success) {
                // Get updated feedback data
                $latestFeedback = $this->feedbackTable->getLatestFeedback($personName);
                $feedbackStats = $this->feedbackTable->getFeedbackStats($personName);

                return Response::json([
                    'success' => true,
                    'message' => 'Feedback successfully added',
                    'latest_feedback' => $latestFeedback,
                    'feedback_stats' => $feedbackStats
                ]);
            } else {
                return Response::json(['error' => 'Failed to add feedback'], 500);
            }
        } catch (\Throwable $e) {
            $this->logger->error('Error adding feedback', [
                'message' => $e->getMessage(),
                'person_name' => $request->getPost('person_name')
            ]);
            return Response::json(['error' => 'Failed to add feedback'], 500);
        }
    }

    /**
     * Update feedback via AJAX
     */
    public function updateFeedback(Request $request): Response
    {
        try {
            $currentUser = $this->authentication->getUser();
            if (!$this->feedbackTable->canManageFeedback($currentUser)) {
                return Response::json(['error' => 'Insufficient permissions'], 403);
            }

            $feedbackId = $request->getPost('feedback_id');
            $feedback = $request->getPost('feedback');
            $ocjena = $request->getPost('ocjena');

            // Validation
            if (empty($feedbackId)) {
                return Response::json(['error' => 'Feedback ID is required'], 400);
            }

            if (empty($feedback) && (empty($ocjena) || $ocjena <= 0)) {
                return Response::json(['error' => 'Either feedback or rating must be provided'], 400);
            }

            // Convert ocjena to float
            $ocjenaFloat = !empty($ocjena) ? (float)$ocjena : null;
            
            // Validate ocjena range
            if ($ocjenaFloat !== null && ($ocjenaFloat < 0 || $ocjenaFloat > 5)) {
                return Response::json(['error' => 'Rating must be between 0 and 5'], 400);
            }

            $success = $this->feedbackTable->updateFeedback(
                (int)$feedbackId, 
                $feedback, 
                $ocjenaFloat, 
                $currentUser->user_firstandlastname ?? 'Unknown'
            );

            if ($success) {
                // Get updated feedback
                $updatedFeedback = $this->feedbackTable->findById($feedbackId);
                
                return Response::json([
                    'success' => true,
                    'message' => 'Feedback successfully updated',
                    'feedback' => $updatedFeedback
                ]);
            } else {
                return Response::json(['error' => 'Failed to update feedback'], 500);
            }
        } catch (\Throwable $e) {
            $this->logger->error('Error updating feedback', [
                'message' => $e->getMessage(),
                'feedback_id' => $request->getPost('feedback_id')
            ]);
            return Response::json(['error' => 'Failed to update feedback'], 500);
        }
    }

    /**
     * Delete feedback via AJAX
     */
    public function deleteFeedback(Request $request): Response
    {
        try {
            $currentUser = $this->authentication->getUser();
            if (!$this->feedbackTable->canManageFeedback($currentUser)) {
                return Response::json(['error' => 'Insufficient permissions'], 403);
            }

            $feedbackId = $request->getPost('feedback_id');

            if (empty($feedbackId)) {
                return Response::json(['error' => 'Feedback ID is required'], 400);
            }

            $success = $this->feedbackTable->deleteFeedback(
                (int)$feedbackId, 
                $currentUser->user_firstandlastname ?? 'Unknown'
            );

            if ($success) {
                return Response::json([
                    'success' => true,
                    'message' => 'Feedback successfully deleted'
                ]);
            } else {
                return Response::json(['error' => 'Failed to delete feedback'], 500);
            }
        } catch (\Throwable $e) {
            $this->logger->error('Error deleting feedback', [
                'message' => $e->getMessage(),
                'feedback_id' => $request->getPost('feedback_id')
            ]);
            return Response::json(['error' => 'Failed to delete feedback'], 500);
        }
    }

    /**
     * Get all feedbacks for a person via AJAX
     */
    public function getFeedbacks(Request $request): Response
    {
        try {
            $personName = $request->getQuery('person_name') ?? $request->getParam('person_name');

            if (empty($personName)) {
                return Response::json(['error' => 'Person name is required'], 400);
            }

            $allFeedbacks = $this->feedbackTable->getAllFeedbacks($personName);
            $feedbackStats = $this->feedbackTable->getFeedbackStats($personName);
            
            $currentUser = $this->authentication->getUser();
            $canManageFeedback = $this->feedbackTable->canManageFeedback($currentUser);

            return Response::json([
                'feedbacks' => $allFeedbacks,
                'stats' => $feedbackStats,
                'can_manage' => $canManageFeedback
            ]);
        } catch (\Throwable $e) {
            $this->logger->error('Error getting feedbacks', [
                'message' => $e->getMessage(),
                'person_name' => $request->getQuery('person_name')
            ]);
            return Response::json(['error' => 'Failed to get feedbacks'], 500);
        }
    }

    /**
     * Get single feedback by ID via AJAX
     */
    public function getFeedback(Request $request): Response
    {
        try {
            $feedbackId = $request->getParam('id');

            if (empty($feedbackId)) {
                return Response::json(['error' => 'Feedback ID is required'], 400);
            }

            $feedback = $this->feedbackTable->findById($feedbackId);
            
            if (!$feedback) {
                return Response::json(['error' => 'Feedback not found'], 404);
            }

            return Response::json([
                'feedback' => $feedback
            ]);
        } catch (\Throwable $e) {
            $this->logger->error('Error getting single feedback', [
                'message' => $e->getMessage(),
                'feedback_id' => $request->getParam('id')
            ]);
            return Response::json(['error' => 'Failed to get feedback'], 500);
        }
    }

    private function removeFromSession(int $osobaId): void 
    {
        $validIds = $this->sessionManager->get('valid_osobe_ids', []);
        $validIds = array_filter($validIds, fn($id) => $id != $osobaId);
        $this->sessionManager->set('valid_osobe_ids', $validIds);
    }

    /**
     * Helper method to render the osoba form with errors
     */
    private function renderOsobaForm(array $osoba, ?string $editToken, array $errors): Response
    {
        $latestFeedback = null;
        $allFeedbacks = [];
        $feedbackStats = [];
        $canManageFeedback = false;

        if (!empty($osoba['name'])) {
            $latestFeedback = $this->feedbackTable->getLatestFeedback($osoba['name']);
            $allFeedbacks = $this->feedbackTable->getAllFeedbacks($osoba['name']);
            $feedbackStats = $this->feedbackTable->getFeedbackStats($osoba['name']);
            
            $currentUser = $this->authentication->getUser();
            $canManageFeedback = $this->feedbackTable->canManageFeedback($currentUser);
        }

        return $this->response()->render(
            'sales/resources/views/osoba/edit.html.php',
            [
                'osoba' => (object)$osoba,
                'edit_token' => $editToken,
                'errors' => $errors,
                'latest_feedback' => $latestFeedback,
                'all_feedbacks' => $allFeedbacks,
                'feedback_stats' => $feedbackStats,
                'can_manage_feedback' => $canManageFeedback,
                'user' => $this->authentication->getUser()
            ],
            'Edit Osoba'
        );
    }
}