<?php

namespace sales\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Models\DatabaseTable;
use Controllers\BaseController;
use sales\Models\DatabaseTableSales;
use sales\Models\DatabaseTableSalesAdmin;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Validation\Validator;
use baseKRIZAN\Database\DatabaseConnection;
use Models\DatatablesSSP;

class Sales extends BaseController
{
    private const MAX_EXPORT_ROWS = 1000;
    private const YES = 'yes';
    private const NO = 'no';

    private DatabaseTable $korisniciTable;
    private DatabaseTableSales $salesTable;
    private DatabaseTableSalesAdmin $adminTable;
    protected DatabaseTableSales $vrste2Table;
    private DatabaseTableSales $osobe2Table;
    private Validator $validator;
    private DatabaseConnection $dbConnection;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableSales $salesTable,
        DatabaseTableSalesAdmin $adminTable,
        DatabaseTableSales $vrste2Table,
        DatabaseTableSales $osobe2Table,
        Validator $validator,
        DatabaseConnection $dbConnection,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->salesTable = $salesTable;
        $this->adminTable = $adminTable;
        $this->vrste2Table = $vrste2Table;
        $this->osobe2Table = $osobe2Table;
        $this->validator = $validator;
        $this->dbConnection = $dbConnection;
    }

    /**
     * DataTables mapping konfiguracija za prodaju
     */
    private function getDataTablesMapping(): array
    {
        $tableConfig = [
            'table_name' => 'prodaja',
            'primary_key' => 'id',
            'fixed_columns' => 1,
            'route_name' => 'saleslist'
        ];

        // Row styling konfiguracija s prioritetima
        $rowStyling = [
            'enabled' => self::YES,
            'rules' => [
                // PRIORITET 1: Status Storno - CRVENA (najviši prioritet)
                [
                    'condition_column' => 'sale_status',
                    'condition_value' => 'Storno',
                    'condition_operator' => '==',
                    'css_class' => 'row-status-storno',
                    'priority' => 1,
                    'description' => 'Crveno bojanje za status Storno - najviši prioritet'
                ],
                // PRIORITET 2: Status Aktivan - ZELENA (visoki prioritet)
                [
                    'condition_column' => 'sale_status',
                    'condition_value' => 'Aktivno',
                    'condition_operator' => '==',
                    'css_class' => 'row-status-aktivan',
                    'priority' => 2,
                    'description' => 'Zeleno bojanje za status Aktivan - visoki prioritet'
                ],
                // PRIORITET 3: Kontrolni poziv DA - ŽUTA (niži prioritet)
                [
                    'condition_column' => 'sale_kontrolnipoziv',
                    'condition_value' => 'DA',
                    'condition_operator' => '==',
                    'css_class' => 'row-kontrolni-poziv-da',
                    'priority' => 3,
                    'description' => 'Žuto bojanje za kontrolni poziv DA - niži prioritet'
                ]
            ]
        ];

        $columns = [
            // ID i akcije - uvijek fiksno
            [
                'type' => 'number',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'id',
                'sqlcolumnname' => 'id',
                'width' => 75,
                'render' => 'actions'
            ],
            // Korisnik ID
            [
                'type' => 'number',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'snimio',
                'sqlcolumnname' => 'user_id',
                'width' => 50
            ],
            // Datum prodaje
            [
                'type' => 'date',
                'searchbox' => self::YES,
                'dropdown' => self::YES,
                'calculateinfoot' => self::NO,
                'headername' => 'datum',
                'sqlcolumnname' => 'sale_date',
                'width' => 100
            ],
            // Ime i prezime
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Ime i prezime',
                'sqlcolumnname' => 'sale_imeiprezime',
                'width' => 150
            ],
            // OIB
            [
                'type' => 'number',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'OIB',
                'sqlcolumnname' => 'sale_oib',
                'width' => 100
            ],
            // Broj OI
            [
                'type' => 'number',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Broj OI',
                'sqlcolumnname' => 'sale_brojoi',
                'width' => 75
            ],
            // Datum rođenja
            [
                'type' => 'date',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Datum rođenja',
                'sqlcolumnname' => 'sale_datumrodj',
                'width' => 75
            ],
            // Adresa
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Adresa',
                'sqlcolumnname' => 'sale_adresa',
                'width' => 200
            ],
            // Broj za prijenos
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Broj za prijenos',
                'sqlcolumnname' => 'sale_brojzaprijenos',
                'width' => 75
            ],
            // Matični operator
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Matični operator',
                'sqlcolumnname' => 'sale_maticnioperator',
                'width' => 150
            ],
            // Kontakt broj
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Kontakt broj',
                'sqlcolumnname' => 'sale_kontaktbroj',
                'width' => 75
            ],
            // E-mail
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'E-mail',
                'sqlcolumnname' => 'sale_mail',
                'width' => 200
            ],
            // NOVA POLJA - Informacije o prijenosu vlasništva
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Ime i prezime (prijenos vlasništva)',
                'sqlcolumnname' => 'sale_imeiprezime_prijenosvl',
                'width' => 180
            ],
            [
                'type' => 'number',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'OIB (prijenos vlasništva)',
                'sqlcolumnname' => 'sale_oib_prijenosvl',
                'width' => 120
            ],
            [
                'type' => 'text',
                'dropdown' => self::YES,
                'searchbox' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Paketi',
                'sqlcolumnname' => 'sale_paketi',
                'width' => 120
            ],
            // Način primanja računa
            [
                'type' => 'text',
                'dropdown' => self::YES,
                'searchbox' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Način primanja računa',
                'sqlcolumnname' => 'sale_nacinprimanja',
                'width' => 75
            ],
            // Broj za P2P
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Broj za P2P',
                'sqlcolumnname' => 'sale_brojzap2p',
                'width' => 75
            ],
            // Broj za FNP
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Broj za FNP',
                'sqlcolumnname' => 'sale_brojzafnp',
                'width' => 75
            ],
            // Prodana usluga
            [
                'type' => 'text',
                'dropdown' => self::YES,
                'searchbox' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Prodana usluga',
                'sqlcolumnname' => 'sale_prodanausluga',
                'width' => 150
            ],
            // TV usluga
            [
                'type' => 'text',
                'dropdown' => self::YES,
                'searchbox' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'TV usluga',
                'sqlcolumnname' => 'sale_tvusluga',
                'width' => 50
            ],
            // Kućni telefon
            [
                'type' => 'text',
                'dropdown' => self::YES,
                'searchbox' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Kućni telefon',
                'sqlcolumnname' => 'sale_kucnitel',
                'width' => 50
            ],
            // Mobilni uređaj
            [
                'type' => 'text',
                'dropdown' => self::YES,
                'searchbox' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Mobilni uređaj',
                'sqlcolumnname' => 'sale_mobilniuredaj',
                'width' => 100
            ],
            // Uređaj
            [
                'type' => 'text',
                'dropdown' => self::YES,
                'searchbox' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Uređaj',
                'sqlcolumnname' => 'sale_uredajnavesti',
                'width' => 150
            ],
            // Uređaj na rate
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Uređaj na rate',
                'sqlcolumnname' => 'sale_uredajnarate',
                'width' => 100
            ],
            // Broj narudžbe
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Broj narudžbe',
                'sqlcolumnname' => 'sale_brojnarudzbe',
                'width' => 100,
                'inlineediting' => [
                    'enabled' => self::YES,
                    'url' => 'sales/api/inline-edit'
                ]
            ],
            // Promocija
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Promocija',
                'sqlcolumnname' => 'sale_promocija',
                'width' => 150
            ],
            // OIB za mozaik
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'OIB za mozaik',
                'sqlcolumnname' => 'sale_oibzamozaik',
                'width' => 100
            ],
            // Djelatnik
            [
                'type' => 'text',
                'dropdown' => self::YES,
                'searchbox' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Djelatnik',
                'sqlcolumnname' => 'sale_djelatnik',
                'width' => 200
            ],
            // Kontrolor
            [
                'type' => 'text',
                'dropdown' => self::YES,
                'searchbox' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Kontrolor',
                'sqlcolumnname' => 'sale_kontrolor',
                'width' => 100
            ],
            // Status - KLJUČNA KOLONA ZA ROW STYLING
            [
                'type' => 'text',
                'dropdown' => self::YES,
                'searchbox' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Status',
                'sqlcolumnname' => 'sale_status',
                'width' => 100,
                'inlineediting' => [
                    'enabled' => self::YES,
                    'url' => 'sales/api/inline-edit'
                ]
            ],
            // Datum promjene statusa
            [
                'type' => 'date',
                'searchbox' => self::NO,
                'dropdown' => self::YES,
                'calculateinfoot' => self::NO,
                'headername' => 'Datum promjene statusa',
                'sqlcolumnname' => 'sale_statusdate',
                'width' => 100,
                'inlineediting' => [
                    'enabled' => self::YES,
                    'url' => 'sales/api/inline-edit'
                ]
            ],
            // Kontrolni poziv - KLJUČNA KOLONA ZA ROW STYLING
            [
                'type' => 'text',
                'dropdown' => self::YES,
                'searchbox' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Kontrolni poziv',
                'sqlcolumnname' => 'sale_kontrolnipoziv',
                'width' => 100,
                'inlineediting' => [
                    'enabled' => self::YES,
                    'url' => 'sales/api/inline-edit'
                ]
            ],
            // Komentar
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Komentar',
                'sqlcolumnname' => 'sale_comment',
                'width' => 200,
                'inlineediting' => [
                    'enabled' => self::YES,
                    'url' => 'sales/api/inline-edit'
                ]
            ]
        ];

        return array_merge($tableConfig, [
            'columns' => $columns,
            'rowStyling' => $rowStyling
        ]);
    }

    public function edit(Request $request): Response 
    {
        try {
            $korisnik = $this->authentication->getUser();
            $vrste = $this->vrste2Table->findAll();
            $osobe = $this->osobe2Table->find('vidljivost', '1');
            
            $sale = null;
            $editToken = null;
            
            if ($id = $request->getParam('id')) {
                $sale = $this->salesTable->findById($id);
                if ($sale) {
                    $editToken = $this->createEditToken($sale->id, 'sale');
                }
            }

            return $this->response()->render(
                'sales/resources/views/sales/edit.html.php',
                [
                    'sale' => $sale,
                    'edit_token' => $editToken,
                    'user' => $korisnik,
                    'vrste' => $vrste,
                    'osobe' => $osobe
                ],
                'Uredi prodaju'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading sales edit form', [
                'message' => $e->getMessage(),
                'id' => $id ?? null
            ]);
            return $this->response()->renderError('Failed to load sales edit form');
        }
    }

    public function saveEdit(Request $request): Response 
    {
        try {
            // 1. DOHVAĆANJE OSNOVNIH PODATAKA
            $editToken = $request->getPost('edit_token');
            $sale = $request->getPost('sale');
            $saleId = null;
            $validToken = null;
            
            // 2. VALIDACIJA TOKENA (AKO POSTOJI)
            if ($editToken) {
                $tokenResult = $this->validateEditTokenForValidation($request, 'sale');
                if (isset($tokenResult['error'])) {
                    return $this->renderSalesForm(
                        ['sale' => $sale, 'edit_token' => $editToken], 
                        $tokenResult
                    );
                }
                $saleId = $tokenResult['id'];
                $validToken = $tokenResult['token'];
                $sale['id'] = $saleId;
            }
            
            // 3. VALIDACIJA PODATAKA FORME
            $rules = [
                'sale[sale_imeiprezime]' => ['required', 'alpha_spaces', ['max', 100]],
                'sale[sale_oib]' => ['required', 'oib'],
                'sale[sale_brojoi]' => ['required', 'numeric', ['min', 9],['max', 9]],
                'sale[sale_datumrodj]' => ['required', 'date'],
                'sale[sale_adresa]' => ['required', 'plain_text'],
                'sale[sale_brojzaprijenos]' => ['optional', 'numeric'],
                'sale[sale_maticnioperator]' => ['optional', 'comment'],
                'sale[sale_kontaktbroj]' => ['required', 'plain_text'],
                'sale[sale_mail]' => ['optional', 'email'],
                // NOVA POLJA
                'sale[sale_imeiprezime_prijenosvl]' => ['optional', 'alpha_spaces', ['max', 255]],
                'sale[sale_oib_prijenosvl]' => ['optional', 'oib'],
                'sale[sale_paketi]' => ['optional', 'plain_text', ['max', 40]],
                // POSTOJEĆA POLJA NASTAVAK
                'sale[sale_nacinprimanja]' => ['optional', 'plain_text'],
                'sale[sale_brojzap2p]' => ['optional', 'numeric'],
                'sale[sale_brojzafnp]' => ['optional', 'numeric'],
                'sale[sale_prodanausluga]' => ['required', 'plain_text'],
                'sale[sale_tvusluga]' => ['optional', 'alpha_num'],
                'sale[sale_kucnitel]' => ['optional', 'alpha_num'],
                'sale[sale_mobilniuredaj]' => ['optional', 'alpha_num'],
                'sale[sale_uredajnavesti]' => ['optional', 'plain_text'],
                'sale[sale_uredajnarate]' => ['optional', 'plain_text'],
                'sale[sale_brojnarudzbe]' => ['optional', 'plain_text'],
                'sale[sale_promocija]' => ['optional', 'plain_text'],
                'sale[sale_oibzamozaik]' => ['optional', 'plain_text'],
                'sale[sale_djelatnik]' => ['optional', 'alpha_spaces'],
                'sale[sale_kontrolor]' => ['optional', 'alpha_spaces'],
                'sale[sale_kontrolnipoziv]' => ['optional', 'alpha_spaces'],
                'sale[sale_status]' => ['optional', 'plain_text'],
                'sale[sale_statusdate]' => ['optional', 'date'],
                'sale[sale_comment]' => ['optional', 'comment']
            ];

            $validation = $request->validate($rules);
            if ($validation->fails()) {
                $newEditToken = null;
                if ($saleId && $validToken) {
                    $newEditToken = $this->refreshEditToken($validToken, 'sale');
                }
                
                return $this->renderSalesForm(
                    ['sale' => $sale, 'edit_token' => $newEditToken],
                    $validation->errors()
                );
            }

            // 4. USPJEŠNA VALIDACIJA - troši token
            if ($validToken) {
                $this->consumeEditToken($validToken);
            }

            // 5. PRIPREMA PODATAKA ZA SPREMANJE
            $korisnik = $this->authentication->getUser();
            $sale['user_id'] = $korisnik->id;
            
            if (empty($sale['id'])) {
                $sale['sale_date'] = new \DateTime(\baseKRIZAN\Config\Config::get('defaulttimezone'));
                $sale['sale_djelatnik'] = $korisnik->user_firstandlastname;
                $sale['sale_kontrolor'] = "Mia Srdanović";
                $sale['sale_status'] = "Potencijal";
            }

            $this->processDateFields($sale);
            
            // 6. SPREMANJE PODATAKA
            $this->salesTable->save($sale);

            // 7. PREUSMJERAVANJE
            return Response::redirect('sales/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error saving sale', [
                'message' => $e->getMessage(),
                'sale' => $sale ?? null
            ]);
            return $this->response()->renderError('Failed to save sale');
        }
    }

    public function list(): Response 
    {
        try {
            return $this->response()->render(
                'sales/resources/views/sales/list.html.php',
                [
                    'title' => 'Sve prodaje'
                ],
                'Sve prodaje'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading sales list', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load sales list');
        }
    }

    public function fetchsales(): Response 
    {
        try {
            $requestData = $_POST;
            $korisnik = $this->authentication->getUser();

            // 1. KORISTI INTERNI MAPPING
            $salesMapping = $this->getDataTablesMapping();
            
            // 2. PRIPREMI KOLUMNE ZA DatatablesSSP
            $columns = [];
            foreach ($salesMapping['columns'] as $index => $column) {
                $columns[] = [
                    'db' => $column['sqlcolumnname'],
                    'dt' => $index
                ];
            }

            // 3. PRIPREMI WHERE UVJETE OVISNO O KORISNIČKOJ ULOZI
            $whereResult = null;
            if ($korisnik->hasPermission(\Models\Entity\Korisnik::KORISNIK_sales_kontrolor) ||
                $korisnik->hasPermission(\Models\Entity\Korisnik::KORISNIK_MASTER)) {
                // Vrati sve podatke
                $whereResult = null;
            } 
            elseif ($korisnik->hasPermission(\Models\Entity\Korisnik::KORISNIK_sales_agent)) {
                // Vrati samo podatke trenutnog korisnika - SIGURNO ESCAPED
                $escapedName = $this->dbConnection->getPdo()->quote($korisnik->user_firstandlastname);
                $whereResult = "`sale_djelatnik` = " . $escapedName;
            }

            // 4. POZOVI DatatablesSSP DIREKTNO
            $data = DatatablesSSP::complexWithUniques(
                $requestData,
                $this->dbConnection,
                $salesMapping['table_name'],
                $salesMapping['primary_key'],
                $columns,
                $salesMapping['columns'],
                $whereResult
            );

            // 5. DODAJ KONFIGURACIJU NA PRVI ZAHTJEV
            if (isset($requestData['init']) && $requestData['init'] === 'true') {
                $data['config'] = [
                    'columns' => $salesMapping['columns'],
                    'fixedColumns' => $salesMapping['fixed_columns'],
                    'routeName' => $salesMapping['route_name']
                ];
                
                if (isset($salesMapping['rowStyling'])) {
                    $data['config']['rowStyling'] = $salesMapping['rowStyling'];
                }
            }
            
            echo json_encode($data);
            exit;
        } catch (\Throwable $e) {
            $this->logger->error('Error fetching sales data', [
                'message' => $e->getMessage()
            ]);
            return Response::json(['error' => 'Failed to fetch sales data'], 500);
        }
    }

    public function delete(Request $request): Response 
    {
        try {
            $korisnik = $this->authentication->getUser();
            $saleId = $request->getPost('id');
            $sale = $this->salesTable->findById($saleId);
            
            if (!$sale) {
                $this->logger->security('Attempt to delete non-existent sale', [
                    'id' => $saleId,
                    'user_id' => $korisnik->id,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Record not found');
            }

            if (!$korisnik->hasPermission(\Models\Entity\Korisnik::KORISNIK_sales_administrator) && 
                !$korisnik->hasPermission(\Models\Entity\Korisnik::KORISNIK_sales_supervisor)) {
                $this->logger->security('Unauthorized sale deletion attempt', [
                    'sale_id' => $saleId,
                    'user_id' => $korisnik->id,
                    'ip' => $request->getIp()
                ]);
                return $this->renderSalesForm(
                    [],
                    ['error' => ['Nemate potrebna prava za ovu akciju']]
                );
            }

            $this->salesTable->delete($sale->id);
            
            $this->logger->security('Sale deleted', [
                'sale_id' => $saleId,
                'deleted_by' => $korisnik->id,
                'ip' => $request->getIp()
            ]);

            return Response::redirect('sales/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error deleting sale', [
                'message' => $e->getMessage(),
                'id' => $saleId ?? null
            ]);
            return $this->response()->renderError('Failed to delete sale');
        }
    }

    public function admin(): Response 
    {
        try {
            $statistics = [
                'totalSales' => $this->adminTable->getTotalSalesCount(),
                'activeSales' => $this->adminTable->getActiveSalesCount(),
                'potentialSales' => $this->adminTable->getPotentialSalesCount(),
                'stornoSales' => $this->adminTable->getStornoSalesCount(),
                'conversionRate' => $this->adminTable->getConversionRate(),
                'stornoRate' => $this->adminTable->getStornoRate(),
                'monthlyGrowth' => $this->adminTable->getMonthlyGrowthRate(),
                'salesFunnel' => $this->adminTable->getSalesFunnelData(),
                'monthlyTrends' => $this->adminTable->getMonthlyTrendData(),
                'topServices' => $this->adminTable->getTopServicesData(),
                'topAgents' => $this->adminTable->getTopAgentsData(),
                'statusDistribution' => $this->adminTable->getStatusDistribution(),
                'todaySales' => $this->adminTable->getTodaySalesCount(),
                'thisWeekSales' => $this->adminTable->getThisWeekSalesCount(),
                'lastWeekSales' => $this->adminTable->getLastWeekSalesCount(),
                'recentConversions' => $this->adminTable->getRecentConversions(),
                'pendingActions' => $this->adminTable->getPendingActionsCount(),
                'bestAgentThisMonth' => $this->adminTable->getBestAgentThisMonth(),
                'topServiceThisMonth' => $this->adminTable->getTopServiceThisMonth()
            ];
            
            return $this->response()->render(
                'sales/resources/views/sales/admin.html.php',
                [
                    'title' => 'Sales Dashboard',
                    'statistics' => $statistics
                ],
                'Sales Dashboard'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading sales admin dashboard', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load sales admin dashboard');
        }
    }

    public function getAdminStats(): Response
    {
        try {
            $statistics = [
                'totalSales' => $this->adminTable->getTotalSalesCount(),
                'activeSales' => $this->adminTable->getActiveSalesCount(),
                'potentialSales' => $this->adminTable->getPotentialSalesCount(),
                'stornoSales' => $this->adminTable->getStornoSalesCount(),
                'conversionRate' => $this->adminTable->getConversionRate(),
                'stornoRate' => $this->adminTable->getStornoRate(),
                'monthlyGrowth' => $this->adminTable->getMonthlyGrowthRate(),
                'todaySales' => $this->adminTable->getTodaySalesCount(),
                'thisWeekSales' => $this->adminTable->getThisWeekSalesCount(),
                'pendingActions' => $this->adminTable->getPendingActionsCount()
            ];
            
            return Response::json($statistics);
        } catch (\Throwable $e) {
            $this->logger->error('Error fetching admin stats', [
                'message' => $e->getMessage()
            ]);
            return Response::json(['error' => 'Failed to fetch admin stats'], 500);
        }
    }

    public function export(Request $request): Response 
    {
        try {
            $currentUser = $this->authentication->getUser();
            if (!$currentUser || !$currentUser->hasPermission(\Models\Entity\Korisnik::KORISNIK_sales_kontrolor)) {
                throw new \RuntimeException('Unauthorized to export sales');
            }
            
            $sales = $this->salesTable->findAll();
            
            if (count($sales) > self::MAX_EXPORT_ROWS) {
                throw new \RuntimeException(
                    sprintf('Export limit exceeded: %d rows', self::MAX_EXPORT_ROWS)
                );
            }
            
            $columns = [
                'id', 'user_id', 'sale_date', 'sale_imeiprezime', 
                'sale_oib', 'sale_brojoi', 'sale_datumrodj', 
                'sale_adresa', 'sale_prodanausluga', 'sale_djelatnik', 
                'sale_status', 'sale_statusdate'
            ];
            
            require_once APP_ROOT . '/app/Helpers/excelexport.php';
            $savefileas = sprintf(
                "sve_prodaje_%s_tjedan_%s",
                date('d-m-Y'),
                date("W", strtotime(date('d-m-Y')))
            );
            
            $exportData = excelexport('sales', $columns, $savefileas, $sales);
            
            return Response::download(
                $exportData['content'],
                $exportData['filename'],
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Export failed', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id', []),
                'ip' => $request->getIp()
            ]);
            return $this->response()->renderError('Failed to export sales');
        }
    }

    public function checkOib(Request $request): Response
    {
        try {
            $oib = $request->getPost('oib');
            $currentId = $request->getPost('currentId');

            if (!$oib) {
                return Response::json([
                    'error' => 'OIB is required',
                    'exists' => false
                ], 400);
            }

            $conditions = [];
            $conditions[] = ['sale_oib', $oib, '='];
            
            if ($currentId) {
                $conditions[] = ['id', $currentId, '<>'];
            }

            $existingRecords = $this->salesTable->findMultiCondition($conditions);
            
            return Response::json([
                'exists' => count($existingRecords) > 0
            ]);
        } catch (\Throwable $e) {
            $this->logger->error('Error checking OIB: ' . $e->getMessage());
            
            return Response::json([
                'error' => 'Internal server error',
                'exists' => false
            ], 500);
        }
    }

    public function inlineEdit(Request $request): Response 
    {
        try {
            $id = $request->getPost('id');
            $field = $request->getPost('field');
            $value = $request->getPost('value');
            
            if (!$id || !$field) {
                throw new \Exception('Nedostaju obavezni parametri');
            }
            
            $sale = $this->salesTable->findById($id);
            if (!$sale) {
                throw new \Exception('Zapis nije pronađen');
            }
            
            $korisnik = $this->authentication->getUser();
            
            if (!$korisnik->hasPermission(\Models\Entity\Korisnik::KORISNIK_sales_administrator) && 
                !$korisnik->hasPermission(\Models\Entity\Korisnik::KORISNIK_sales_supervisor) &&
                !$korisnik->hasPermission(\Models\Entity\Korisnik::KORISNIK_sales_kontrolor)) {
                $this->logger->security('Unauthorized inline edit attempt', [
                    'user_id' => $korisnik->id,
                    'sale_id' => $id,
                    'field' => $field,
                    'ip' => $request->getIp()
                ]);
                throw new \Exception('Nedovoljno prava za uređivanje');
            }
            
            // KORISTI INTERNI MAPPING UMJESTO UČITAVANJA DATOTEKE
            $salesMapping = $this->getDataTablesMapping();
            
            // PROVJERI DA LI JE POLJE DOPUŠTENO ZA INLINE EDITING
            $isEditable = false;
            foreach ($salesMapping['columns'] as $column) {
                if ($column['sqlcolumnname'] === $field && 
                    isset($column['inlineediting']) && 
                    $column['inlineediting']['enabled'] === self::YES) {
                    $isEditable = true;
                    break;
                }
            }
            
            if (!$isEditable) {
                $this->logger->security('Attempt to edit non-editable field', [
                    'user_id' => $korisnik->id,
                    'sale_id' => $id,
                    'field' => $field,
                    'ip' => $request->getIp()
                ]);
                throw new \Exception('Polje nije dopušteno za uređivanje');
            }
            
            // AŽURIRAJ POLJE
            $sale->{$field} = $value;
            $this->salesTable->save((array)$sale);
            
            $this->logger->info('Sale record updated via inline edit', [
                'user_id' => $korisnik->id,
                'sale_id' => $id,
                'field' => $field
            ]);
            
            return Response::json(['success' => true]);
        } catch (\Throwable $e) {
            $this->logger->error('Error in inline edit', [
                'message' => $e->getMessage(),
                'id' => $id ?? null,
                'field' => $field ?? null
            ]);
            
            return Response::json([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    // HELPER METHODS
    private function processDateFields(array &$sale): void 
    {
        require_once APP_ROOT . '/moduli/sales/Helpers/convertDateToEnglishFormat.php';
        
        if (isset($sale['sale_datumrodj'])) {
            $sale['sale_datumrodj'] = convertDateTimeToEnglishFormat($sale['sale_datumrodj']);
        }
        if (isset($sale['sale_statusdate'])) {
            $sale['sale_statusdate'] = convertDateTimeToEnglishFormat($sale['sale_statusdate']);
        }
    }

    private function renderSalesForm(array $data, array $errors): Response 
    {
        $sale = null;

        if (isset($data['sale'])) {
            $sale = is_array($data['sale']) 
                ? (object)$data['sale'] 
                : $data['sale'];
        }

        $vrste = $this->vrste2Table->findAll();
        $osobe = $this->osobe2Table->find('vidljivost', '1');

        return $this->response()->render(
            'sales/resources/views/sales/edit.html.php',
            [
                'errors' => $errors,
                'sale' => $sale,
                'edit_token' => $data['edit_token'] ?? null,
                'user' => $this->authentication->getUser(),
                'vrste' => $vrste,
                'osobe' => $osobe
            ],
            'Uredi prodaju'
        );
    }
}