<?php

namespace sales\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Controllers\BaseController;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use sales\Models\DatabaseTableSales;
use baseKRIZAN\Validation\Validator;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;

class Target extends BaseController 
{
    private DatabaseTable $korisniciTable;
    private DatabaseTableSales $salesTable;
    private DatabaseTableSales $targetTable;
    private Validator $validator;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableSales $salesTable,
        DatabaseTableSales $targetTable,
        Validator $validator,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->salesTable = $salesTable;
        $this->targetTable = $targetTable;
        $this->validator = $validator;
    }

    public function edit(Request $request): Response 
    {
        try {
            $target = null;
            $editToken = null;

            if ($id = $request->getParam('id')) {
                $target = $this->targetTable->findById($id);
                if ($target) {
                    $editToken = $this->createEditToken($target->id, 'target');
                }
            }

            return $this->response()->render(
                'sales/resources/views/target/edit.html.php',
                [
                    'target' => $target,
                    'edit_token' => $editToken
                ],
                'Edit Target'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading target edit form', [
                'message' => $e->getMessage(),
                'id' => $id ?? null
            ]);
            return $this->response()->renderError('Failed to load target edit form');
        }
    }

    public function saveEdit(Request $request): Response 
    {
        try {
            // 1. DOHVAĆANJE OSNOVNIH PODATAKA
            $editToken = $request->getPost('edit_token');
            $target = $request->getPost('target');
            $targetId = null;
            $validToken = null;
            
            // 2. VALIDACIJA TOKENA (AKO POSTOJI) - ovo NE troši token
            if ($editToken) {
                $tokenResult = $this->validateEditTokenForValidation($request, 'target');
                if (isset($tokenResult['error'])) {
                    return $this->renderTargetForm(
                        $target, 
                        $editToken, 
                        $tokenResult
                    );
                }
                // Token je valjan
                $targetId = $tokenResult['id'];
                $validToken = $tokenResult['token'];
                $target['id'] = $targetId;
            }
            
            // 3. VALIDACIJA PODATAKA FORME
            $rules = [
                'target[prijenos]' => ['required', 'numeric'], // Promjena: prijenos_bez_fokusa -> prijenos
                'target[p2p]' => ['required', 'numeric'],
                'target[tomato]' => ['required', 'numeric'],
                'target[target_period]' => ['required', 'regex:/^\d{4}-\d{2}$/'] // New validation rule
            ];

            $validation = $request->validate($rules);
            if ($validation->fails()) {
                // Stvori novi token za ponovni pokušaj
                $newEditToken = null;
                if ($targetId && $validToken) {
                    $newEditToken = $this->refreshEditToken($validToken, 'target');
                }
                
                return $this->renderTargetForm(
                    $target, 
                    $newEditToken, 
                    $validation->errors()
                );
            }

            // 4. USPJEŠNA VALIDACIJA - SADA troši token (jednokratno)
            if ($validToken) {
                $this->consumeEditToken($validToken);
            }

            // 5. SPREMANJE PODATAKA
            $this->targetTable->save($target);
            
            // 6. PREUSMJERAVANJE
            return Response::redirect('saletarget/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error saving target', [
                'message' => $e->getMessage(),
                'target' => $target ?? null
            ]);
            return $this->response()->renderError('Failed to save target');
        }
    }

    public function list(): Response 
    {
        try {
            $targets = $this->targetTable->findAll();
            $this->sessionManager->set('valid_target_ids', array_map(fn($target) => $target->id, $targets));

            return $this->response()->render(
                'sales/resources/views/target/list.html.php',
                [
                    'user' => $this->authentication->getUser(),
                    'targets' => $targets
                ],
                'Svi Targeti'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading targets list', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load targets list');
        }
    }

    public function delete(Request $request): Response 
    {
        try {
            $targetId = $request->getPost('id');
            
            // Sigurnosna provjera da li je ID na listi valjanih ID-ova
            $validTargetIds = $this->sessionManager->get('valid_target_ids', []);
            if (!in_array($targetId, $validTargetIds ?? [])) {
                $this->logger->security('Unauthorized target deletion attempt', [
                    'id' => $targetId,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Unauthorized action');
            }

            // Brisanje targeta
            $this->targetTable->delete($targetId);
            $this->removeFromSession($targetId);

            return Response::redirect('saletarget/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error deleting target', [
                'message' => $e->getMessage(),
                'id' => $targetId ?? null
            ]);
            return $this->response()->renderError('Failed to delete target');
        }
    }
    
    /**
     * Remove target ID from session
     */
    private function removeFromSession(int $targetId): void 
    {
        $validIds = $this->sessionManager->get('valid_target_ids', []);
        $validIds = array_filter($validIds, fn($id) => $id != $targetId);
        $this->sessionManager->set('valid_target_ids', $validIds);
    }
    
    /**
     * Helper method to render the target form with errors
     */
    private function renderTargetForm(array $target, ?string $editToken, array $errors): Response
    {
        return $this->response()->render(
            'sales/resources/views/target/edit.html.php',
            [
                'target' => (object)$target,
                'edit_token' => $editToken,
                'errors' => $errors
            ],
            'Edit Target'
        );
    }
}