<?php

namespace sales\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Models\DatabaseTable;
use Controllers\BaseController;
use sales\Models\DatabaseTableTeamBonus;     // Novi TeamBonus model
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Error\Logger;

class TeamBonus extends BaseController
{
    private DatabaseTable $korisniciTable;
    private DatabaseTableTeamBonus $teamBonusTable;    // Specializirani model

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableTeamBonus $teamBonusTable,        // Novi model umjesto salesTable
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->teamBonusTable = $teamBonusTable;        // Specializirani model
    }

    /**
     * Team bonus dashboard page
     * 
     * @param Request $request The request object
     * @return Response The response object
     */
    public function index(Request $request): Response 
    {
        try {
            // Koristi teamBonusTable za sve TeamBonus analytics
            $statistics = [
                'monthlyActiveSales' => $this->teamBonusTable->getMonthlyActiveSalesByService(),
                'activeSalesByService' => $this->teamBonusTable->getActiveSalesByService(),
                'totalActivesByMonth' => $this->teamBonusTable->getTotalActivesByMonth(),
                'serviceGrowthTrends' => $this->teamBonusTable->getServiceGrowthTrends(),
                'currentMonthActives' => $this->teamBonusTable->getCurrentMonthActiveCount(),
                'lastMonthActives' => $this->teamBonusTable->getLastMonthActiveCount(),
                'totalActiveServices' => $this->teamBonusTable->getTotalActiveServices(),
                'topPerformingServices' => $this->teamBonusTable->getTopPerformingServices(),
                'monthlyComparison' => $this->teamBonusTable->getMonthlyActivesComparison()
            ];
            
            return $this->response()->render(
                'sales/resources/views/teambonus/index.html.php',
                [
                    'title' => 'Team Bonus Dashboard',
                    'statistics' => $statistics,
                    'user' => $this->authentication->getUser()
                ],
                'Team Bonus Dashboard'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading team bonus dashboard', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load team bonus dashboard');
        }
    }

    /**
     * Get team bonus statistics via AJAX
     */
    public function getStats(Request $request): Response
    {
        try {
            $statistics = [
                'monthlyActiveSales' => $this->teamBonusTable->getMonthlyActiveSalesByService(),
                'activeSalesByService' => $this->teamBonusTable->getActiveSalesByService(),
                'totalActivesByMonth' => $this->teamBonusTable->getTotalActivesByMonth(),
                'serviceGrowthTrends' => $this->teamBonusTable->getServiceGrowthTrends(),
                'currentMonthActives' => $this->teamBonusTable->getCurrentMonthActiveCount(),
                'lastMonthActives' => $this->teamBonusTable->getLastMonthActiveCount(),
                'totalActiveServices' => $this->teamBonusTable->getTotalActiveServices(),
                'topPerformingServices' => $this->teamBonusTable->getTopPerformingServices(),
                'monthlyComparison' => $this->teamBonusTable->getMonthlyActivesComparison()
            ];
            
            return Response::json($statistics);
        } catch (\Throwable $e) {
            $this->logger->error('Error fetching team bonus stats', [
                'message' => $e->getMessage()
            ]);
            return Response::json(['error' => 'Failed to fetch team bonus stats'], 500);
        }
    }

    /**
     * Export team bonus report to Excel
     * 
     * @param Request $request The request object
     * @return Response The response object
     */
    public function export(Request $request): Response 
    {
        try {
            $currentUser = $this->authentication->getUser();
            if (!$currentUser || !$currentUser->hasPermission(\Models\Entity\Korisnik::KORISNIK_sales_supervisor)) {
                throw new \RuntimeException('Unauthorized to export team bonus data');
            }
            
            // FIXED: koristi teamBonusTable umjesto salesTable
            $monthlyData = $this->teamBonusTable->getMonthlyActiveSalesByService();
            $serviceData = $this->teamBonusTable->getActiveSalesByService();
            
            // Prepare export data
            $exportData = [];
            foreach ($monthlyData as $record) {
                $exportData[] = [
                    'mjesec' => $record['month'],
                    'godina' => $record['year'],
                    'usluga' => $record['service_type'],
                    'broj_aktivnih' => $record['active_count']
                ];
            }
            
            $columns = ['mjesec', 'godina', 'usluga', 'broj_aktivnih'];
            
            require_once APP_ROOT . '/app/Helpers/excelexport.php';
            $savefileas = sprintf(
                "team_bonus_report_%s",
                date('Y-m-d')
            );
            
            $export = excelexport('team_bonus', $columns, $savefileas, $exportData);
            
            return Response::download(
                $export['content'],
                $export['filename'],
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Team bonus export failed', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id', []),
                'ip' => $request->getIp()
            ]);
            return $this->response()->renderError('Failed to export team bonus data');
        }
    }

    /**
     * Generate detailed report for specific service and time period
     * 
     * @param Request $request The request object
     * @return Response The response object
     */
    public function detailReport(Request $request): Response 
    {
        try {
            $serviceType = $request->getParam('service', '');
            $month = $request->getParam('month', date('m'));
            $year = $request->getParam('year', date('Y'));
            
            if (empty($serviceType)) {
                return $this->response()->renderError('Service type is required for detailed report');
            }
            
            // FIXED: koristi teamBonusTable umjesto salesTable
            $detailData = $this->teamBonusTable->getDetailedActiveSalesForService($serviceType, $month, $year);
            $summary = $this->teamBonusTable->getServiceSummaryForPeriod($serviceType, $month, $year);
            
            return $this->response()->render(
                'sales/resources/views/teambonus/detail_report.html.php',
                [
                    'title' => 'Detaljni izvještaj - ' . $serviceType,
                    'serviceType' => $serviceType,
                    'month' => $month,
                    'year' => $year,
                    'detailData' => $detailData,
                    'summary' => $summary,
                    'user' => $this->authentication->getUser()
                ],
                'Detaljni izvještaj team bonusa'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Failed to generate detailed team bonus report', [
                'message' => $e->getMessage(),
                'service' => $serviceType ?? 'unknown',
                'month' => $month ?? 'unknown',
                'year' => $year ?? 'unknown'
            ]);
            return $this->response()->renderError('Failed to generate detailed report');
        }
    }

    /**
     * Print report view
     * 
     * @param Request $request The request object
     * @return Response The response object
     */
    public function print(Request $request): Response 
    {
        try {
            $currentUser = $this->authentication->getUser();
            
            $month = $request->getParam('month', date('m'));
            $year = $request->getParam('year', date('Y'));
            
            // FIXED: koristi teamBonusTable umjesto salesTable
            $monthlyData = $this->teamBonusTable->getMonthlyActiveSalesByService($month, $year);
            $serviceData = $this->teamBonusTable->getActiveSalesByService();
            
            $title = "Team Bonus izvještaj za {$month}/{$year}";
            
            return $this->response()->render(
                'sales/resources/views/teambonus/print.html.php',
                [
                    'monthlyData' => $monthlyData,
                    'serviceData' => $serviceData,
                    'title' => $title,
                    'month' => $month,
                    'year' => $year,
                    'user' => $currentUser,
                    'print_date' => date('d.m.Y H:i:s')
                ],
                $title
            );
        } catch (\Throwable $e) {
            $this->logger->error('Team bonus print report failed', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id', []),
                'ip' => $request->getIp()
            ]);
            return $this->response()->renderError('Failed to generate print report');
        }
    }
}