<?php

namespace sales\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Models\DatabaseTable;
use Controllers\BaseController;
use sales\Models\DatabaseTableTeamBonusFinancije;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Error\Logger;

class TeamBonusFinancije extends BaseController
{
    private DatabaseTable $korisniciTable;
    private DatabaseTableTeamBonusFinancije $teamBonusFinancijeTable;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableTeamBonusFinancije $teamBonusFinancijeTable,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->teamBonusFinancijeTable = $teamBonusFinancijeTable;
    }

    /**
     * Team bonus financije dashboard page
     * 
     * @param Request $request The request object
     * @return Response The response object
     */
    public function index(Request $request): Response 
    {
        try {
            // Pravilno dohvaćanje parametara kao u Bonus kontroleru
            $month = $request->getQuery('month') ?? 
                    $request->getParam('month') ?? 
                    $request->input('month') ??
                    $_GET['month'] ?? 
                    date('m');
                    
            $year = $request->getQuery('year') ?? 
                   $request->getParam('year') ?? 
                   $request->input('year') ??
                   $_GET['year'] ?? 
                   date('Y');
            
            // Debug log
            if ($this->logger) {
                $this->logger->info('TeamBonusFinancije filter request', [
                    'month' => $month,
                    'year' => $year,
                    'query_params' => $request->getQuery(),
                    'get_params' => $_GET,
                    'url' => $_SERVER['REQUEST_URI'] ?? 'unknown'
                ]);
            }
            
            // Get financial statistics
            $earnings = $this->teamBonusFinancijeTable->getFinancialEarningsByType($month, $year);
            $totalEarnings = $this->teamBonusFinancijeTable->getTotalCompanyEarnings($month, $year);
            
            // Get comparison with previous month
            $prevMonth = $month == 1 ? 12 : $month - 1;
            $prevYear = $month == 1 ? $year - 1 : $year;
            $comparison = $this->teamBonusFinancijeTable->getEarningsComparison(
                $month, $year, 
                $prevMonth, $prevYear
            );
            
            // Get monthly trend for charts
            $monthlyTrend = $this->teamBonusFinancijeTable->getMonthlyEarningsTrend(6);
            
            // Prepare statistics
            $statistics = [
                'earnings' => $earnings,
                'totalEarnings' => $totalEarnings,
                'comparison' => $comparison,
                'monthlyTrend' => $monthlyTrend,
                'currentMonth' => $month,
                'currentYear' => $year,
                'selectedPeriod' => "$year-$month"
            ];
            
            return $this->response()->render(
                'sales/resources/views/teambonusfinancije/index.html.php',
                [
                    'title' => 'Team Bonus Financije Dashboard',
                    'statistics' => $statistics,
                    'user' => $this->authentication->getUser()
                ],
                'Team Bonus Financije Dashboard'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading team bonus financije dashboard', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load team bonus financije dashboard');
        }
    }

    /**
     * Get team bonus financije statistics via AJAX
     */
    public function getStats(Request $request): Response
    {
        try {
            // Pravilno dohvaćanje parametara iz AJAX zahtjeva
            $month = $request->getPost('month') ?? 
                    $request->getQuery('month') ?? 
                    $request->getParam('month') ?? 
                    $request->input('month') ??
                    $_GET['month'] ?? 
                    date('m');
                    
            $year = $request->getPost('year') ?? 
                   $request->getQuery('year') ?? 
                   $request->getParam('year') ?? 
                   $request->input('year') ??
                   $_GET['year'] ?? 
                   date('Y');
            
            $earnings = $this->teamBonusFinancijeTable->getFinancialEarningsByType($month, $year);
            $totalEarnings = $this->teamBonusFinancijeTable->getTotalCompanyEarnings($month, $year);
            
            $prevMonth = $month == 1 ? 12 : $month - 1;
            $prevYear = $month == 1 ? $year - 1 : $year;
            $comparison = $this->teamBonusFinancijeTable->getEarningsComparison(
                $month, $year, 
                $prevMonth, $prevYear
            );
            
            $monthlyTrend = $this->teamBonusFinancijeTable->getMonthlyEarningsTrend(6);
            
            return Response::json([
                'earnings' => $earnings,
                'totalEarnings' => $totalEarnings,
                'comparison' => $comparison,
                'monthlyTrend' => $monthlyTrend,
                'selectedPeriod' => "$year-$month"
            ]);
        } catch (\Throwable $e) {
            $this->logger->error('Error fetching team bonus financije stats', [
                'message' => $e->getMessage()
            ]);
            return Response::json(['error' => 'Failed to fetch team bonus financije stats'], 500);
        }
    }

    /**
     * Export team bonus financije report to Excel
     * 
     * @param Request $request The request object
     * @return Response The response object
     */
    public function export(Request $request): Response 
    {
        try {
            $currentUser = $this->authentication->getUser();
            if (!$currentUser || !$currentUser->hasPermission(\Models\Entity\Korisnik::KORISNIK_sales_supervisor)) {
                throw new \RuntimeException('Unauthorized to export team bonus financije data');
            }
            
            $month = $request->getParam('month', date('m'));
            $year = $request->getParam('year', date('Y'));
            
            $earnings = $this->teamBonusFinancijeTable->getFinancialEarningsByType($month, $year);
            
            // Prepare export data
            $exportData = [];
            
            foreach ($earnings as $bonusType => $data) {
                foreach ($data['services'] as $service) {
                    $exportData[] = [
                        'bonus_tip' => $bonusType,
                        'tarifa' => $service['tarifa'],
                        'broj_prodanih' => $service['count'],
                        'zarada_eur' => number_format($service['earnings'], 2),
                        'mjesec' => $month,
                        'godina' => $year
                    ];
                }
            }
            
            $columns = ['bonus_tip', 'tarifa', 'broj_prodanih', 'zarada_eur', 'mjesec', 'godina'];
            
            require_once APP_ROOT . '/app/Helpers/excelexport.php';
            $savefileas = sprintf(
                "team_bonus_financije_report_%s_%s",
                $year,
                str_pad($month, 2, '0', STR_PAD_LEFT)
            );
            
            $export = excelexport('team_bonus_financije', $columns, $savefileas, $exportData);
            
            return Response::download(
                $export['content'],
                $export['filename'],
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Team bonus financije export failed', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id', []),
                'ip' => $request->getIp()
            ]);
            return $this->response()->renderError('Failed to export team bonus financije data');
        }
    }

    /**
     * Generate detailed report for specific bonus type and time period
     * 
     * @param Request $request The request object
     * @return Response The response object
     */
    public function detailReport(Request $request): Response 
    {
        try {
            // Pravilno dohvaćanje parametara
            $bonusType = $request->getQuery('type') ?? 
                        $request->getParam('type') ?? 
                        $request->input('type') ??
                        $_GET['type'] ?? 
                        '';
                        
            $month = $request->getQuery('month') ?? 
                    $request->getParam('month') ?? 
                    $request->input('month') ??
                    $_GET['month'] ?? 
                    date('m');
                    
            $year = $request->getQuery('year') ?? 
                   $request->getParam('year') ?? 
                   $request->input('year') ??
                   $_GET['year'] ?? 
                   date('Y');
            
            if (empty($bonusType)) {
                return $this->response()->renderError('Bonus type is required for detailed report');
            }
            
            $earnings = $this->teamBonusFinancijeTable->getFinancialEarningsByType($month, $year);
            $detailData = $earnings[$bonusType] ?? null;
            
            if (!$detailData) {
                return $this->response()->renderError('No data found for specified bonus type');
            }
            
            return $this->response()->render(
                'sales/resources/views/teambonusfinancije/detail_report.html.php',
                [
                    'title' => 'Detaljni izvještaj - ' . ucfirst($bonusType),
                    'bonusType' => $bonusType,
                    'month' => $month,
                    'year' => $year,
                    'detailData' => $detailData,
                    'user' => $this->authentication->getUser()
                ],
                'Detaljni izvještaj team bonus financija'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Failed to generate detailed team bonus financije report', [
                'message' => $e->getMessage(),
                'type' => $bonusType ?? 'unknown',
                'month' => $month ?? 'unknown',
                'year' => $year ?? 'unknown'
            ]);
            return $this->response()->renderError('Failed to generate detailed report');
        }
    }

    /**
     * Print report view
     * 
     * @param Request $request The request object
     * @return Response The response object
     */
    public function print(Request $request): Response 
    {
        try {
            $currentUser = $this->authentication->getUser();
            
            // Pravilno dohvaćanje parametara
            $month = $request->getQuery('month') ?? 
                    $request->getParam('month') ?? 
                    $request->input('month') ??
                    $_GET['month'] ?? 
                    date('m');
                    
            $year = $request->getQuery('year') ?? 
                   $request->getParam('year') ?? 
                   $request->input('year') ??
                   $_GET['year'] ?? 
                   date('Y');
            
            $earnings = $this->teamBonusFinancijeTable->getFinancialEarningsByType($month, $year);
            $totalEarnings = $this->teamBonusFinancijeTable->getTotalCompanyEarnings($month, $year);
            
            $title = "Team Bonus Financije izvještaj za {$month}/{$year}";
            
            return $this->response()->render(
                'sales/resources/views/teambonusfinancije/print.html.php',
                [
                    'earnings' => $earnings,
                    'totalEarnings' => $totalEarnings,
                    'title' => $title,
                    'month' => $month,
                    'year' => $year,
                    'user' => $currentUser,
                    'print_date' => date('d.m.Y H:i:s')
                ],
                $title
            );
        } catch (\Throwable $e) {
            $this->logger->error('Team bonus financije print report failed', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id', []),
                'ip' => $request->getIp()
            ]);
            return $this->response()->renderError('Failed to generate print report');
        }
    }
}