<?php

namespace sales\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Controllers\BaseController;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use sales\Models\DatabaseTableSales;
use baseKRIZAN\Validation\Validator;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;

class Vrsta extends BaseController 
{
    private DatabaseTable $korisniciTable;
    private DatabaseTableSales $salesTable;
    private DatabaseTableSales $vrste2Table;
    private Validator $validator;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableSales $salesTable,
        DatabaseTableSales $vrste2Table,
        Validator $validator,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->salesTable = $salesTable;
        $this->vrste2Table = $vrste2Table;
        $this->validator = $validator;
    }

    public function edit(Request $request): Response 
    {
        try {
            $vrsta = null;
            $editToken = null;

            if ($id = $request->getParam('id')) {
                $vrsta = $this->vrste2Table->findById($id);
                if ($vrsta) {
                    $editToken = $this->createEditToken($vrsta->id, 'vrsta');
                }
            }

            return $this->response()->render(
                'sales/resources/views/vrsta/edit.html.php',
                [
                    'vrsta' => $vrsta,
                    'edit_token' => $editToken
                ],
                'Edit Vrsta'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading vrsta edit form', [
                'message' => $e->getMessage(),
                'id' => $id ?? null
            ]);
            return $this->response()->renderError('Failed to load vrsta edit form');
        }
    }

    public function saveEdit(Request $request): Response 
    {
        try {
            // 1. DOHVAĆANJE OSNOVNIH PODATAKA
            $editToken = $request->getPost('edit_token');
            $vrsta = $request->getPost('vrsta');
            $vrstaId = null;
            $validToken = null;
            
            // 2. VALIDACIJA TOKENA (AKO POSTOJI) - ovo NE troši token
            if ($editToken) {
                $tokenResult = $this->validateEditTokenForValidation($request, 'vrsta');
                if (isset($tokenResult['error'])) {
                    return $this->renderVrstaForm(
                        $vrsta, 
                        $editToken, 
                        $tokenResult
                    );
                }
                // Token je valjan
                $vrstaId = $tokenResult['id'];
                $validToken = $tokenResult['token'];
                $vrsta['id'] = $vrstaId;
            }
            
            // 3. VALIDACIJA PODATAKA FORME
            $rules = [
                'vrsta[TARIFA]' => ['required', 'string', ['max', 40]],
                'vrsta[GRUPA]' => ['required', 'string', ['in', $this->getAllowedGroups()]],
                'vrsta[bodovi]' => ['required', 'numeric', ['min', 0], ['max', 100]],
            ];

            $validation = $request->validate($rules);
            if ($validation->fails()) {
                // Stvori novi token za ponovni pokušaj
                $newEditToken = null;
                if ($vrstaId && $validToken) {
                    $newEditToken = $this->refreshEditToken($validToken, 'vrsta');
                }
                
                return $this->renderVrstaForm(
                    $vrsta, 
                    $newEditToken, 
                    $validation->errors()
                );
            }

            // 4. USPJEŠNA VALIDACIJA - SADA troši token (jednokratno)
            if ($validToken) {
                $this->consumeEditToken($validToken);
            }

            // 5. SPREMANJE PODATAKA
            $this->vrste2Table->save($vrsta);
            
            // 6. PREUSMJERAVANJE
            return Response::redirect('salevrsta/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error saving vrsta', [
                'message' => $e->getMessage(),
                'vrsta' => $vrsta ?? null
            ]);
            return $this->response()->renderError('Failed to save vrsta');
        }
    }
    
    /**
     * Get allowed group values for validation
     * 
     * @return array
     */
    private function getAllowedGroups(): array
    {
        return [
            'MNP HT',
            'Parica', 
            'Homebox',
            'Sat',
            'MNP Telemach',
            'Optika',
            'P2P',
            'Tomato',
            'Tomato P2P'
        ];
    }

    public function list(): Response 
    {
        try {
            $vrste = $this->vrste2Table->findAll();
            $this->sessionManager->set('valid_vrsta_ids', array_map(fn($vrsta) => $vrsta->id, $vrste));

            return $this->response()->render(
                'sales/resources/views/vrsta/list.html.php',
                [
                    'user' => $this->authentication->getUser(),
                    'vrste' => $vrste
                ],
                'Sve Vrste'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading vrste list', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load vrste list');
        }
    }

    public function delete(Request $request): Response 
    {
        try {
            $vrstaId = $request->getPost('id');
            
            // Sigurnosna provjera da li je ID na listi valjanih ID-ova
            $validVrstaIds = $this->sessionManager->get('valid_vrsta_ids', []);
            if (!in_array($vrstaId, $validVrstaIds ?? [])) {
                $this->logger->security('Unauthorized vrsta deletion attempt', [
                    'id' => $vrstaId,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Unauthorized action');
            }

            // Brisanje vrsta
            $this->vrste2Table->delete($vrstaId);
            $this->removeFromSession($vrstaId);

            return Response::redirect('salevrsta/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error deleting vrsta', [
                'message' => $e->getMessage(),
                'id' => $vrstaId ?? null
            ]);
            return $this->response()->renderError('Failed to delete vrsta');
        }
    }
    
    /**
     * Remove vrsta ID from session
     */
    private function removeFromSession(int $vrstaId): void 
    {
        $validIds = $this->sessionManager->get('valid_vrsta_ids', []);
        $validIds = array_filter($validIds, fn($id) => $id != $vrstaId);
        $this->sessionManager->set('valid_vrsta_ids', $validIds);
    }
    
    /**
     * Helper method to render the vrsta form with errors
     */
    private function renderVrstaForm(array $vrsta, ?string $editToken, array $errors): Response
    {
        return $this->response()->render(
            'sales/resources/views/vrsta/edit.html.php',
            [
                'vrsta' => (object)$vrsta,
                'edit_token' => $editToken,
                'errors' => $errors
            ],
            'Edit Vrsta'
        );
    }
}