<?php
// ============================================================================
// 3. DatabaseTableAgentBonus.php - Agent Bonus Analytics Model
// ============================================================================

namespace sales\Models;

use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

class DatabaseTableAgentBonus extends DatabaseTable
{
    /**
     * @var \baseKRIZAN\Database\DatabaseConnection
     */
    protected $dbConnection;
    
    /**
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Constructor for Agent Bonus Analytics (agent-focused)
     * 
     * @param DatabaseConnection $dbConnection Database connection
     * @param string $table Table name
     * @param string $primaryKey Primary key
     * @param string $className Class name
     * @param array $constructorArgs Constructor arguments
     * @param Logger|null $logger Logger instance
     */
    public function __construct(
        DatabaseConnection $dbConnection, 
        string $table, 
        string $primaryKey, 
        string $className = 'stdClass', 
        array $constructorArgs = [],
        ?Logger $logger = null
    ) {
        parent::__construct($dbConnection, $table, $primaryKey, $className, $constructorArgs);
        $this->dbConnection = $dbConnection;
        $this->logger = $logger;
        
        if ($this->logger) {
            $this->logger->modules('DatabaseTableAgentBonus initialized', [
                'table' => $table
            ]);
        }
    }

    /**
     * ===================================================================
     * TEAM BONUS METHODS - Add these methods to DatabaseTableSales.php
     * ===================================================================
     */

    /**
     * Get total active sales by month (all services combined)
     * 
     * @return array Total actives by month
     */
    public function getTotalActivesByMonth(): array
    {
        try {
            $query = "SELECT 
                        MONTH(sale_statusdate) as month,
                        YEAR(sale_statusdate) as year,
                        COUNT(*) as total_active_count
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_statusdate IS NOT NULL
                    GROUP BY YEAR(sale_statusdate), MONTH(sale_statusdate) 
                    ORDER BY year DESC, month DESC";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getTotalActivesByMonth'
                ]);
            }
            return [];
        }
    }

    /**
     * Get current month active sales count
     * 
     * @return int Current month active count
     */
    public function getCurrentMonthActiveCount(): int
    {
        try {
            $currentMonth = date('m');
            $currentYear = date('Y');
            
            $query = "SELECT COUNT(*) as count 
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND MONTH(sale_statusdate) = :month 
                    AND YEAR(sale_statusdate) = :year";
            
            return (int)$this->dbConnection->querySingleValue($query, [
                'month' => $currentMonth,
                'year' => $currentYear
            ]) ?? 0;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getCurrentMonthActiveCount'
                ]);
            }
            return 0;
        }
    }

    /**
     * Get last month active sales count
     * 
     * @return int Last month active count
     */
    public function getLastMonthActiveCount(): int
    {
        try {
            $lastMonth = date('m', strtotime('-1 month'));
            $lastMonthYear = date('Y', strtotime('-1 month'));
            
            $query = "SELECT COUNT(*) as count 
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND MONTH(sale_statusdate) = :month 
                    AND YEAR(sale_statusdate) = :year";
            
            return (int)$this->dbConnection->querySingleValue($query, [
                'month' => $lastMonth,
                'year' => $lastMonthYear
            ]) ?? 0;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getLastMonthActiveCount'
                ]);
            }
            return 0;
        }
    }

    /**
     * ===================================================================
     * AGENT BONUS METHODS - Add these methods to DatabaseTableSales.php
     * ===================================================================
     */

    /**
     * Get monthly active sales grouped by agent
     * 
     * @param string|null $specificMonth Specific month to filter (MM format)
     * @param string|null $specificYear Specific year to filter (YYYY format)
     * @return array Monthly active sales by agent
     */
    public function getMonthlyActivesByAgent(?string $specificMonth = null, ?string $specificYear = null): array
    {
        try {
            $whereClause = "WHERE sale_status = 'Aktivno' AND sale_statusdate IS NOT NULL AND sale_djelatnik IS NOT NULL";
            
            if ($specificMonth && $specificYear) {
                $whereClause .= " AND MONTH(sale_statusdate) = :month AND YEAR(sale_statusdate) = :year";
            }
            
            $query = "SELECT 
                        MONTH(sale_statusdate) as month,
                        YEAR(sale_statusdate) as year,
                        sale_djelatnik as agent_name,
                        COUNT(*) as active_count
                    FROM {$this->table} 
                    {$whereClause}
                    GROUP BY YEAR(sale_statusdate), MONTH(sale_statusdate), sale_djelatnik 
                    ORDER BY year DESC, month DESC, active_count DESC";
            
            $params = [];
            if ($specificMonth && $specificYear) {
                $params = ['month' => (int)$specificMonth, 'year' => (int)$specificYear];
            }
            
            return $this->dbConnection->queryAndFetchAllAssoc($query, $params);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getMonthlyActivesByAgent'
                ]);
            }
            return [];
        }
    }

    /**
     * Get active sales count by agent (all time)
     * 
     * @return array Active sales by agent
     */
    public function getActiveSalesByAgent(): array
    {
        try {
            $query = "SELECT 
                        sale_djelatnik as agent_name,
                        COUNT(*) as active_count
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_djelatnik IS NOT NULL
                    GROUP BY sale_djelatnik 
                    ORDER BY active_count DESC";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getActiveSalesByAgent'
                ]);
            }
            return [];
        }
    }

    /**
     * Get agent service breakdown (what services each agent sold)
     * 
     * @return array Agent service breakdown
     */
    public function getAgentServiceBreakdown(): array
    {
        try {
            $query = "SELECT 
                        sale_djelatnik as agent_name,
                        sale_prodanausluga as service_type,
                        COUNT(*) as service_count
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_djelatnik IS NOT NULL
                    AND sale_prodanausluga IS NOT NULL
                    GROUP BY sale_djelatnik, sale_prodanausluga 
                    ORDER BY agent_name, service_count DESC";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getAgentServiceBreakdown'
                ]);
            }
            return [];
        }
    }

    /**
     * Get agent growth trends (comparing last 6 months)
     * 
     * @return array Agent growth trends
     */
    public function getAgentGrowthTrends(): array
    {
        try {
            $query = "SELECT 
                        sale_djelatnik as agent_name,
                        MONTH(sale_statusdate) as month,
                        YEAR(sale_statusdate) as year,
                        COUNT(*) as active_count
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_statusdate >= DATE_SUB(CURRENT_DATE, INTERVAL 6 MONTH)
                    AND sale_djelatnik IS NOT NULL
                    GROUP BY sale_djelatnik, YEAR(sale_statusdate), MONTH(sale_statusdate) 
                    ORDER BY agent_name, year, month";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getAgentGrowthTrends'
                ]);
            }
            return [];
        }
    }

    /**
     * Get total number of different active agents
     * 
     * @return int Total active agents count
     */
    public function getTotalActiveAgents(): int
    {
        try {
            $query = "SELECT COUNT(DISTINCT sale_djelatnik) as count 
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_djelatnik IS NOT NULL";
            
            return (int)$this->dbConnection->querySingleValue($query) ?? 0;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getTotalActiveAgents'
                ]);
            }
            return 0;
        }
    }

    /**
     * Get top performing agents (by active count)
     * 
     * @param int $limit Number of top agents to return
     * @return array Top performing agents
     */
    public function getTopPerformingAgents(int $limit = 5): array
    {
        try {
            $query = "SELECT 
                        sale_djelatnik as agent_name,
                        COUNT(*) as active_count,
                        COUNT(CASE WHEN MONTH(sale_statusdate) = MONTH(CURRENT_DATE) 
                                  AND YEAR(sale_statusdate) = YEAR(CURRENT_DATE) THEN 1 END) as current_month_count,
                        COUNT(DISTINCT sale_prodanausluga) as services_sold
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_djelatnik IS NOT NULL
                    GROUP BY sale_djelatnik 
                    ORDER BY active_count DESC 
                    LIMIT " . (int)$limit;
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getTopPerformingAgents'
                ]);
            }
            return [];
        }
    }

    /**
     * Get monthly agents comparison (current vs previous month)
     * 
     * @return array Monthly comparison data
     */
    public function getMonthlyAgentComparison(): array
    {
        try {
            $query = "SELECT 
                        sale_djelatnik as agent_name,
                        COUNT(CASE WHEN MONTH(sale_statusdate) = MONTH(CURRENT_DATE) 
                                  AND YEAR(sale_statusdate) = YEAR(CURRENT_DATE) THEN 1 END) as current_month,
                        COUNT(CASE WHEN MONTH(sale_statusdate) = MONTH(DATE_SUB(CURRENT_DATE, INTERVAL 1 MONTH)) 
                                  AND YEAR(sale_statusdate) = YEAR(DATE_SUB(CURRENT_DATE, INTERVAL 1 MONTH)) THEN 1 END) as previous_month
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_djelatnik IS NOT NULL
                    AND sale_statusdate >= DATE_SUB(CURRENT_DATE, INTERVAL 2 MONTH)
                    GROUP BY sale_djelatnik 
                    ORDER BY current_month DESC";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getMonthlyAgentComparison'
                ]);
            }
            return [];
        }
    }

    /**
     * Get detailed active sales for specific agent and period
     * 
     * @param string $agentName Agent name to filter
     * @param string $month Month to filter (MM format)
     * @param string $year Year to filter (YYYY format)
     * @return array Detailed sales data
     */
    public function getDetailedActiveSalesForAgent(string $agentName, string $month, string $year): array
    {
        try {
            $query = "SELECT 
                        id,
                        sale_imeiprezime,
                        sale_oib,
                        sale_adresa,
                        sale_prodanausluga,
                        sale_statusdate,
                        sale_kontaktbroj
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_djelatnik = :agent_name
                    AND MONTH(sale_statusdate) = :month 
                    AND YEAR(sale_statusdate) = :year
                    ORDER BY sale_statusdate DESC";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query, [
                'agent_name' => $agentName,
                'month' => (int)$month,
                'year' => (int)$year
            ]);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getDetailedActiveSalesForAgent'
                ]);
            }
            return [];
        }
    }

    /**
     * Get agent summary for specific period
     * 
     * @param string $agentName Agent name to filter
     * @param string $month Month to filter (MM format)
     * @param string $year Year to filter (YYYY format)
     * @return array|null Agent summary data
     */
    public function getAgentSummaryForPeriod(string $agentName, string $month, string $year): ?array
    {
        try {
            $query = "SELECT 
                        COUNT(*) as total_active,
                        COUNT(DISTINCT sale_prodanausluga) as unique_services,
                        MIN(sale_statusdate) as first_active_date,
                        MAX(sale_statusdate) as last_active_date
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_djelatnik = :agent_name
                    AND MONTH(sale_statusdate) = :month 
                    AND YEAR(sale_statusdate) = :year";
            
            return $this->dbConnection->queryAndFetchAssoc($query, [
                'agent_name' => $agentName,
                'month' => (int)$month,
                'year' => (int)$year
            ]);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getAgentSummaryForPeriod'
                ]);
            }
            return null;
        }
    }

    /**
     * Get agent service breakdown for specific period
     * 
     * @param string $agentName Agent name to filter
     * @param string $month Month to filter (MM format)
     * @param string $year Year to filter (YYYY format)
     * @return array Agent service breakdown for period
     */
    public function getAgentServiceBreakdownForPeriod(string $agentName, string $month, string $year): array
    {
        try {
            $query = "SELECT 
                        sale_prodanausluga as service_type,
                        COUNT(*) as service_count
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_djelatnik = :agent_name
                    AND MONTH(sale_statusdate) = :month 
                    AND YEAR(sale_statusdate) = :year
                    AND sale_prodanausluga IS NOT NULL
                    GROUP BY sale_prodanausluga 
                    ORDER BY service_count DESC";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query, [
                'agent_name' => $agentName,
                'month' => (int)$month,
                'year' => (int)$year
            ]);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getAgentServiceBreakdownForPeriod'
                ]);
            }
            return [];
        }
    }
}