<?php

namespace sales\Models;

use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

class DatabaseTableAgentBonusBodovanje extends DatabaseTable
{
    /**
     * @var \baseKRIZAN\Database\DatabaseConnection
     */
    protected $dbConnection;
    
    /**
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Ranking systems for each agent group
     */
    private array $rankingSystems = [
        'mnp' => [
            ['points' => 130, 'bonus' => 130, 'name' => 'Bronza'],
            ['points' => 190, 'bonus' => 250, 'name' => 'Srebro'],
            ['points' => 250, 'bonus' => 400, 'name' => 'Zlato'],
            ['points' => 400, 'bonus' => 800, 'name' => 'Platina']
        ],
        'optika' => [
            ['points' => 250, 'bonus' => 250, 'name' => 'Bronza'],
            ['points' => 400, 'bonus' => 400, 'name' => 'Srebro'],
            ['points' => 650, 'bonus' => 800, 'name' => 'Zlato']
        ],
        'p2p' => [
            ['points' => 95, 'bonus' => 95, 'name' => 'Bronza'],
            ['points' => 115, 'bonus' => 150, 'name' => 'Srebro'],
            ['points' => 150, 'bonus' => 250, 'name' => 'Zlato'],
            ['points' => 200, 'bonus' => 350, 'name' => 'Platina'],
            ['points' => 300, 'bonus' => 500, 'name' => 'Dijamant']
        ]
    ];

    public function __construct(
        DatabaseConnection $dbConnection, 
        string $table, 
        string $primaryKey, 
        string $className = 'stdClass', 
        array $constructorArgs = [],
        ?Logger $logger = null
    ) {
        parent::__construct($dbConnection, $table, $primaryKey, $className, $constructorArgs);
        $this->dbConnection = $dbConnection;
        $this->logger = $logger;
        
        if ($this->logger) {
            $this->logger->modules('DatabaseTableAgentBonusBodovanje initialized', [
                'table' => $table
            ]);
        }
    }

    /**
     * Get MNP agents statistics with points and rankings
     */
    public function getMNPAgentsStatistics(string $month, string $year): array
    {
        try {
            $mnpAgents = $this->getMNPAgents();
            
            $results = [];
            foreach ($mnpAgents as $agent) {
                $agentStats = $this->calculateAgentPoints($agent['name'], $month, $year, 'mnp');
                if ($agentStats['total_points'] > 0) {
                    $results[] = $agentStats;
                }
            }
            
            usort($results, function($a, $b) {
                return $b['total_points'] <=> $a['total_points'];
            });
            
            return $results;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting MNP agents statistics', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Get Optika agents statistics with points and rankings
     */
    public function getOptikaAgentsStatistics(string $month, string $year): array
    {
        try {
            $optikaAgents = $this->getOptikaAgents();
            
            $results = [];
            foreach ($optikaAgents as $agent) {
                $agentStats = $this->calculateAgentPoints($agent['name'], $month, $year, 'optika');
                if ($agentStats['total_points'] > 0) {
                    $results[] = $agentStats;
                }
            }
            
            usort($results, function($a, $b) {
                return $b['total_points'] <=> $a['total_points'];
            });
            
            return $results;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting Optika agents statistics', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Get P2P agents statistics with points and rankings (AŽURIRANO - uvjet 280 P2P usluga)
     */
    public function getP2PAgentsStatistics(string $month, string $year): array
    {
        try {
            $p2pAgents = $this->getP2PAgents();
            
            $results = [];
            foreach ($p2pAgents as $agent) {
                // NOVA LOGIKA: Provjeri da li agent ima minimalno 280 P2P prodaja u trenutnom mjesecu
                if ($this->hasMinimumP2PSales($agent['name'], $month, $year, 280)) {
                    $agentStats = $this->calculateAgentPoints($agent['name'], $month, $year, 'p2p');
                    if ($agentStats['total_points'] > 0) {
                        $results[] = $agentStats;
                    }
                }
            }
            
            usort($results, function($a, $b) {
                return $b['total_points'] <=> $a['total_points'];
            });
            
            return $results;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting P2P agents statistics', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Get MNP agents from prodaja_osobe table
     */
    private function getMNPAgents(): array
    {
        try {
            $query = "SELECT name FROM prodaja_osobe WHERE orgjedinica = 'Backoffice MNP'";
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting MNP agents', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Get Optika agents from prodaja_osobe table
     */
    private function getOptikaAgents(): array
    {
        try {
            $query = "SELECT name FROM prodaja_osobe WHERE orgjedinica = 'Backoffice Optika'";
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting Optika agents', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Get P2P agents from prodaja_osobe table
     */
    private function getP2PAgents(): array
    {
        try {
            $query = "SELECT name FROM prodaja_osobe WHERE orgjedinica = 'Backoffice P2P'";
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting P2P agents', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * NOVA METODA: Check if P2P agent has minimum required P2P sales (280+)
     * Zamjenjuje staru hasMinimumMNPOptika metodu
     */
    private function hasMinimumP2PSales(string $agentName, string $month, string $year, int $minimum = 280): bool
    {
        try {
            // Trebamo dobiti P2P tarife iz prodaja_bonus_p2p tablice
            $p2pTarife = $this->getP2PTarife();
            
            if (empty($p2pTarife)) {
                return false;
            }
            
            $placeholders = str_repeat('?,', count($p2pTarife) - 1) . '?';
            
            $query = "SELECT COUNT(*) as count
                     FROM {$this->table}
                     WHERE sale_status = 'Aktivno'
                     AND sale_djelatnik = ?
                     AND MONTH(sale_statusdate) = ?
                     AND YEAR(sale_statusdate) = ?
                     AND sale_prodanausluga IN ($placeholders)";
            
            $params = array_merge([$agentName, (int)$month, (int)$year], $p2pTarife);
            
            $count = (int)$this->dbConnection->querySingleValue($query, $params);
            
            return $count >= $minimum;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error checking minimum P2P sales for agent', [
                    'error' => $e->getMessage(),
                    'agent' => $agentName
                ]);
            }
            return false;
        }
    }

    /**
     * NOVA METODA: Get P2P tarife from prodaja_bonus_p2p table
     */
    private function getP2PTarife(): array
    {
        try {
            $query = "SELECT DISTINCT vrsta_tarifa FROM prodaja_bonus_p2p";
            $results = $this->dbConnection->queryAndFetchAllAssoc($query);
            
            return array_column($results, 'vrsta_tarifa');
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting P2P tarife', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Calculate points for specific agent based on their group
     */
    private function calculateAgentPoints(string $agentName, string $month, string $year, string $agentGroup): array
    {
        try {
            $totalPoints = 0;
            $serviceBreakdown = [];
            
            // Define which groups to include based on agent type
            $groupsToInclude = [];
            switch ($agentGroup) {
                case 'mnp':
                    $groupsToInclude = ['MNP HT', 'MNP Telemach', 'Parica', 'Optika', 'Tomato'];
                    break;
                case 'optika':
                    $groupsToInclude = ['Optika', 'MNP HT', 'MNP Telemach', 'Parica', 'Tomato'];
                    break;
                case 'p2p':
                    $groupsToInclude = ['P2P', 'Tomato P2P'];
                    break;
            }
            
            // Get sold services for this agent
            $soldServices = $this->getAgentSoldServices($agentName, $month, $year, $groupsToInclude);
            
            // Get tarifa points from database
            $tarifaPoints = $this->getTarifaPointsFromDB();
            
            foreach ($soldServices as $service) {
                $tarifa = $service['tarifa'];
                $count = (int)$service['count'];
                $points = $tarifaPoints[$tarifa] ?? 0;
                $servicePoints = $points * $count;
                
                $totalPoints += $servicePoints;
                $serviceBreakdown[] = [
                    'tarifa' => $tarifa,
                    'count' => $count,
                    'points_per_item' => $points,
                    'total_points' => $servicePoints
                ];
            }
            
            // Calculate ranking and bonus
            $ranking = $this->calculateRanking($totalPoints, $agentGroup);
            
            return [
                'agent_name' => $agentName,
                'agent_group' => $agentGroup,
                'total_points' => $totalPoints,
                'current_rank' => $ranking['rank'],
                'bonus_amount' => $ranking['bonus'],
                'services_sold' => count($serviceBreakdown),
                'service_breakdown' => $serviceBreakdown
            ];
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error calculating agent points', [
                    'error' => $e->getMessage(),
                    'agent' => $agentName
                ]);
            }
            return [
                'agent_name' => $agentName,
                'agent_group' => $agentGroup,
                'total_points' => 0,
                'current_rank' => 'Nema',
                'bonus_amount' => 0,
                'services_sold' => 0,
                'service_breakdown' => []
            ];
        }
    }

    /**
     * Get sold services for specific agent and groups
     */
    private function getAgentSoldServices(string $agentName, string $month, string $year, array $groups): array
    {
        try {
            $placeholders = str_repeat('?,', count($groups) - 1) . '?';
            
            $query = "SELECT 
                        p.sale_prodanausluga as tarifa,
                        COUNT(*) as count
                     FROM {$this->table} p
                     JOIN prodaja_vrste pv ON p.sale_prodanausluga = pv.tarifa
                     WHERE p.sale_status = 'Aktivno'
                     AND p.sale_djelatnik = ?
                     AND MONTH(p.sale_statusdate) = ?
                     AND YEAR(p.sale_statusdate) = ?
                     AND pv.grupa IN ($placeholders)
                     GROUP BY p.sale_prodanausluga";
            
            $params = array_merge([$agentName, (int)$month, (int)$year], $groups);
            
            return $this->dbConnection->queryAndFetchAllAssoc($query, $params);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting agent sold services', [
                    'error' => $e->getMessage(),
                    'agent' => $agentName
                ]);
            }
            return [];
        }
    }

    /**
     * Calculate ranking and bonus based on total points
     */
    private function calculateRanking(int $totalPoints, string $agentGroup): array
    {
        $rankings = $this->rankingSystems[$agentGroup] ?? [];
        
        $currentRank = 'Nema';
        $bonusAmount = 0;
        
        // Find highest achieved rank
        foreach ($rankings as $rank) {
            if ($totalPoints >= $rank['points']) {
                $currentRank = $rank['name'];
                $bonusAmount = $rank['bonus'];
            }
        }
        
        return [
            'rank' => $currentRank,
            'bonus' => $bonusAmount
        ];
    }

    /**
     * Get total points for all groups
     */
    public function getTotalPointsAllGroups(string $month, string $year): int
    {
        try {
            $totalPoints = 0;
            
            $mnpStats = $this->getMNPAgentsStatistics($month, $year);
            $optikaStats = $this->getOptikaAgentsStatistics($month, $year);
            $p2pStats = $this->getP2PAgentsStatistics($month, $year);
            
            foreach ($mnpStats as $agent) {
                $totalPoints += $agent['total_points'];
            }
            
            foreach ($optikaStats as $agent) {
                $totalPoints += $agent['total_points'];
            }
            
            foreach ($p2pStats as $agent) {
                $totalPoints += $agent['total_points'];
            }
            
            return $totalPoints;
        } catch (\Exception $e) {
            if ($this->logger) {
                $this->logger->error('Error getting total points all groups', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }

    /**
     * Get total bonus amount for all groups
     */
    public function getTotalBonusAllGroups(string $month, string $year): float
    {
        try {
            $totalBonus = 0;
            
            $mnpStats = $this->getMNPAgentsStatistics($month, $year);
            $optikaStats = $this->getOptikaAgentsStatistics($month, $year);
            $p2pStats = $this->getP2PAgentsStatistics($month, $year);
            
            foreach ($mnpStats as $agent) {
                $totalBonus += $agent['bonus_amount'];
            }
            
            foreach ($optikaStats as $agent) {
                $totalBonus += $agent['bonus_amount'];
            }
            
            foreach ($p2pStats as $agent) {
                $totalBonus += $agent['bonus_amount'];
            }
            
            return $totalBonus;
        } catch (\Exception $e) {
            if ($this->logger) {
                $this->logger->error('Error getting total bonus all groups', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }

    /**
     * Get count of active agents (agents with points > 0)
     */
    public function getActiveAgentsCount(string $month, string $year): int
    {
        try {
            $count = 0;
            
            $mnpStats = $this->getMNPAgentsStatistics($month, $year);
            $optikaStats = $this->getOptikaAgentsStatistics($month, $year);
            $p2pStats = $this->getP2PAgentsStatistics($month, $year);
            
            $count += count($mnpStats);
            $count += count($optikaStats);
            $count += count($p2pStats);
            
            return $count;
        } catch (\Exception $e) {
            if ($this->logger) {
                $this->logger->error('Error getting active agents count', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }

    /**
     * Get top performing agents across all groups
     */
    public function getTopPerformers(string $month, string $year, int $limit = 10): array
    {
        try {
            $allAgents = [];
            
            $mnpStats = $this->getMNPAgentsStatistics($month, $year);
            $optikaStats = $this->getOptikaAgentsStatistics($month, $year);
            $p2pStats = $this->getP2PAgentsStatistics($month, $year);
            
            $allAgents = array_merge($allAgents, $mnpStats, $optikaStats, $p2pStats);
            
            // Sort by total points descending
            usort($allAgents, function($a, $b) {
                return $b['total_points'] <=> $a['total_points'];
            });
            
            return array_slice($allAgents, 0, $limit);
        } catch (\Exception $e) {
            if ($this->logger) {
                $this->logger->error('Error getting top performers', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Get monthly points trend for charts
     */
    public function getMonthlyPointsTrend(int $monthsBack = 6): array
    {
        $results = [];
        
        for ($i = $monthsBack - 1; $i >= 0; $i--) {
            $date = new \DateTime();
            $date->modify("-$i months");
            $month = $date->format('m');
            $year = $date->format('Y');
            
            $totalPoints = $this->getTotalPointsAllGroups($month, $year);
            $totalBonus = $this->getTotalBonusAllGroups($month, $year);
            
            $results[] = [
                'month' => (int)$month,
                'year' => (int)$year,
                'period' => $date->format('Y-m'),
                'total_points' => $totalPoints,
                'total_bonus' => $totalBonus
            ];
        }
        
        return $results;
    }

    /**
     * Get group points distribution for charts
     */
    public function getGroupPointsDistribution(string $month, string $year): array
    {
        try {
            $distribution = [];
            
            $mnpStats = $this->getMNPAgentsStatistics($month, $year);
            $optikaStats = $this->getOptikaAgentsStatistics($month, $year);
            $p2pStats = $this->getP2PAgentsStatistics($month, $year);
            
            $mnpPoints = array_sum(array_column($mnpStats, 'total_points'));
            $optikaPoints = array_sum(array_column($optikaStats, 'total_points'));
            $p2pPoints = array_sum(array_column($p2pStats, 'total_points'));
            
            $distribution[] = ['group' => 'MNP', 'points' => $mnpPoints, 'agents' => count($mnpStats)];
            $distribution[] = ['group' => 'Optika', 'points' => $optikaPoints, 'agents' => count($optikaStats)];
            $distribution[] = ['group' => 'P2P', 'points' => $p2pPoints, 'agents' => count($p2pStats)];
            
            return $distribution;
        } catch (\Exception $e) {
            if ($this->logger) {
                $this->logger->error('Error getting group points distribution', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Get all agents with their rankings for detailed reports
     */
    public function getAllAgentsWithRankings(string $month, string $year): array
    {
        try {
            $allAgents = [];
            
            $mnpStats = $this->getMNPAgentsStatistics($month, $year);
            $optikaStats = $this->getOptikaAgentsStatistics($month, $year);
            $p2pStats = $this->getP2PAgentsStatistics($month, $year);
            
            $allAgents = array_merge($allAgents, $mnpStats, $optikaStats, $p2pStats);
            
            // Sort by total points descending
            usort($allAgents, function($a, $b) {
                return $b['total_points'] <=> $a['total_points'];
            });
            
            return $allAgents;
        } catch (\Exception $e) {
            if ($this->logger) {
                $this->logger->error('Error getting all agents with rankings', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Get ranking system for specific agent group
     */
    public function getRankingSystem(string $agentGroup): array
    {
        return $this->rankingSystems[$agentGroup] ?? [];
    }

    /**
     * Get tariff points mapping from database
     */
    public function getTarifaPoints(): array
    {
        return $this->getTarifaPointsFromDB();
    }

    /**
     * Get tariff points from database instead of hardcoded array
     */
    private function getTarifaPointsFromDB(): array
    {
        try {
            $query = "SELECT tarifa, bodovi FROM prodaja_vrste WHERE bodovi > 0";
            $results = $this->dbConnection->queryAndFetchAllAssoc($query);
            
            $tarifaPoints = [];
            foreach ($results as $row) {
                $tarifaPoints[$row['tarifa']] = (int)$row['bodovi'];
            }
            
            return $tarifaPoints;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting tarifa points from DB', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Get agent's organizational unit/group
     */
    public function getAgentGroup(string $agentName): ?string
    {
        try {
            $query = "SELECT orgjedinica FROM prodaja_osobe WHERE name = ?";
            $orgUnit = $this->dbConnection->querySingleValue($query, [$agentName]);
            
            if (!$orgUnit) {
                return null;
            }
            
            // Map organizational unit to agent group
            switch ($orgUnit) {
                case 'Backoffice MNP':
                    return 'mnp';
                case 'Backoffice Optika':
                    return 'optika';
                case 'Backoffice P2P':
                    return 'p2p';
                default:
                    return null;
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting agent group', [
                    'error' => $e->getMessage(),
                    'agent' => $agentName
                ]);
            }
            return null;
        }
    }

    /**
     * Get detailed statistics for specific agent
     */
    public function getAgentDetailedStats(string $agentName, string $month, string $year, string $agentGroup): array
    {
        try {
            $agentStats = $this->calculateAgentPoints($agentName, $month, $year, $agentGroup);
            
            // Add additional details like monthly comparison
            $previousMonth = $month == '01' ? '12' : str_pad((int)$month - 1, 2, '0', STR_PAD_LEFT);
            $previousYear = $month == '01' ? (string)((int)$year - 1) : $year;
            
            $previousMonthStats = $this->calculateAgentPoints($agentName, $previousMonth, $previousYear, $agentGroup);
            
            $agentStats['previous_month_points'] = $previousMonthStats['total_points'];
            $agentStats['points_difference'] = $agentStats['total_points'] - $previousMonthStats['total_points'];
            $agentStats['previous_month_rank'] = $previousMonthStats['current_rank'];
            
            // Add progress to next rank
            $nextRank = $this->getNextRankRequirements($agentStats['total_points'], $agentGroup);
            $agentStats['next_rank_info'] = $nextRank;
            
            // Get detailed service breakdown with groups
            $agentStats['service_breakdown'] = $this->getAgentServiceBreakdown($agentName, $month, $year);
            
            return $agentStats;
        } catch (\Exception $e) {
            if ($this->logger) {
                $this->logger->error('Error getting agent detailed stats', [
                    'error' => $e->getMessage(),
                    'agent' => $agentName
                ]);
            }
            return [];
        }
    }

    /**
     * Get agent's historical performance over last N months
     */
    public function getAgentHistoricalPerformance(string $agentName, int $monthsBack = 6): array
    {
        try {
            $results = [];
            $agentGroup = $this->getAgentGroup($agentName);
            
            if (!$agentGroup) {
                return [];
            }
            
            for ($i = $monthsBack - 1; $i >= 0; $i--) {
                $date = new \DateTime();
                $date->modify("-$i months");
                $month = $date->format('m');
                $year = $date->format('Y');
                
                $stats = $this->calculateAgentPoints($agentName, $month, $year, $agentGroup);
                
                $results[] = [
                    'month' => (int)$month,
                    'year' => (int)$year,
                    'period' => $date->format('Y-m'),
                    'month_name' => $date->format('M'),
                    'total_points' => $stats['total_points'],
                    'rank' => $stats['current_rank'],
                    'bonus_amount' => $stats['bonus_amount'],
                    'services_sold' => $stats['services_sold']
                ];
            }
            
            return $results;
        } catch (\Exception $e) {
            if ($this->logger) {
                $this->logger->error('Error getting agent historical performance', [
                    'error' => $e->getMessage(),
                    'agent' => $agentName
                ]);
            }
            return [];
        }
    }

    /**
     * Get agent's ranking within their group
     */
    public function getAgentGroupRanking(string $agentName, string $month, string $year, string $agentGroup): array
    {
        try {
            $groupStats = [];
            
            switch ($agentGroup) {
                case 'mnp':
                    $groupStats = $this->getMNPAgentsStatistics($month, $year);
                    break;
                case 'optika':
                    $groupStats = $this->getOptikaAgentsStatistics($month, $year);
                    break;
                case 'p2p':
                    $groupStats = $this->getP2PAgentsStatistics($month, $year);
                    break;
            }
            
            // Find agent's position
            $agentPosition = 0;
            $totalAgents = count($groupStats);
            $agentStats = null;
            
            foreach ($groupStats as $index => $agent) {
                if ($agent['agent_name'] === $agentName) {
                    $agentPosition = $index + 1;
                    $agentStats = $agent;
                    break;
                }
            }
            
            return [
                'position' => $agentPosition,
                'total_agents' => $totalAgents,
                'agent_stats' => $agentStats,
                'group_stats' => $groupStats,
                'top_3' => array_slice($groupStats, 0, 3)
            ];
        } catch (\Exception $e) {
            if ($this->logger) {
                $this->logger->error('Error getting agent group ranking', [
                    'error' => $e->getMessage(),
                    'agent' => $agentName
                ]);
            }
            return [];
        }
    }

    /**
     * Get requirements for next rank
     */
    public function getNextRankRequirements(int $currentPoints, string $agentGroup): array
    {
        $rankings = $this->rankingSystems[$agentGroup] ?? [];
        
        $nextRank = null;
        $pointsNeeded = 0;
        $currentRankIndex = -1;
        
        // Find current rank index
        foreach ($rankings as $index => $rank) {
            if ($currentPoints >= $rank['points']) {
                $currentRankIndex = $index;
            }
        }
        
        // Find next rank
        if ($currentRankIndex < count($rankings) - 1) {
            $nextRank = $rankings[$currentRankIndex + 1];
            $pointsNeeded = $nextRank['points'] - $currentPoints;
        }
        
        return [
            'next_rank' => $nextRank,
            'points_needed' => max(0, $pointsNeeded),
            'current_rank_index' => $currentRankIndex,
            'total_ranks' => count($rankings),
            'progress_percentage' => $nextRank ? min(100, ($currentPoints / $nextRank['points']) * 100) : 100
        ];
    }

    /**
     * Get agent's service breakdown with points
     */
    public function getAgentServiceBreakdown(string $agentName, string $month, string $year): array
    {
        try {
            $query = "SELECT 
                        p.sale_prodanausluga as tarifa,
                        pv.grupa,
                        pv.bodovi,
                        COUNT(*) as count,
                        (COUNT(*) * pv.bodovi) as total_points
                    FROM {$this->table} p
                    JOIN prodaja_vrste pv ON p.sale_prodanausluga = pv.tarifa
                    WHERE p.sale_status = 'Aktivno'
                    AND p.sale_djelatnik = ?
                    AND MONTH(p.sale_statusdate) = ?
                    AND YEAR(p.sale_statusdate) = ?
                    AND pv.bodovi > 0
                    GROUP BY p.sale_prodanausluga, pv.grupa, pv.bodovi
                    ORDER BY total_points DESC";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query, [
                $agentName, (int)$month, (int)$year
            ]);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting agent service breakdown', [
                    'error' => $e->getMessage(),
                    'agent' => $agentName
                ]);
            }
            return [];
        }
    }
}