<?php

namespace sales\Models;

use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

/**
 * DatabaseTableBonus - Specialized model for managing bonus tables
 * AŽURIRANO za novu strukturu tablica
 */
class DatabaseTableBonus extends DatabaseTable
{
    /**
     * @var \baseKRIZAN\Database\DatabaseConnection
     */
    protected $dbConnection;
    
    /**
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Constructor for Bonus tables management
     * 
     * @param DatabaseConnection $dbConnection Database connection
     * @param string $table Table name (will be overridden by type)
     * @param string $primaryKey Primary key
     * @param string $className Class name
     * @param array $constructorArgs Constructor arguments
     * @param Logger|null $logger Logger instance
     */
    public function __construct(
        DatabaseConnection $dbConnection, 
        string $table, 
        string $primaryKey, 
        string $className = 'stdClass', 
        array $constructorArgs = [],
        ?Logger $logger = null
    ) {
        parent::__construct($dbConnection, $table, $primaryKey, $className, $constructorArgs);
        $this->dbConnection = $dbConnection;
        $this->logger = $logger;
        
        // Ensure all bonus tables exist
        $this->ensureBonusTablesExist();
        
        if ($this->logger) {
            $this->logger->modules('DatabaseTableBonus initialized', [
                'table' => $table
            ]);
        }
    }
    
    /**
     * Ensure all bonus tables exist - AŽURIRANO
     * 
     * @return void
     */
    private function ensureBonusTablesExist(): void
    {
        $this->ensureBonusPrijenosTableExists();
        $this->ensureBonusP2pTableExists();
        $this->ensureBonusTomatoTableExists();
    }
    
    /**
     * Ensure prodaja_bonus_prijenos table exists - AŽURIRANO
     * 
     * @return void
     */
    private function ensureBonusPrijenosTableExists(): void
    {
        try {
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'prodaja_bonus_prijenos' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                $sql = "CREATE TABLE `prodaja_bonus_prijenos` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `vrsta_tarifa` varchar(100) NOT NULL,
                    `grupa` varchar(50) NOT NULL,
                    `osnovna_naknada` decimal(10,2) NOT NULL,
                    `rang_249` decimal(10,2) NOT NULL,
                    `rang_250_449` decimal(10,2) NOT NULL,
                    `rang_450_749` decimal(10,2) NOT NULL,
                    `rang_750_1049` decimal(10,2) NOT NULL,
                    `rang_70_80` decimal(10,2) NOT NULL,
                    `rang_80_90` decimal(10,2) NOT NULL,
                    `rang_90_120` decimal(10,2) NOT NULL,
                    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
                    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    PRIMARY KEY (`id`),
                    KEY `idx_vrsta_tarifa` (`vrsta_tarifa`),
                    KEY `idx_grupa` (`grupa`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created prodaja_bonus_prijenos table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error creating prodaja_bonus_prijenos table', [
                    'error' => $e->getMessage()
                ]);
            }
            throw $e;
        }
    }
    
    /**
     * Ensure prodaja_bonus_p2p table exists
     * 
     * @return void
     */
    private function ensureBonusP2pTableExists(): void
    {
        try {
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'prodaja_bonus_p2p' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                $sql = "CREATE TABLE `prodaja_bonus_p2p` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `vrsta_tarifa` varchar(100) NOT NULL,
                    `grupa` varchar(50) NOT NULL,
                    `osnovna_naknada` decimal(10,2) NOT NULL,
                    `rang_150` decimal(10,2) NOT NULL,
                    `rang_150_400` decimal(10,2) NOT NULL,
                    `rang_400_650` decimal(10,2) NOT NULL,
                    `rang_650_plus` decimal(10,2) NOT NULL,
                    `rang_70_80` decimal(10,2) NOT NULL,
                    `rang_80_90` decimal(10,2) NOT NULL,
                    `rang_90_120` decimal(10,2) NOT NULL,
                    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
                    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    PRIMARY KEY (`id`),
                    KEY `idx_vrsta_tarifa` (`vrsta_tarifa`),
                    KEY `idx_grupa` (`grupa`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created prodaja_bonus_p2p table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error creating prodaja_bonus_p2p table', [
                    'error' => $e->getMessage()
                ]);
            }
            throw $e;
        }
    }
    
    /**
     * Ensure prodaja_bonus_tomato table exists
     * 
     * @return void
     */
    private function ensureBonusTomatoTableExists(): void
    {
        try {
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'prodaja_bonus_tomato' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                $sql = "CREATE TABLE `prodaja_bonus_tomato` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `vrsta_tarifa` varchar(100) NOT NULL,
                    `grupa` varchar(50) NOT NULL,
                    `osnovna_naknada` decimal(10,2) NOT NULL,
                    `rang_299` decimal(10,2) NULL,
                    `rang_300_549` decimal(10,2) NULL,
                    `rang_550_779` decimal(10,2) NULL,
                    `rang_800_1099` decimal(10,2) NULL,
                    `rang_70_80` decimal(10,2) NOT NULL,
                    `rang_80_90` decimal(10,2) NOT NULL,
                    `rang_90_120` decimal(10,2) NOT NULL,
                    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
                    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    PRIMARY KEY (`id`),
                    KEY `idx_vrsta_tarifa` (`vrsta_tarifa`),
                    KEY `idx_grupa` (`grupa`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created prodaja_bonus_tomato table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error creating prodaja_bonus_tomato table', [
                    'error' => $e->getMessage()
                ]);
            }
            throw $e;
        }
    }
    
    /**
     * Get table name by bonus type - AŽURIRANO
     * 
     * @param string $type
     * @return string
     */
    public function getTableByType(string $type): string
    {
        $tables = [
            'prijenos' => 'prodaja_bonus_prijenos',
            'p2p' => 'prodaja_bonus_p2p',
            'tomato' => 'prodaja_bonus_tomato'
        ];
        
        return $tables[$type] ?? 'prodaja_bonus_prijenos';
    }
    
    /**
     * Override table name for specific bonus type operations
     * 
     * @param string $type
     * @return void
     */
    public function setTableForType(string $type): void
    {
        $this->table = $this->getTableByType($type);
    }
    
    /**
     * Get all bonus data with enhanced analytics
     * 
     * @param string $type
     * @return array
     */
    public function getBonusAnalytics(string $type): array
    {
        $this->setTableForType($type);
        
        try {
            $sql = "SELECT 
                        COUNT(*) as total_schemes,
                        AVG(osnovna_naknada) as avg_base_amount,
                        MAX(osnovna_naknada) as max_base_amount,
                        MIN(osnovna_naknada) as min_base_amount,
                        COUNT(DISTINCT grupa) as group_count
                    FROM {$this->table}";
            
            $result = $this->dbConnection->queryAndFetchAllAssoc($sql);
            return $result ? $result[0] : [];
            
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting bonus analytics', [
                    'error' => $e->getMessage(),
                    'type' => $type
                ]);
            }
            return [];
        }
    }
    
    /**
     * Get bonus schemes grouped by type - AŽURIRANO
     * 
     * @return array
     */
    public function getAllBonusSchemes(): array
    {
        $types = ['prijenos', 'p2p', 'tomato'];
        $result = [];
        
        foreach ($types as $type) {
            $this->setTableForType($type);
            $result[$type] = $this->findAll();
        }
        
        return $result;
    }
    
    /**
     * Get bonus records by type (override parent findAll for dynamic table)
     * 
     * @param string $type
     * @return array
     */
    public function findAllByType(string $type): array
    {
        $this->setTableForType($type);
        return $this->findAll();
    }
    
    /**
     * Find bonus by ID and type
     * 
     * @param string $type
     * @param int $id
     * @return object|null
     */
    public function findByIdAndType(string $type, int $id): ?object
    {
        $this->setTableForType($type);
        return $this->findById($id);
    }
    
    /**
     * Save bonus record for specific type
     * 
     * @param string $type
     * @param array $record
     * @return object
     */
    public function saveByType(string $type, array $record): object
    {
        $this->setTableForType($type);
        return $this->save($record);
    }
    
    /**
     * Delete bonus record by type and ID
     * 
     * @param string $type
     * @param int $id
     * @return void
     */
    public function deleteByType(string $type, int $id): void
    {
        $this->setTableForType($type);
        $this->delete($id);
    }
}