<?php

namespace sales\Models;

use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

class DatabaseTableFeedback extends DatabaseTable
{
    /**
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Constructor for Feedback Management
     */
    public function __construct(
        DatabaseConnection $dbConnection, 
        string $table, 
        string $primaryKey, 
        string $className = 'stdClass', 
        array $constructorArgs = [],
        ?Logger $logger = null
    ) {
        parent::__construct($dbConnection, $table, $primaryKey, $className, $constructorArgs);
        $this->logger = $logger;
        
        if ($this->logger) {
            $this->logger->modules('DatabaseTableFeedback initialized', [
                'table' => $table
            ]);
        }
    }

    /**
     * Get latest feedback for a person
     */
    public function getLatestFeedback(string $personName): ?object
    {
        try {
            $query = "SELECT * FROM {$this->table} 
                     WHERE name = ? 
                     AND (feedback IS NOT NULL OR ocjena IS NOT NULL)
                     ORDER BY datum DESC 
                     LIMIT 1";
            
            $result = $this->dbConnection->queryAndFetchAllAssoc($query, [$personName]);
            
            return !empty($result) ? (object)$result[0] : null;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting latest feedback', [
                    'error' => $e->getMessage(),
                    'person' => $personName
                ]);
            }
            return null;
        }
    }

    /**
     * Get all feedbacks for a person
     */
    public function getAllFeedbacks(string $personName): array
    {
        try {
            $query = "SELECT * FROM {$this->table} 
                     WHERE name = ? 
                     AND (feedback IS NOT NULL OR ocjena IS NOT NULL)
                     ORDER BY datum DESC";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query, [$personName]);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting all feedbacks', [
                    'error' => $e->getMessage(),
                    'person' => $personName
                ]);
            }
            return [];
        }
    }

    /**
     * Add new feedback
     */
    public function addFeedback(string $personName, ?string $feedback, ?float $ocjena, string $createdBy): bool
    {
        try {
            // Validate that at least one field is provided
            if (empty($feedback) && ($ocjena === null || $ocjena <= 0)) {
                throw new \InvalidArgumentException('Either feedback or ocjena must be provided');
            }

            $data = [
                'name' => $personName,
                'feedback' => $feedback,
                'ocjena' => $ocjena,
                'datum' => date('Y-m-d H:i:s'),
                'created_by' => $createdBy
            ];

            $this->save($data);
            
            if ($this->logger) {
                $this->logger->info('Feedback added successfully', [
                    'person' => $personName,
                    'created_by' => $createdBy,
                    'has_feedback' => !empty($feedback),
                    'has_ocjena' => $ocjena !== null
                ]);
            }
            
            return true;
        } catch (\Exception $e) {
            if ($this->logger) {
                $this->logger->error('Error adding feedback', [
                    'error' => $e->getMessage(),
                    'person' => $personName,
                    'created_by' => $createdBy
                ]);
            }
            return false;
        }
    }

    /**
     * Update existing feedback
     */
    public function updateFeedback(int $feedbackId, ?string $feedback, ?float $ocjena, string $updatedBy): bool
    {
        try {
            // Get existing feedback first
            $existingFeedback = $this->findById($feedbackId);
            if (!$existingFeedback) {
                throw new \RuntimeException('Feedback not found');
            }

            // Validate that at least one field is provided
            if (empty($feedback) && ($ocjena === null || $ocjena <= 0)) {
                throw new \InvalidArgumentException('Either feedback or ocjena must be provided');
            }

            $data = [
                'id' => $feedbackId,
                'name' => $existingFeedback->name, // Keep original name
                'feedback' => $feedback,
                'ocjena' => $ocjena,
                'datum' => $existingFeedback->datum, // Keep original date
                'created_by' => $existingFeedback->created_by, // Keep original creator
                'updated_at' => date('Y-m-d H:i:s')
            ];

            $this->save($data);
            
            if ($this->logger) {
                $this->logger->info('Feedback updated successfully', [
                    'feedback_id' => $feedbackId,
                    'person' => $existingFeedback->name,
                    'updated_by' => $updatedBy
                ]);
            }
            
            return true;
        } catch (\Exception $e) {
            if ($this->logger) {
                $this->logger->error('Error updating feedback', [
                    'error' => $e->getMessage(),
                    'feedback_id' => $feedbackId,
                    'updated_by' => $updatedBy
                ]);
            }
            return false;
        }
    }

    /**
     * Delete feedback
     */
    public function deleteFeedback(int $feedbackId, string $deletedBy): bool
    {
        try {
            // Get feedback info before deleting for logging
            $feedback = $this->findById($feedbackId);
            if (!$feedback) {
                throw new \RuntimeException('Feedback not found');
            }

            $this->delete($feedbackId);
            
            if ($this->logger) {
                $this->logger->info('Feedback deleted successfully', [
                    'feedback_id' => $feedbackId,
                    'person' => $feedback->name,
                    'deleted_by' => $deletedBy
                ]);
            }
            
            return true;
        } catch (\Exception $e) {
            if ($this->logger) {
                $this->logger->error('Error deleting feedback', [
                    'error' => $e->getMessage(),
                    'feedback_id' => $feedbackId,
                    'deleted_by' => $deletedBy
                ]);
            }
            return false;
        }
    }

    /**
     * Get feedback statistics for a person
     */
    public function getFeedbackStats(string $personName): array
    {
        try {
            $query = "SELECT 
                        COUNT(*) as total_feedbacks,
                        AVG(CASE WHEN ocjena IS NOT NULL THEN ocjena END) as avg_ocjena,
                        MAX(datum) as latest_feedback_date,
                        MIN(datum) as first_feedback_date,
                        COUNT(CASE WHEN ocjena IS NOT NULL THEN 1 END) as total_ratings
                     FROM {$this->table} 
                     WHERE name = ? 
                     AND (feedback IS NOT NULL OR ocjena IS NOT NULL)";
            
            $result = $this->dbConnection->queryAndFetchAllAssoc($query, [$personName]);
            
            if (!empty($result)) {
                $stats = $result[0];
                // Convert numeric values
                $stats['total_feedbacks'] = (int)$stats['total_feedbacks'];
                $stats['total_ratings'] = (int)$stats['total_ratings'];
                $stats['avg_ocjena'] = $stats['avg_ocjena'] ? round((float)$stats['avg_ocjena'], 2) : null;
                
                return $stats;
            }
            
            return [
                'total_feedbacks' => 0,
                'avg_ocjena' => null,
                'latest_feedback_date' => null,
                'first_feedback_date' => null,
                'total_ratings' => 0
            ];
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting feedback stats', [
                    'error' => $e->getMessage(),
                    'person' => $personName
                ]);
            }
            return [];
        }
    }

    /**
     * Get all people with their latest feedback info
     */
    public function getAllPeopleWithLatestFeedback(): array
    {
        try {
            $query = "SELECT 
                        po.id,
                        po.name,
                        po.radnomjesto,
                        po.area,
                        po.orgjedinica,
                        po.vidljivost,
                        lf.feedback,
                        lf.ocjena,
                        lf.datum,
                        lf.created_by,
                        fs.total_feedbacks,
                        fs.avg_ocjena
                     FROM prodaja_osobe po
                     LEFT JOIN (
                         SELECT name, feedback, ocjena, datum, created_by,
                                ROW_NUMBER() OVER (PARTITION BY name ORDER BY datum DESC) as rn
                         FROM {$this->table}
                         WHERE feedback IS NOT NULL OR ocjena IS NOT NULL
                     ) lf ON po.name = lf.name AND lf.rn = 1
                     LEFT JOIN (
                         SELECT name,
                                COUNT(*) as total_feedbacks,
                                AVG(CASE WHEN ocjena IS NOT NULL THEN ocjena END) as avg_ocjena
                         FROM {$this->table}
                         WHERE feedback IS NOT NULL OR ocjena IS NOT NULL
                         GROUP BY name
                     ) fs ON po.name = fs.name
                     ORDER BY po.name";
            
            $results = $this->dbConnection->queryAndFetchAllAssoc($query);
            
            // Process results
            foreach ($results as &$result) {
                $result['total_feedbacks'] = (int)($result['total_feedbacks'] ?? 0);
                $result['avg_ocjena'] = $result['avg_ocjena'] ? round((float)$result['avg_ocjena'], 2) : null;
                
                // Ensure all required fields have default values
                $result['id'] = $result['id'] ?? null;
                $result['radnomjesto'] = $result['radnomjesto'] ?? '';
                $result['area'] = $result['area'] ?? '';
                $result['orgjedinica'] = $result['orgjedinica'] ?? '';
                $result['vidljivost'] = $result['vidljivost'] ?? '0';
            }
            
            return $results;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting all people with latest feedback', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Check if user can manage feedback (admin/supervisor only)
     */
    public function canManageFeedback(object $user): bool
    {
        if (!$user || !method_exists($user, 'hasPermission')) {
            return false;
        }

        return $user->hasPermission(\Models\Entity\Korisnik::KORISNIK_sales_administrator) ||
               $user->hasPermission(\Models\Entity\Korisnik::KORISNIK_sales_supervisor);
    }

    /**
     * Get recent feedback activity (for dashboard)
     */
    public function getRecentFeedbackActivity(int $limit = 10): array
    {
        try {
            $query = "SELECT name, feedback, ocjena, datum, created_by
                     FROM {$this->table}
                     WHERE feedback IS NOT NULL OR ocjena IS NOT NULL
                     ORDER BY datum DESC
                     LIMIT ?";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query, [$limit]);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting recent feedback activity', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }
}