<?php
// ============================================================================
// 4. DatabaseTableSales.php - Očišćeni osnovni model
// ============================================================================

namespace sales\Models;

use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

class DatabaseTableSales extends DatabaseTable
{
    /**
     * @var \baseKRIZAN\Database\DatabaseConnection
     */
    protected $dbConnection;
    
    /**
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Constructor for basic Sales CRUD operations
     * 
     * @param DatabaseConnection $dbConnection Database connection
     * @param string $table Table name
     * @param string $primaryKey Primary key
     * @param string $className Class name
     * @param array $constructorArgs Constructor arguments
     * @param Logger|null $logger Logger instance
     */
    public function __construct(
        DatabaseConnection $dbConnection, 
        string $table, 
        string $primaryKey, 
        string $className = 'stdClass', 
        array $constructorArgs = [],
        ?Logger $logger = null
    ) {
        parent::__construct($dbConnection, $table, $primaryKey, $className, $constructorArgs);
        $this->dbConnection = $dbConnection;
        $this->logger = $logger;
        
        // Ensure all required tables exist
        $this->ensureTablesExist();
        
        if ($this->logger) {
            $this->logger->modules('DatabaseTableSales initialized', [
                'table' => $table
            ]);
        }
    }
    
    /**
     * Ensure all required tables exist
     * 
     * @return void
     */
    private function ensureTablesExist(): void
    {
        $this->ensureProdajaTableExists();
        $this->ensureProdajaOsobeTableExists();
        $this->ensureProdajaOsobaTableExists(); // Nova tablica
        $this->ensureProdajaVrsteTableExists();
        $this->ensureProdajaTargetTableExists();
    }
    
    /**
     * Ensure prodaja (main sales) table exists
     * 
     * @return void
     */
    private function ensureProdajaTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'prodaja' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                // Create table
                $sql = "CREATE TABLE `prodaja` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `user_id` int(11) NOT NULL,
                    `sale_date` datetime NOT NULL,
                    `sale_imeiprezime` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `sale_oib` bigint(20) DEFAULT NULL,
                    `sale_brojoi` int(11) DEFAULT NULL,
                    `sale_datumrodj` date DEFAULT NULL,
                    `sale_adresa` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `sale_brojzaprijenos` bigint(20) DEFAULT NULL,
                    `sale_maticnioperator` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `sale_kontaktbroj` bigint(20) DEFAULT NULL,
                    `sale_mail` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `sale_nacinprimanja` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `sale_brojzap2p` bigint(20) DEFAULT NULL,
                    `sale_prodanausluga` varchar(40) COLLATE utf8_general_ci DEFAULT NULL,
                    `sale_tvusluga` varchar(2) COLLATE utf8_general_ci DEFAULT NULL,
                    `sale_kucnitel` varchar(2) COLLATE utf8_general_ci DEFAULT NULL,
                    `sale_uredajnavesti` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `sale_brojnarudzbe` bigint(20) DEFAULT NULL,
                    `sale_promocija` varchar(50) COLLATE utf8_general_ci DEFAULT NULL,
                    `sale_oibzamozaik` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `sale_djelatnik` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `sale_kontrolor` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `sale_kontrolnipoziv` varchar(2) COLLATE utf8_general_ci DEFAULT NULL,
                    `sale_status` varchar(20) COLLATE utf8_general_ci DEFAULT NULL,
                    `sale_statusdate` datetime DEFAULT NULL,
                    `sale_comment` text COLLATE utf8_general_ci DEFAULT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created prodaja table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureProdajaTableExists'
                ]);
            }
            throw $e;
        }
    }
    
    /**
     * Ensure prodaja_osobe table exists (stara struktura)
     * 
     * @return void
     */
    private function ensureProdajaOsobeTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'prodaja_osobe' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                // Create table s ispravnom strukturom prema screenshotu
                $sql = "CREATE TABLE `prodaja_osobe` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `name` varchar(255) COLLATE utf8_general_ci NOT NULL,
                    `radnomjesto` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `area` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `orgjednica` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `vidljivost` int(11) NOT NULL DEFAULT 1,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created prodaja_osobe table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureProdajaOsobeTableExists'
                ]);
            }
            throw $e;
        }
    }
    
    /**
     * Ensure prodaja_osoba table exists (nova tablica prema drugom screenshotu)
     * 
     * @return void
     */
    private function ensureProdajaOsobaTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'prodaja_osoba' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                // Create table prema strukturi iz drugog screenshota
                $sql = "CREATE TABLE `prodaja_osoba` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `created_by` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `datum` date DEFAULT NULL,
                    `updated_at` datetime DEFAULT NULL,
                    `name` varchar(255) COLLATE utf8_general_ci NOT NULL,
                    `feedback` text COLLATE utf8_general_ci DEFAULT NULL,
                    `ocjena` decimal(3,2) DEFAULT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created prodaja_osoba table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureProdajaOsobaTableExists'
                ]);
            }
            throw $e;
        }
    }
    
    /**
     * Ensure prodaja_vrsta table exists
     * 
     * @return void
     */
    private function ensureProdajaVrsteTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'prodaja_vrste' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                // Create table
                $sql = "CREATE TABLE `prodaja_vrste` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `BRAND` varchar(6) COLLATE utf8_general_ci DEFAULT NULL,
                    `TARIFA` varchar(32) COLLATE utf8_general_ci DEFAULT NULL,
                    `CLUSTER` varchar(9) COLLATE utf8_general_ci DEFAULT NULL,
                    `NAGRADA_OSNOVNA` int(11) DEFAULT NULL,
                    `UGOVORNA_OBVEZA` int(11) DEFAULT NULL,
                    `NAGRADA_KVALITETA` int(11) DEFAULT NULL,
                    `NAGRADA_KLJU_NI_PROIZVOD` int(11) DEFAULT NULL,
                    `NAGRADA_TV` int(11) DEFAULT NULL,
                    `NAGRADA_TELEFON` int(11) DEFAULT NULL,
                    `VRSTA_IFRA` varchar(30) COLLATE utf8_general_ci DEFAULT NULL,
                    `VRSTA_OSNOVNI` decimal(2,1) DEFAULT NULL,
                    `VRSTA_FOKUS` int(11) DEFAULT NULL,
                    `NAGRADA_FOKUS` int(11) DEFAULT NULL,
                    `AKTUALNOST_PROIZVODA` varchar(8) COLLATE utf8_general_ci DEFAULT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created prodaja_vrste table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureProdajaVrsteTableExists'
                ]);
            }
            throw $e;
        }
    }
    
    /**
     * Ensure prodaja_target table exists
     * 
     * @return void
     */
    private function ensureProdajaTargetTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'prodaja_target' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                // Create table
                $sql = "CREATE TABLE `prodaja_target` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `br_o_nakada` int(11) DEFAULT NULL,
                    `osn_target` decimal(10,2) DEFAULT NULL,
                    `postotak_osn_targeta` decimal(5,2) DEFAULT NULL,
                    `osn_cijena` decimal(10,2) DEFAULT NULL,
                    `br_f_nakada` int(11) DEFAULT NULL,
                    `fok_target` decimal(10,2) DEFAULT NULL,
                    `postotak_fok_targeta` decimal(5,2) DEFAULT NULL,
                    `fok_cijena` decimal(10,2) DEFAULT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created prodaja_target table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureProdajaTargetTableExists'
                ]);
            }
            throw $e;
        }
    }
}