<?php
// ============================================================================
// 1. DatabaseTableSalesAdmin.php - Admin Analytics Model
// ============================================================================

namespace sales\Models;

use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

class DatabaseTableSalesAdmin extends DatabaseTable
{
    /**
     * @var \baseKRIZAN\Database\DatabaseConnection
     */
    protected $dbConnection;
    
    /**
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Constructor for Sales Admin Analytics
     * 
     * @param DatabaseConnection $dbConnection Database connection
     * @param string $table Table name
     * @param string $primaryKey Primary key
     * @param string $className Class name
     * @param array $constructorArgs Constructor arguments
     * @param Logger|null $logger Logger instance
     */
    public function __construct(
        DatabaseConnection $dbConnection, 
        string $table, 
        string $primaryKey, 
        string $className = 'stdClass', 
        array $constructorArgs = [],
        ?Logger $logger = null
    ) {
        parent::__construct($dbConnection, $table, $primaryKey, $className, $constructorArgs);
        $this->dbConnection = $dbConnection;
        $this->logger = $logger;
        
        if ($this->logger) {
            $this->logger->modules('DatabaseTableSalesAdmin initialized', [
                'table' => $table
            ]);
        }
    }

    /**
     * ===================================================================
     * REAL DATA ADMIN DASHBOARD METHODS
     * ===================================================================
     */

    /**
     * Get total sales count
     */
    public function getTotalSalesCount(): int
    {
        try {
            $query = "SELECT COUNT(*) as count FROM {$this->table}";
            return (int)$this->dbConnection->querySingleValue($query) ?? 0;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getTotalSalesCount', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }

    /**
     * Get active sales count (status = 'Aktivno')
     */
    public function getActiveSalesCount(): int
    {
        try {
            $query = "SELECT COUNT(*) as count FROM {$this->table} WHERE sale_status = 'Aktivno'";
            return (int)$this->dbConnection->querySingleValue($query) ?? 0;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getActiveSalesCount', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }

    /**
     * Get potential sales count (status = 'Potencijal')
     */
    public function getPotentialSalesCount(): int
    {
        try {
            $query = "SELECT COUNT(*) as count FROM {$this->table} WHERE sale_status = 'Potencijal'";
            return (int)$this->dbConnection->querySingleValue($query) ?? 0;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getPotentialSalesCount', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }

    /**
     * Get storno sales count (status = 'Storno')
     */
    public function getStornoSalesCount(): int
    {
        try {
            $query = "SELECT COUNT(*) as count FROM {$this->table} WHERE sale_status = 'Storno'";
            return (int)$this->dbConnection->querySingleValue($query) ?? 0;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getStornoSalesCount', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }

    /**
     * Get conversion rate (Aktivno / Potencijal * 100)
     */
    public function getConversionRate(): float
    {
        try {
            $active = $this->getActiveSalesCount();
            $potential = $this->getPotentialSalesCount();
            
            if ($potential == 0) return 0;
            
            return round(($active / $potential) * 100, 2);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getConversionRate', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }

    /**
     * Get storno rate (Storno / Total * 100)
     */
    public function getStornoRate(): float
    {
        try {
            $storno = $this->getStornoSalesCount();
            $total = $this->getTotalSalesCount();
            
            if ($total == 0) return 0;
            
            return round(($storno / $total) * 100, 2);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getStornoRate', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }

    /**
     * Get monthly growth rate (current month vs previous month)
     */
    public function getMonthlyGrowthRate(): float
    {
        try {
            $currentMonth = date('m');
            $currentYear = date('Y');
            $lastMonth = $currentMonth == 1 ? 12 : $currentMonth - 1;
            $lastMonthYear = $currentMonth == 1 ? $currentYear - 1 : $currentYear;
            
            $currentQuery = "SELECT COUNT(*) as count FROM {$this->table} 
                           WHERE MONTH(sale_statusdate) = :current_month 
                           AND YEAR(sale_statusdate) = :current_year";
            $currentCount = (int)$this->dbConnection->querySingleValue($currentQuery, [
                'current_month' => $currentMonth,
                'current_year' => $currentYear
            ]) ?? 0;
            
            $lastQuery = "SELECT COUNT(*) as count FROM {$this->table} 
                        WHERE MONTH(sale_statusdate) = :last_month 
                        AND YEAR(sale_statusdate) = :last_month_year";
            $lastCount = (int)$this->dbConnection->querySingleValue($lastQuery, [
                'last_month' => $lastMonth,
                'last_month_year' => $lastMonthYear
            ]) ?? 0;
            
            if ($lastCount == 0) return 0;
            
            return round((($currentCount - $lastCount) / $lastCount) * 100, 2);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getMonthlyGrowthRate', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }

    /**
     * Get sales funnel data
     */
    public function getSalesFunnelData(): array
    {
        try {
            $query = "SELECT sale_status as status, COUNT(*) as count 
                    FROM {$this->table} 
                    WHERE sale_status IN ('Potencijal', 'Uneseno', 'Aktivno', 'Storno')
                    GROUP BY sale_status 
                    ORDER BY FIELD(sale_status, 'Potencijal', 'Uneseno', 'Aktivno', 'Storno')";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getSalesFunnelData', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Get monthly trend data (last 12 months)
     */
    public function getMonthlyTrendData(): array
    {
        try {
            $query = "SELECT 
                        YEAR(sale_statusdate) as year,
                        MONTH(sale_statusdate) as month,
                        COUNT(*) as total_sales,
                        COUNT(CASE WHEN sale_status = 'Aktivno' THEN 1 END) as active_sales,
                        COUNT(CASE WHEN sale_status = 'Potencijal' THEN 1 END) as potential_sales
                    FROM {$this->table} 
                    WHERE sale_statusdate >= DATE_SUB(CURRENT_DATE, INTERVAL 12 MONTH)
                    AND sale_statusdate IS NOT NULL
                    GROUP BY YEAR(sale_statusdate), MONTH(sale_statusdate)
                    ORDER BY year ASC, month ASC";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getMonthlyTrendData', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Get top services data
     */
    public function getTopServicesData(): array
    {
        try {
            $query = "SELECT 
                        sale_prodanausluga as service,
                        COUNT(*) as total_count,
                        COUNT(CASE WHEN sale_status = 'Aktivno' THEN 1 END) as active_count
                    FROM {$this->table} 
                    WHERE sale_prodanausluga IS NOT NULL
                    GROUP BY sale_prodanausluga 
                    ORDER BY total_count DESC 
                    LIMIT 10";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getTopServicesData', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Get top agents data
     */
    public function getTopAgentsData(): array
    {
        try {
            $query = "SELECT 
                        sale_djelatnik as agent,
                        COUNT(*) as total_sales,
                        COUNT(CASE WHEN sale_status = 'Aktivno' THEN 1 END) as active_sales,
                        ROUND((COUNT(CASE WHEN sale_status = 'Aktivno' THEN 1 END) / COUNT(*)) * 100, 1) as conversion_rate
                    FROM {$this->table} 
                    WHERE sale_djelatnik IS NOT NULL
                    GROUP BY sale_djelatnik 
                    ORDER BY active_sales DESC 
                    LIMIT 10";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getTopAgentsData', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Get status distribution
     */
    public function getStatusDistribution(): array
    {
        try {
            $query = "SELECT sale_status as status, COUNT(*) as count 
                    FROM {$this->table} 
                    WHERE sale_status IS NOT NULL
                    GROUP BY sale_status 
                    ORDER BY count DESC";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getStatusDistribution', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Get today's sales count
     */
    public function getTodaySalesCount(): int
    {
        try {
            $today = date('Y-m-d');
            $query = "SELECT COUNT(*) as count FROM {$this->table} 
                    WHERE DATE(sale_statusdate) = :today";
            
            return (int)$this->dbConnection->querySingleValue($query, ['today' => $today]) ?? 0;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getTodaySalesCount', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }

    /**
     * Get this week's sales count
     */
    public function getThisWeekSalesCount(): int
    {
        try {
            $weekStart = date('Y-m-d', strtotime('monday this week'));
            $weekEnd = date('Y-m-d', strtotime('sunday this week'));
            
            $query = "SELECT COUNT(*) as count FROM {$this->table} 
                    WHERE DATE(sale_statusdate) BETWEEN :week_start AND :week_end";
            
            return (int)$this->dbConnection->querySingleValue($query, [
                'week_start' => $weekStart,
                'week_end' => $weekEnd
            ]) ?? 0;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getThisWeekSalesCount', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }

    /**
     * Get last week's sales count
     */
    public function getLastWeekSalesCount(): int
    {
        try {
            $lastWeekStart = date('Y-m-d', strtotime('monday last week'));
            $lastWeekEnd = date('Y-m-d', strtotime('sunday last week'));
            
            $query = "SELECT COUNT(*) as count FROM {$this->table} 
                    WHERE DATE(sale_statusdate) BETWEEN :week_start AND :week_end";
            
            return (int)$this->dbConnection->querySingleValue($query, [
                'week_start' => $lastWeekStart,
                'week_end' => $lastWeekEnd
            ]) ?? 0;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getLastWeekSalesCount', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }

    /**
     * Get recent conversions (recently changed to 'Aktivno')
     */
    public function getRecentConversions(): array
    {
        try {
            $query = "SELECT 
                        sale_imeiprezime,
                        sale_prodanausluga,
                        sale_djelatnik,
                        sale_statusdate
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno'
                    AND sale_statusdate >= DATE_SUB(CURRENT_DATE, INTERVAL 7 DAYS)
                    ORDER BY sale_statusdate DESC 
                    LIMIT 10";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getRecentConversions', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }

    /**
     * Get pending actions count (status = 'Potencijal' older than 7 days)
     */
    public function getPendingActionsCount(): int
    {
        try {
            $query = "SELECT COUNT(*) as count FROM {$this->table} 
                    WHERE sale_status = 'Potencijal'
                    AND sale_statusdate <= DATE_SUB(CURRENT_DATE, INTERVAL 7 DAYS)";
            
            return (int)$this->dbConnection->querySingleValue($query) ?? 0;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getPendingActionsCount', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }

    /**
     * Get best agent this month
     */
    public function getBestAgentThisMonth(): ?string
    {
        try {
            $currentMonth = date('m');
            $currentYear = date('Y');
            
            $query = "SELECT sale_djelatnik 
                    FROM {$this->table} 
                    WHERE MONTH(sale_statusdate) = :month 
                    AND YEAR(sale_statusdate) = :year
                    AND sale_status = 'Aktivno'
                    AND sale_djelatnik IS NOT NULL
                    GROUP BY sale_djelatnik 
                    ORDER BY COUNT(*) DESC 
                    LIMIT 1";
            
            return $this->dbConnection->querySingleValue($query, [
                'month' => $currentMonth,
                'year' => $currentYear
            ]);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getBestAgentThisMonth', [
                    'error' => $e->getMessage()
                ]);
            }
            return null;
        }
    }

    /**
     * Get top service this month
     */
    public function getTopServiceThisMonth(): ?string
    {
        try {
            $currentMonth = date('m');
            $currentYear = date('Y');
            
            $query = "SELECT sale_prodanausluga 
                    FROM {$this->table} 
                    WHERE MONTH(sale_statusdate) = :month 
                    AND YEAR(sale_statusdate) = :year
                    AND sale_status = 'Aktivno'
                    AND sale_prodanausluga IS NOT NULL
                    GROUP BY sale_prodanausluga 
                    ORDER BY COUNT(*) DESC 
                    LIMIT 1";
            
            return $this->dbConnection->querySingleValue($query, [
                'month' => $currentMonth,
                'year' => $currentYear
            ]);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getTopServiceThisMonth', [
                    'error' => $e->getMessage()
                ]);
            }
            return null;
        }
    }
}