<?php
// ============================================================================
// 2. DatabaseTableTeamBonus.php - Team Bonus Analytics Model  
// ============================================================================

namespace sales\Models;

use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

class DatabaseTableTeamBonus extends DatabaseTable
{
    /**
     * @var \baseKRIZAN\Database\DatabaseConnection
     */
    protected $dbConnection;
    
    /**
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Constructor for Team Bonus Analytics (service-focused)
     * 
     * @param DatabaseConnection $dbConnection Database connection
     * @param string $table Table name
     * @param string $primaryKey Primary key
     * @param string $className Class name
     * @param array $constructorArgs Constructor arguments
     * @param Logger|null $logger Logger instance
     */
    public function __construct(
        DatabaseConnection $dbConnection, 
        string $table, 
        string $primaryKey, 
        string $className = 'stdClass', 
        array $constructorArgs = [],
        ?Logger $logger = null
    ) {
        parent::__construct($dbConnection, $table, $primaryKey, $className, $constructorArgs);
        $this->dbConnection = $dbConnection;
        $this->logger = $logger;
        
        if ($this->logger) {
            $this->logger->modules('DatabaseTableTeamBonus initialized', [
                'table' => $table
            ]);
        }
    }

    /**
     * ===================================================================
     * TEAM BONUS METHODS - Add these methods to DatabaseTableSales.php
     * ===================================================================
     */

    /**
     * Get monthly active sales grouped by service type
     * 
     * @param string|null $specificMonth Specific month to filter (MM format)
     * @param string|null $specificYear Specific year to filter (YYYY format)
     * @return array Monthly active sales by service
     */
    public function getMonthlyActiveSalesByService(?string $specificMonth = null, ?string $specificYear = null): array
    {
        try {
            $whereClause = "WHERE sale_status = 'Aktivno' AND sale_statusdate IS NOT NULL AND sale_prodanausluga IS NOT NULL";
            
            if ($specificMonth && $specificYear) {
                $whereClause .= " AND MONTH(sale_statusdate) = :month AND YEAR(sale_statusdate) = :year";
            }
            
            $query = "SELECT 
                        MONTH(sale_statusdate) as month,
                        YEAR(sale_statusdate) as year,
                        sale_prodanausluga as service_type,
                        COUNT(*) as active_count
                    FROM {$this->table} 
                    {$whereClause}
                    GROUP BY YEAR(sale_statusdate), MONTH(sale_statusdate), sale_prodanausluga 
                    ORDER BY year DESC, month DESC, active_count DESC";
            
            $params = [];
            if ($specificMonth && $specificYear) {
                $params = ['month' => (int)$specificMonth, 'year' => (int)$specificYear];
            }
            
            return $this->dbConnection->queryAndFetchAllAssoc($query, $params);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getMonthlyActiveSalesByService'
                ]);
            }
            return [];
        }
    }

    /**
     * Get active sales count by service type (all time)
     * 
     * @return array Active sales by service type
     */
    public function getActiveSalesByService(): array
    {
        try {
            $query = "SELECT 
                        sale_prodanausluga as service_type,
                        COUNT(*) as active_count
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_prodanausluga IS NOT NULL
                    GROUP BY sale_prodanausluga 
                    ORDER BY active_count DESC";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getActiveSalesByService'
                ]);
            }
            return [];
        }
    }

    /**
     * Get total active sales by month (all services combined)
     * 
     * @return array Total actives by month
     */
    public function getTotalActivesByMonth(): array
    {
        try {
            $query = "SELECT 
                        MONTH(sale_statusdate) as month,
                        YEAR(sale_statusdate) as year,
                        COUNT(*) as total_active_count
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_statusdate IS NOT NULL
                    GROUP BY YEAR(sale_statusdate), MONTH(sale_statusdate) 
                    ORDER BY year DESC, month DESC";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getTotalActivesByMonth'
                ]);
            }
            return [];
        }
    }

    /**
     * Get service growth trends (comparing last 6 months)
     * 
     * @return array Service growth trends
     */
    public function getServiceGrowthTrends(): array
    {
        try {
            $query = "SELECT 
                        sale_prodanausluga as service_type,
                        MONTH(sale_statusdate) as month,
                        YEAR(sale_statusdate) as year,
                        COUNT(*) as active_count
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_statusdate >= DATE_SUB(CURRENT_DATE, INTERVAL 6 MONTH)
                    AND sale_prodanausluga IS NOT NULL
                    GROUP BY sale_prodanausluga, YEAR(sale_statusdate), MONTH(sale_statusdate) 
                    ORDER BY service_type, year, month";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getServiceGrowthTrends'
                ]);
            }
            return [];
        }
    }

    /**
     * Get current month active sales count
     * 
     * @return int Current month active count
     */
    public function getCurrentMonthActiveCount(): int
    {
        try {
            $currentMonth = date('m');
            $currentYear = date('Y');
            
            $query = "SELECT COUNT(*) as count 
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND MONTH(sale_statusdate) = :month 
                    AND YEAR(sale_statusdate) = :year";
            
            return (int)$this->dbConnection->querySingleValue($query, [
                'month' => $currentMonth,
                'year' => $currentYear
            ]) ?? 0;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getCurrentMonthActiveCount'
                ]);
            }
            return 0;
        }
    }

    /**
     * Get last month active sales count
     * 
     * @return int Last month active count
     */
    public function getLastMonthActiveCount(): int
    {
        try {
            $lastMonth = date('m', strtotime('-1 month'));
            $lastMonthYear = date('Y', strtotime('-1 month'));
            
            $query = "SELECT COUNT(*) as count 
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND MONTH(sale_statusdate) = :month 
                    AND YEAR(sale_statusdate) = :year";
            
            return (int)$this->dbConnection->querySingleValue($query, [
                'month' => $lastMonth,
                'year' => $lastMonthYear
            ]) ?? 0;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getLastMonthActiveCount'
                ]);
            }
            return 0;
        }
    }

    /**
     * Get total number of different active services
     * 
     * @return int Total active services count
     */
    public function getTotalActiveServices(): int
    {
        try {
            $query = "SELECT COUNT(DISTINCT sale_prodanausluga) as count 
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_prodanausluga IS NOT NULL";
            
            return (int)$this->dbConnection->querySingleValue($query) ?? 0;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getTotalActiveServices'
                ]);
            }
            return 0;
        }
    }

    /**
     * Get top performing services (by active count)
     * 
     * @param int $limit Number of top services to return
     * @return array Top performing services
     */
    public function getTopPerformingServices(int $limit = 5): array
    {
        try {
            $query = "SELECT 
                        sale_prodanausluga as service_type,
                        COUNT(*) as active_count,
                        COUNT(CASE WHEN MONTH(sale_statusdate) = MONTH(CURRENT_DATE) 
                                  AND YEAR(sale_statusdate) = YEAR(CURRENT_DATE) THEN 1 END) as current_month_count
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_prodanausluga IS NOT NULL
                    GROUP BY sale_prodanausluga 
                    ORDER BY active_count DESC 
                    LIMIT " . (int)$limit;
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getTopPerformingServices'
                ]);
            }
            return [];
        }
    }

    /**
     * Get monthly actives comparison (current vs previous month)
     * 
     * @return array Monthly comparison data
     */
    public function getMonthlyActivesComparison(): array
    {
        try {
            $query = "SELECT 
                        sale_prodanausluga as service_type,
                        COUNT(CASE WHEN MONTH(sale_statusdate) = MONTH(CURRENT_DATE) 
                                  AND YEAR(sale_statusdate) = YEAR(CURRENT_DATE) THEN 1 END) as current_month,
                        COUNT(CASE WHEN MONTH(sale_statusdate) = MONTH(DATE_SUB(CURRENT_DATE, INTERVAL 1 MONTH)) 
                                  AND YEAR(sale_statusdate) = YEAR(DATE_SUB(CURRENT_DATE, INTERVAL 1 MONTH)) THEN 1 END) as previous_month
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_prodanausluga IS NOT NULL
                    AND sale_statusdate >= DATE_SUB(CURRENT_DATE, INTERVAL 2 MONTH)
                    GROUP BY sale_prodanausluga 
                    ORDER BY current_month DESC";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getMonthlyActivesComparison'
                ]);
            }
            return [];
        }
    }

    /**
     * Get detailed active sales for specific service and period
     * 
     * @param string $serviceType Service type to filter
     * @param string $month Month to filter (MM format)
     * @param string $year Year to filter (YYYY format)
     * @return array Detailed sales data
     */
    public function getDetailedActiveSalesForService(string $serviceType, string $month, string $year): array
    {
        try {
            $query = "SELECT 
                        id,
                        sale_imeiprezime,
                        sale_oib,
                        sale_adresa,
                        sale_statusdate,
                        sale_djelatnik,
                        sale_kontaktbroj
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_prodanausluga = :service_type
                    AND MONTH(sale_statusdate) = :month 
                    AND YEAR(sale_statusdate) = :year
                    ORDER BY sale_statusdate DESC";
            
            return $this->dbConnection->queryAndFetchAllAssoc($query, [
                'service_type' => $serviceType,
                'month' => (int)$month,
                'year' => (int)$year
            ]);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getDetailedActiveSalesForService'
                ]);
            }
            return [];
        }
    }

    /**
     * Get service summary for specific period
     * 
     * @param string $serviceType Service type to filter
     * @param string $month Month to filter (MM format)
     * @param string $year Year to filter (YYYY format)
     * @return array|null Service summary data
     */
    public function getServiceSummaryForPeriod(string $serviceType, string $month, string $year): ?array
    {
        try {
            $query = "SELECT 
                        COUNT(*) as total_active,
                        COUNT(DISTINCT sale_djelatnik) as unique_employees,
                        MIN(sale_statusdate) as first_active_date,
                        MAX(sale_statusdate) as last_active_date
                    FROM {$this->table} 
                    WHERE sale_status = 'Aktivno' 
                    AND sale_prodanausluga = :service_type
                    AND MONTH(sale_statusdate) = :month 
                    AND YEAR(sale_statusdate) = :year";
            
            return $this->dbConnection->queryAndFetchAssoc($query, [
                'service_type' => $serviceType,
                'month' => (int)$month,
                'year' => (int)$year
            ]);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableSales', [
                    'error' => $e->getMessage(),
                    'function' => 'getServiceSummaryForPeriod'
                ]);
            }
            return null;
        }
    }
}