<?php
// ============================================================================
// DatabaseTableTeamBonusFinancije.php - Team Bonus Financial Analytics Model  
// AŽURIRANO za novu strukturu bonus tablica
// ============================================================================

namespace sales\Models;

use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

class DatabaseTableTeamBonusFinancije extends DatabaseTable
{
    /**
     * @var \baseKRIZAN\Database\DatabaseConnection
     */
    protected $dbConnection;
    
    /**
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Constructor for Team Bonus Financial Analytics
     * 
     * @param DatabaseConnection $dbConnection Database connection
     * @param string $table Table name
     * @param string $primaryKey Primary key
     * @param string $className Class name
     * @param array $constructorArgs Constructor arguments
     * @param Logger|null $logger Logger instance
     */
    public function __construct(
        DatabaseConnection $dbConnection, 
        string $table, 
        string $primaryKey, 
        string $className = 'stdClass', 
        array $constructorArgs = [],
        ?Logger $logger = null
    ) {
        parent::__construct($dbConnection, $table, $primaryKey, $className, $constructorArgs);
        $this->dbConnection = $dbConnection;
        $this->logger = $logger;
        
        // Ensure target table has target_period column
        $this->ensureTargetPeriodColumn();
        
        if ($this->logger) {
            $this->logger->modules('DatabaseTableTeamBonusFinancije initialized', [
                'table' => $table
            ]);
        }
    }

    /**
     * Ensure prodaja_target table has target_period column - AŽURIRANO za novu strukturu
     */
    private function ensureTargetPeriodColumn(): void
    {
        try {
            $columnExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.columns 
                 WHERE table_name = 'prodaja_target' 
                 AND column_name = 'target_period' 
                 AND table_schema = DATABASE()"
            );
            
            if ($columnExists === null) {
                $this->dbConnection->execute(
                    "ALTER TABLE prodaja_target ADD COLUMN target_period VARCHAR(7) DEFAULT NULL COMMENT 'Format: YYYY-MM'"
                );
                
                // Add default target for current month if table is empty
                $hasData = $this->dbConnection->querySingleValue("SELECT COUNT(*) FROM prodaja_target");
                if ($hasData == 0) {
                    $currentPeriod = date('Y-m');
                    // AŽURIRANO - koristimo novu strukturu kolona
                    $this->dbConnection->execute(
                        "INSERT INTO prodaja_target (target_period, prijenos, p2p, tomato) 
                         VALUES (?, 350, 165, 80)",
                        [$currentPeriod]
                    );
                }
                
                if ($this->logger) {
                    $this->logger->modules('Added target_period column to prodaja_target table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error ensuring target_period column', [
                    'error' => $e->getMessage()
                ]);
            }
        }
    }

    /**
     * Get financial earnings by bonus type for specific period - AŽURIRANO
     * 
     * @param string $month Month in MM format
     * @param string $year Year in YYYY format
     * @return array Financial earnings by bonus type
     */
    public function getFinancialEarningsByType(string $month, string $year): array
    {
        try {
            $period = "$year-$month";
            $results = [];
            
            // Get target for the period
            $targets = $this->getTargetsForPeriod($period);
            if (!$targets) {
                return [];
            }
            
            // Calculate earnings for each bonus type - AŽURIRANO
            $results['tomato'] = $this->calculateTomatoEarnings($month, $year, $targets['tomato']);
            $results['p2p'] = $this->calculateP2pEarnings($month, $year, $targets['p2p']);
            $results['prijenos'] = $this->calculatePrijenosEarnings($month, $year, $targets['prijenos']);
            
            return $results;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting financial earnings by type', [
                    'error' => $e->getMessage(),
                    'month' => $month,
                    'year' => $year
                ]);
            }
            return [];
        }
    }

    /**
     * Get targets for specific period - AŽURIRANO za novu strukturu
     */
    private function getTargetsForPeriod(string $period): ?array
    {
        try {
            $query = "SELECT prijenos, p2p, tomato 
                     FROM prodaja_target 
                     WHERE target_period = :period
                     ORDER BY id DESC LIMIT 1";
            
            $result = $this->dbConnection->queryAndFetchAssoc($query, ['period' => $period]);
            
            if (!$result) {
                // Fallback to most recent target
                $query = "SELECT prijenos, p2p, tomato 
                         FROM prodaja_target 
                         ORDER BY id DESC LIMIT 1";
                $result = $this->dbConnection->queryAndFetchAssoc($query);
            }
            
            return $result;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting targets for period', [
                    'error' => $e->getMessage(),
                    'period' => $period
                ]);
            }
            return null;
        }
    }

    /**
     * Calculate Tomato earnings
     */
    private function calculateTomatoEarnings(string $month, string $year, int $target): array
    {
        try {
            // Get sold services for Tomato groups
            $soldServices = $this->getSoldServicesByGroups(['Tomato', 'Tomato P2P'], $month, $year);
            $totalSold = array_sum(array_column($soldServices, 'count'));
            
            if ($totalSold == 0) {
                return ['total_earnings' => 0, 'services' => [], 'sold_count' => 0, 'target' => $target];
            }
            
            // Get bonus schemes
            $bonusSchemes = $this->getBonusSchemes('prodaja_bonus_tomato');
            
            $totalEarnings = 0;
            $serviceDetails = [];
            
            foreach ($soldServices as $service) {
                $scheme = $this->findSchemeForService($bonusSchemes, $service['tarifa']);
                if (!$scheme) continue;
                
                $serviceEarnings = $this->calculateServiceEarnings(
                    $service['count'], 
                    $scheme, 
                    $totalSold, 
                    $target,
                    'tomato'
                );
                
                $totalEarnings += $serviceEarnings;
                $serviceDetails[] = [
                    'tarifa' => $service['tarifa'],
                    'count' => $service['count'],
                    'earnings' => $serviceEarnings
                ];
            }
            
            return [
                'total_earnings' => $totalEarnings,
                'services' => $serviceDetails,
                'sold_count' => $totalSold,
                'target' => $target,
                'achievement_percent' => $target > 0 ? round(($totalSold / $target) * 100, 1) : 0
            ];
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error calculating Tomato earnings', [
                    'error' => $e->getMessage()
                ]);
            }
            return ['total_earnings' => 0, 'services' => [], 'sold_count' => 0, 'target' => $target];
        }
    }

    /**
     * Calculate P2P earnings
     */
    private function calculateP2pEarnings(string $month, string $year, int $target): array
    {
        try {
            $soldServices = $this->getSoldServicesByGroups(['P2P'], $month, $year);
            $totalSold = array_sum(array_column($soldServices, 'count'));
            
            if ($totalSold == 0) {
                return ['total_earnings' => 0, 'services' => [], 'sold_count' => 0, 'target' => $target];
            }
            
            $bonusSchemes = $this->getBonusSchemes('prodaja_bonus_p2p');
            
            $totalEarnings = 0;
            $serviceDetails = [];
            
            foreach ($soldServices as $service) {
                $scheme = $this->findSchemeForService($bonusSchemes, $service['tarifa']);
                if (!$scheme) continue;
                
                $serviceEarnings = $this->calculateServiceEarnings(
                    $service['count'], 
                    $scheme, 
                    $totalSold, 
                    $target,
                    'p2p'
                );
                
                $totalEarnings += $serviceEarnings;
                $serviceDetails[] = [
                    'tarifa' => $service['tarifa'],
                    'count' => $service['count'],
                    'earnings' => $serviceEarnings
                ];
            }
            
            return [
                'total_earnings' => $totalEarnings,
                'services' => $serviceDetails,
                'sold_count' => $totalSold,
                'target' => $target,
                'achievement_percent' => $target > 0 ? round(($totalSold / $target) * 100, 1) : 0
            ];
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error calculating P2P earnings', [
                    'error' => $e->getMessage()
                ]);
            }
            return ['total_earnings' => 0, 'services' => [], 'sold_count' => 0, 'target' => $target];
        }
    }

    /**
     * Calculate Prijenos earnings - AŽURIRANO za kombinirane grupe
     */
    private function calculatePrijenosEarnings(string $month, string $year, int $target): array
    {
        try {
            // Get all prijenos services (combines all groups that were previously split)
            $allPrijenosGroups = ['Sat', 'Homebox', 'Parica', 'MNP HT', 'Optika', 'MNP Telemach'];
            $soldServices = $this->getSoldServicesByGroups($allPrijenosGroups, $month, $year);
            $totalSold = array_sum(array_column($soldServices, 'count'));
            
            if ($totalSold == 0) {
                return ['total_earnings' => 0, 'services' => [], 'sold_count' => 0, 'target' => $target];
            }
            
            // Use unified prijenos bonus schemes
            $bonusSchemes = $this->getBonusSchemes('prodaja_bonus_prijenos');
            
            $totalEarnings = 0;
            $serviceDetails = [];
            
            foreach ($soldServices as $service) {
                $scheme = $this->findSchemeForService($bonusSchemes, $service['tarifa']);
                if (!$scheme) continue;
                
                $serviceEarnings = $this->calculateServiceEarnings(
                    $service['count'], 
                    $scheme, 
                    $totalSold, 
                    $target,
                    'prijenos'
                );
                
                $totalEarnings += $serviceEarnings;
                $serviceDetails[] = [
                    'tarifa' => $service['tarifa'],
                    'count' => $service['count'],
                    'earnings' => $serviceEarnings
                ];
            }
            
            return [
                'total_earnings' => $totalEarnings,
                'services' => $serviceDetails,
                'sold_count' => $totalSold,
                'target' => $target,
                'achievement_percent' => $target > 0 ? round(($totalSold / $target) * 100, 1) : 0
            ];
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error calculating Prijenos earnings', [
                    'error' => $e->getMessage()
                ]);
            }
            return ['total_earnings' => 0, 'services' => [], 'sold_count' => 0, 'target' => $target];
        }
    }

    /**
     * Get sold services by groups for specific period
     */
    private function getSoldServicesByGroups(array $groups, string $month, string $year): array
    {
        try {
            $placeholders = str_repeat('?,', count($groups) - 1) . '?';
            
            $query = "SELECT 
                        p.sale_prodanausluga as tarifa,
                        COUNT(*) as count
                     FROM {$this->table} p
                     JOIN prodaja_vrste pv ON p.sale_prodanausluga = pv.tarifa
                     WHERE p.sale_status = 'Aktivno'
                     AND MONTH(p.sale_statusdate) = ?
                     AND YEAR(p.sale_statusdate) = ?
                     AND pv.grupa IN ($placeholders)
                     GROUP BY p.sale_prodanausluga";
            
            $params = array_merge([(int)$month, (int)$year], $groups);
            
            return $this->dbConnection->queryAndFetchAllAssoc($query, $params);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting sold services by groups', [
                    'error' => $e->getMessage(),
                    'groups' => $groups
                ]);
            }
            return [];
        }
    }

    /**
     * Get bonus schemes for specific table
     */
    private function getBonusSchemes(string $tableName): array
    {
        try {
            $query = "SELECT * FROM $tableName";
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Error getting bonus schemes', [
                    'error' => $e->getMessage(),
                    'table' => $tableName
                ]);
            }
            return [];
        }
    }

    /**
     * Find bonus scheme for specific service
     */
    private function findSchemeForService(array $schemes, string $tarifa): ?array
    {
        foreach ($schemes as $scheme) {
            if ($scheme['vrsta_tarifa'] === $tarifa) {
                return $scheme;
            }
        }
        return null;
    }

    /**
     * Calculate earnings for specific service - AŽURIRANO za novu strukturu
     */
    private function calculateServiceEarnings(int $count, array $scheme, int $totalSold, int $target, string $bonusType): float
    {
        if ($count == 0) return 0;
        
        $baseAmount = (float)$scheme['osnovna_naknada'];
        $rangeBonus = $this->getRangeBonus($scheme, $totalSold, $bonusType);
        $percentageBonus = $this->getPercentageBonus($scheme, $totalSold, $target);
        
        $perUnitEarning = $baseAmount + $rangeBonus + $percentageBonus;
        
        return $count * $perUnitEarning;
    }

    /**
     * Get range bonus based on total sold count - AŽURIRANO
     */
    private function getRangeBonus(array $scheme, int $totalSold, string $bonusType): float
    {
        if ($bonusType === 'p2p') {
            if ($totalSold >= 650) return (float)$scheme['rang_650_plus'];
            if ($totalSold >= 400) return (float)$scheme['rang_400_650'];
            if ($totalSold >= 150) return (float)$scheme['rang_150_400'];
            return (float)$scheme['rang_150'];
        } else {
            // tomato, prijenos (unified)
            // Za tomato će i dalje koristiti stare nazive kolona
            if ($totalSold >= 750) return (float)($scheme['rang_750_1049'] ?? $scheme['rang_800_1099'] ?? 0);
            if ($totalSold >= 450) return (float)($scheme['rang_450_749'] ?? $scheme['rang_550_779'] ?? 0);
            if ($totalSold >= 250) return (float)($scheme['rang_250_449'] ?? $scheme['rang_300_549'] ?? 0);
            return (float)($scheme['rang_249'] ?? $scheme['rang_299'] ?? 0);
        }
    }

    /**
     * Get percentage bonus based on target achievement
     */
    private function getPercentageBonus(array $scheme, int $totalSold, int $target): float
    {
        if ($target == 0) return 0;
        
        $percentage = ($totalSold / $target) * 100;
        
        if ($percentage >= 90) return (float)$scheme['rang_90_120'];
        if ($percentage >= 80) return (float)$scheme['rang_80_90'];
        if ($percentage >= 70) return (float)$scheme['rang_70_80'];
        
        return 0;
    }

    /**
     * Get total company earnings for period - AŽURIRANO
     */
    public function getTotalCompanyEarnings(string $month, string $year): float
    {
        $earnings = $this->getFinancialEarningsByType($month, $year);
        
        return array_sum([
            $earnings['tomato']['total_earnings'] ?? 0,
            $earnings['p2p']['total_earnings'] ?? 0,
            $earnings['prijenos']['total_earnings'] ?? 0
        ]);
    }

    /**
     * Get earnings comparison between periods
     */
    public function getEarningsComparison(string $currentMonth, string $currentYear, string $compareMonth, string $compareYear): array
    {
        $currentEarnings = $this->getFinancialEarningsByType($currentMonth, $currentYear);
        $compareEarnings = $this->getFinancialEarningsByType($compareMonth, $compareYear);
        
        $currentTotal = $this->getTotalCompanyEarnings($currentMonth, $currentYear);
        $compareTotal = $this->getTotalCompanyEarnings($compareMonth, $compareYear);
        
        $growthPercent = $compareTotal > 0 ? round((($currentTotal - $compareTotal) / $compareTotal) * 100, 1) : 0;
        
        return [
            'current' => [
                'period' => "$currentYear-$currentMonth",
                'total' => $currentTotal,
                'by_type' => $currentEarnings
            ],
            'compare' => [
                'period' => "$compareYear-$compareMonth", 
                'total' => $compareTotal,
                'by_type' => $compareEarnings
            ],
            'growth_percent' => $growthPercent
        ];
    }

    /**
     * Get monthly earnings trend for charts
     */
    public function getMonthlyEarningsTrend(int $monthsBack = 6): array
    {
        $results = [];
        
        for ($i = $monthsBack - 1; $i >= 0; $i--) {
            $date = new \DateTime();
            $date->modify("-$i months");
            $month = $date->format('m');
            $year = $date->format('Y');
            
            $total = $this->getTotalCompanyEarnings($month, $year);
            
            $results[] = [
                'month' => (int)$month,
                'year' => (int)$year,
                'period' => $date->format('Y-m'),
                'total_earnings' => $total
            ];
        }
        
        return $results;
    }
}