// agentbonusbodovanje.js
document.addEventListener('DOMContentLoaded', function() {
  // Card value animation for points and bonuses
  const cardValues = document.querySelectorAll('.card-value');
  cardValues.forEach(valueElement => {
    let originalText = valueElement.textContent;
    
    // Extract numeric value
    let numericValue = parseFloat(originalText.replace(/[^\d.-]/g, ''));
    if (!isNaN(numericValue)) {
      if (originalText.includes('€')) {
        // EUR value animation
        animateEuroValue(valueElement, 0, numericValue, 1200);
      } else {
        // Simple integer animation for points and counts
        animateValue(valueElement, 0, numericValue, 1000);
      }
    }
  });
  
  // Initialize stat values animation
  const statValues = document.querySelectorAll('.stat-value');
  statValues.forEach(valueElement => {
    let originalText = valueElement.textContent;
    let numericValue = parseFloat(originalText.replace(/[^\d.-]/g, ''));
    
    if (!isNaN(numericValue)) {
      if (originalText.includes('€')) {
        animateEuroValue(valueElement, 0, numericValue, 800);
      } else {
        animateValue(valueElement, 0, numericValue, 800);
      }
    }
  });
  
  // Refresh stats periodically (every 5 minutes)
  if (typeof refreshBodovanjeInterval === 'undefined') {
    let refreshBodovanjeInterval = setInterval(refreshBodovanjeStats, 300000);
  }

  // Add hover effects to agent items
  const agentItems = document.querySelectorAll('.agent-item');
  agentItems.forEach(item => {
    item.addEventListener('mouseenter', function() {
      this.style.transform = 'translateX(5px)';
    });
    
    item.addEventListener('mouseleave', function() {
      this.style.transform = 'translateX(0)';
    });
  });

  // Add hover effects to table rows
  const tableRows = document.querySelectorAll('.table-row');
  tableRows.forEach(row => {
    row.addEventListener('mouseenter', function() {
      this.style.transform = 'translateY(-2px)';
    });
    
    row.addEventListener('mouseleave', function() {
      this.style.transform = 'translateY(0)';
    });
  });

  // Add hover effects to group cards
  const groupCards = document.querySelectorAll('.group-card');
  groupCards.forEach(card => {
    card.addEventListener('mouseenter', function() {
      this.style.transform = 'translateY(-5px)';
    });
    
    card.addEventListener('mouseleave', function() {
      this.style.transform = 'translateY(0)';
    });
  });

  // Period filter auto-submit on change
  const monthSelect = document.getElementById('month');
  const yearSelect = document.getElementById('year');
  
  if (monthSelect && yearSelect) {
    monthSelect.addEventListener('change', function() {
      this.form.submit();
    });
    
    yearSelect.addEventListener('change', function() {
      this.form.submit();
    });
  }

  // Ranking badge animations
  const rankingBadges = document.querySelectorAll('.agent-rank, .rank-badge');
  rankingBadges.forEach(badge => {
    if (!badge.classList.contains('rank-nema')) {
      badge.addEventListener('mouseenter', function() {
        this.style.transform = 'scale(1.1)';
      });
      
      badge.addEventListener('mouseleave', function() {
        this.style.transform = 'scale(1)';
      });
    }
  });

  // Position badge animations
  const positionBadges = document.querySelectorAll('.position-badge');
  positionBadges.forEach(badge => {
    badge.addEventListener('mouseenter', function() {
      this.style.transform = 'scale(1.2) rotate(5deg)';
    });
    
    badge.addEventListener('mouseleave', function() {
      this.style.transform = 'scale(1) rotate(0deg)';
    });
  });
});

// Animate counter value from start to end
function animateValue(element, start, end, duration) {
  let startTimestamp = null;
  
  const step = (timestamp) => {
    if (!startTimestamp) startTimestamp = timestamp;
    const progress = Math.min((timestamp - startTimestamp) / duration, 1);
    const value = Math.floor(progress * (end - start) + start);
    
    element.textContent = value.toLocaleString();
    
    if (progress < 1) {
      window.requestAnimationFrame(step);
    }
  };
  
  window.requestAnimationFrame(step);
}

// Animate EUR value from start to end
function animateEuroValue(element, start, end, duration) {
  let startTimestamp = null;
  
  const step = (timestamp) => {
    if (!startTimestamp) startTimestamp = timestamp;
    const progress = Math.min((timestamp - startTimestamp) / duration, 1);
    const value = progress * (end - start) + start;
    
    // Format as currency
    const formattedValue = value.toLocaleString('hr-HR', {
      minimumFractionDigits: 0,
      maximumFractionDigits: 0
    });
    
    element.textContent = formattedValue + ' €';
    
    if (progress < 1) {
      window.requestAnimationFrame(step);
    }
  };
  
  window.requestAnimationFrame(step);
}

// Function to refresh agent bonus bodovanje stats via AJAX
function refreshBodovanjeStats() {
  const urlParams = new URLSearchParams(window.location.search);
  const month = urlParams.get('month') || new Date().getMonth() + 1;
  const year = urlParams.get('year') || new Date().getFullYear();
  
  fetch('/agentbonusbodovanje/api/stats', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
    },
    body: JSON.stringify({
      month: month,
      year: year
    })
  })
    .then(response => response.json())
    .then(data => {
      // Update main dashboard cards
      updateValue('.points-card .card-value', data.totalPoints || 0);
      updateEuroValue('.bonus-card .card-value', data.totalBonus || 0);
      updateValue('.agents-card .card-value', data.activeAgents || 0);
      
      // Update group summary stats
      if (data.mnpAgents) {
        updateGroupStats('mnp', data.mnpAgents);
      }
      if (data.optikaAgents) {
        updateGroupStats('optika', data.optikaAgents);
      }
      if (data.p2pAgents) {
        updateGroupStats('p2p', data.p2pAgents);
      }
      
      // Update top performers if needed
      updateTopPerformers(data.topPerformers);
      
      console.log('Agent bonus bodovanje stats refreshed successfully');
    })
    .catch(error => {
      console.error('Error refreshing agent bonus bodovanje stats:', error);
    });
}

// Helper to update simple values
function updateValue(selector, value) {
  const element = document.querySelector(selector);
  if (element) {
    element.textContent = value.toLocaleString();
  }
}

// Helper to update EUR values
function updateEuroValue(selector, value) {
  const element = document.querySelector(selector);
  if (element) {
    const formattedValue = value.toLocaleString('hr-HR', {
      minimumFractionDigits: 0,
      maximumFractionDigits: 0
    });
    element.textContent = formattedValue + ' €';
  }
}

// Update group statistics
function updateGroupStats(groupType, agentsData) {
  if (!agentsData || !Array.isArray(agentsData)) return;
  
  const totalAgents = agentsData.length;
  const totalPoints = agentsData.reduce((sum, agent) => sum + (agent.total_points || 0), 0);
  const totalBonus = agentsData.reduce((sum, agent) => sum + (agent.bonus_amount || 0), 0);
  
  // Update group summary stats
  const groupCard = document.querySelector(`.${groupType}-card`);
  if (groupCard) {
    const statItems = groupCard.querySelectorAll('.stat-value');
    if (statItems.length >= 3) {
      statItems[0].textContent = totalAgents;
      statItems[1].textContent = totalPoints.toLocaleString();
      statItems[2].textContent = totalBonus.toLocaleString() + ' €';
    }
  }
}

// Update top performers section
function updateTopPerformers(performersData) {
  if (!performersData || !Array.isArray(performersData)) return;
  
  const performersTable = document.querySelector('.top-performers-table');
  if (!performersTable) return;
  
  // Keep header, update rows
  const header = performersTable.querySelector('.table-header');
  performersTable.innerHTML = '';
  performersTable.appendChild(header);
  
  if (performersData.length === 0) {
    const emptyState = document.createElement('div');
    emptyState.className = 'empty-state';
    emptyState.innerHTML = '<i class="fas fa-trophy"></i><p>Nema djelatnika s bodovima ovaj mjesec</p>';
    performersTable.appendChild(emptyState);
    return;
  }
  
  // Add updated performer rows
  performersData.forEach((performer, index) => {
    const row = document.createElement('div');
    row.className = 'table-row';
    row.innerHTML = `
      <div class="table-col position">
        <span class="position-badge position-${index + 1}">${index + 1}</span>
      </div>
      <div class="table-col agent-name">${escapeHtml(performer.agent_name)}</div>
      <div class="table-col agent-group">
        <span class="group-badge group-${performer.agent_group}">${performer.agent_group.toUpperCase()}</span>
      </div>
      <div class="table-col points">${performer.total_points.toLocaleString()}</div>
      <div class="table-col rank">
        <span class="rank-badge rank-${performer.current_rank.toLowerCase()}">${performer.current_rank}</span>
      </div>
      <div class="table-col bonus">${performer.bonus_amount.toLocaleString()} €</div>
    `;
    performersTable.appendChild(row);
  });
}

// Helper function to escape HTML
function escapeHtml(text) {
  const map = {
    '&': '&amp;',
    '<': '&lt;',
    '>': '&gt;',
    '"': '&quot;',
    "'": '&#039;'
  };
  return text.replace(/[&<>"']/g, function(m) { return map[m]; });
}

// Export functions for global access
window.AgentBonusBodovanje = {
  refreshStats: refreshBodovanjeStats,
  updateValue: updateValue,
  updateEuroValue: updateEuroValue
};