/**
 * JavaScript for Agent Bonus Dashboard
 * Handles chart creation functionality
 */
document.addEventListener('DOMContentLoaded', function() {
  // Get data elements
  const chartsData = document.getElementById('agentbonus-charts-data');
  
  if (!chartsData) {
    console.error('Agent bonus charts data element not found!');
    return;
  }
  
  try {
    // Parse data for the charts
    const monthlyData = JSON.parse(chartsData.getAttribute('data-monthly') || '[]');
    const agentsData = JSON.parse(chartsData.getAttribute('data-agents') || '[]');
    const trendsData = JSON.parse(chartsData.getAttribute('data-trends') || '[]');
    const comparisonData = JSON.parse(chartsData.getAttribute('data-comparison') || '[]');
    const breakdownData = JSON.parse(chartsData.getAttribute('data-breakdown') || '[]');
    
    // Create Monthly Agents chart
    createMonthlyAgentsChart(monthlyData);
    
    // Create Agent Actives chart
    createAgentActivesChart(agentsData);
    
    // Create Agent Service breakdown chart
    createAgentServiceChart(breakdownData);
    
  } catch (error) {
    console.error('Error initializing agent bonus charts:', error);
  }
});

/**
 * Create the monthly active sales by agents chart
 * @param {Array} monthlyData - Data for the monthly chart
 */
function createMonthlyAgentsChart(monthlyData) {
  if (!document.getElementById('monthlyAgentsChart')) {
    console.error('Monthly agents chart element not found');
    return;
  }
  
  // Group data by month and calculate totals
  const monthlyTotals = {};
  monthlyData.forEach(item => {
    const key = `${item.year}-${String(item.month).padStart(2, '0')}`;
    if (!monthlyTotals[key]) {
      monthlyTotals[key] = {
        year: item.year,
        month: item.month,
        total: 0
      };
    }
    monthlyTotals[key].total += parseInt(item.active_count || 0);
  });
  
  // Sort by date and get last 12 months
  const sortedMonths = Object.values(monthlyTotals)
    .sort((a, b) => new Date(a.year, a.month - 1) - new Date(b.year, b.month - 1))
    .slice(-12);
  
  const monthNames = ['Sij', 'Velj', 'Ožu', 'Tra', 'Svi', 'Lip', 'Srp', 'Kol', 'Ruj', 'Lis', 'Stu', 'Pro'];
  const labels = sortedMonths.map(item => `${monthNames[item.month - 1]} ${item.year}`);
  const values = sortedMonths.map(item => item.total);
  
  const monthlyCtx = document.getElementById('monthlyAgentsChart').getContext('2d');
  new Chart(monthlyCtx, {
    type: 'line',
    data: {
      labels: labels,
      datasets: [{
        label: 'Aktivne prodaje (svi djelatnici)',
        data: values,
        backgroundColor: 'rgba(255, 99, 132, 0.2)',
        borderColor: 'rgba(255, 99, 132, 1)',
        borderWidth: 3,
        fill: true,
        tension: 0.4,
        pointBackgroundColor: 'rgba(255, 99, 132, 1)',
        pointBorderColor: '#fff',
        pointBorderWidth: 2,
        pointRadius: 5
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      scales: {
        y: {
          beginAtZero: true,
          ticks: {
            precision: 0
          }
        },
        x: {
          ticks: {
            maxRotation: 45
          }
        }
      },
      plugins: {
        legend: {
          position: 'top',
          labels: {
            boxWidth: 12,
            font: {
              size: 11
            }
          }
        },
        tooltip: {
          callbacks: {
            label: function(context) {
              return `${context.dataset.label}: ${context.raw} aktivnih prodaja`;
            }
          }
        }
      }
    }
  });
}

/**
 * Create the active sales by agent chart
 * @param {Array} agentsData - Data for the agents chart
 */
function createAgentActivesChart(agentsData) {
  if (!document.getElementById('agentActivesChart')) {
    console.error('Agent actives chart element not found');
    return;
  }
  
  // Limit to top 8 agents if there are many
  let limitedData = agentsData;
  if (agentsData.length > 8) {
    const sortedData = [...agentsData].sort((a, b) => b.active_count - a.active_count);
    const topData = sortedData.slice(0, 7);
    const otherCount = sortedData.slice(7).reduce((sum, item) => sum + parseInt(item.active_count), 0);
    limitedData = [
      ...topData,
      { agent_name: 'Ostali', active_count: otherCount }
    ];
  }
  
  const agentLabels = limitedData.map(item => item.agent_name || 'Nepoznato');
  const agentValues = limitedData.map(item => parseInt(item.active_count));
  const agentColors = [
    '#ff6384', '#36a2eb', '#ffce56', '#4bc0c0', 
    '#9966ff', '#ff9f40', '#ff6384', '#c9cbcf'
  ];
  
  const agentCtx = document.getElementById('agentActivesChart').getContext('2d');
  new Chart(agentCtx, {
    type: 'doughnut',
    data: {
      labels: agentLabels,
      datasets: [{
        data: agentValues,
        backgroundColor: agentColors.slice(0, agentLabels.length),
        borderWidth: 2,
        borderColor: '#fff'
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      plugins: {
        legend: {
          position: 'right',
          labels: {
            boxWidth: 12,
            font: {
              size: 10
            },
            generateLabels: function(chart) {
              const data = chart.data;
              if (data.labels.length && data.datasets.length) {
                return data.labels.map((label, i) => {
                  const value = data.datasets[0].data[i];
                  const total = data.datasets[0].data.reduce((acc, val) => acc + val, 0);
                  const percentage = ((value / total) * 100).toFixed(1);
                  return {
                    text: `${label} (${percentage}%)`,
                    fillStyle: data.datasets[0].backgroundColor[i],
                    hidden: false,
                    index: i
                  };
                });
              }
              return [];
            }
          }
        },
        tooltip: {
          callbacks: {
            label: function(context) {
              const value = context.raw;
              const total = context.dataset.data.reduce((acc, val) => acc + val, 0);
              const percentage = ((value / total) * 100).toFixed(1);
              return `${context.label}: ${value} aktivnih (${percentage}%)`;
            }
          }
        }
      }
    }
  });
}

/**
 * Create the agent service breakdown chart
 * @param {Array} breakdownData - Data for the breakdown chart
 */
function createAgentServiceChart(breakdownData) {
  if (!document.getElementById('agentServiceChart')) {
    console.error('Agent service chart element not found');
    return;
  }
  
  if (!breakdownData || breakdownData.length === 0) {
    const ctx = document.getElementById('agentServiceChart').getContext('2d');
    new Chart(ctx, {
      type: 'bar',
      data: {
        labels: [],
        datasets: []
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          legend: {
            display: false
          }
        }
      }
    });
    return;
  }
  
  // Group by agent and get top 5 agents
  const agentServices = {};
  breakdownData.forEach(item => {
    const agent = item.agent_name;
    if (!agentServices[agent]) {
      agentServices[agent] = {
        total: 0,
        services: {}
      };
    }
    agentServices[agent].services[item.service_type] = parseInt(item.service_count);
    agentServices[agent].total += parseInt(item.service_count);
  });
  
  // Get top 5 agents by total count
  const topAgents = Object.keys(agentServices)
    .sort((a, b) => agentServices[b].total - agentServices[a].total)
    .slice(0, 5);
  
  // Get all unique services from top agents
  const allServices = new Set();
  topAgents.forEach(agent => {
    Object.keys(agentServices[agent].services).forEach(service => {
      allServices.add(service);
    });
  });
  
  const servicesList = Array.from(allServices);
  const colors = [
    'rgba(255, 99, 132, 0.8)',
    'rgba(54, 162, 235, 0.8)',
    'rgba(255, 206, 86, 0.8)',
    'rgba(75, 192, 192, 0.8)',
    'rgba(153, 102, 255, 0.8)',
    'rgba(255, 159, 64, 0.8)',
    'rgba(199, 199, 199, 0.8)',
    'rgba(83, 102, 255, 0.8)'
  ];
  
  // Create datasets for each service
  const datasets = servicesList.map((service, index) => {
    const data = topAgents.map(agent => {
      return agentServices[agent].services[service] || 0;
    });
    
    return {
      label: service,
      data: data,
      backgroundColor: colors[index % colors.length],
      borderColor: colors[index % colors.length].replace('0.8', '1'),
      borderWidth: 1
    };
  });
  
  const serviceCtx = document.getElementById('agentServiceChart').getContext('2d');
  new Chart(serviceCtx, {
    type: 'bar',
    data: {
      labels: topAgents,
      datasets: datasets
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      scales: {
        x: {
          stacked: true,
          ticks: {
            maxRotation: 45
          }
        },
        y: {
          stacked: true,
          beginAtZero: true,
          ticks: {
            precision: 0
          }
        }
      },
      plugins: {
        legend: {
          position: 'top',
          labels: {
            boxWidth: 12,
            font: {
              size: 10
            },
            usePointStyle: true
          }
        },
        tooltip: {
          mode: 'index',
          intersect: false,
          callbacks: {
            title: function(context) {
              return `Djelatnik: ${context[0].label}`;
            },
            label: function(context) {
              return `${context.dataset.label}: ${context.raw} prodaja`;
            },
            footer: function(tooltipItems) {
              let sum = 0;
              tooltipItems.forEach(function(tooltipItem) {
                sum += tooltipItem.raw;
              });
              return `Ukupno: ${sum} prodaja`;
            }
          }
        }
      }
    }
  });
}