/**
 * editbonus.js - JavaScript functionality for bonus edit form
 */
document.addEventListener('DOMContentLoaded', function() {
    // Cancel button functionality
    const cancelButton = document.getElementById('cancelButton');
    if (cancelButton) {
        cancelButton.addEventListener('click', function() {
            history.back();
        });
    }
    
    // Add form validation and visual enhancements
    initBonusFormValidation();
    addVisualEnhancements();
    
    /**
     * Initialize validation for bonus fields
     */
    function initBonusFormValidation() {
        const numberInputs = document.querySelectorAll('input[type="number"]');
        
        numberInputs.forEach(input => {
            // Add input event listener for real-time validation
            input.addEventListener('input', function() {
                validateBonusField(this);
            });
            
            // Add blur event listener for formatting
            input.addEventListener('blur', function() {
                formatBonusField(this);
            });
            
            // Add focus event listener for highlighting
            input.addEventListener('focus', function() {
                highlightField(this);
            });
        });
    }
    
    /**
     * Validate individual bonus field
     */
    function validateBonusField(field) {
        const value = parseFloat(field.value);
        const fieldGroup = field.closest('.form-group');
        
        // Remove existing validation classes
        fieldGroup.classList.remove('has-error', 'has-success');
        
        // Remove existing error messages
        const existingError = fieldGroup.querySelector('.field-error');
        if (existingError) {
            existingError.remove();
        }
        
        if (field.value === '') {
            // Empty is ok for nullable fields (tomato type)
            if (!field.required) {
                return;
            }
        }
        
        if (field.value !== '' && (isNaN(value) || value < 0)) {
            fieldGroup.classList.add('has-error');
            showFieldError(field, 'Vrijednost mora biti pozitivan broj');
        } else if (field.value !== '' && value > 999999) {
            fieldGroup.classList.add('has-error');
            showFieldError(field, 'Vrijednost je previsoka');
        } else if (field.value !== '') {
            fieldGroup.classList.add('has-success');
        }
    }
    
    /**
     * Format bonus field value
     */
    function formatBonusField(field) {
        if (field.value === '') return;
        
        const value = parseFloat(field.value);
        if (!isNaN(value)) {
            field.value = value.toFixed(2);
        }
    }
    
    /**
     * Highlight field on focus
     */
    function highlightField(field) {
        const fieldGroup = field.closest('.form-group');
        fieldGroup.classList.add('focused');
        
        field.addEventListener('blur', function() {
            fieldGroup.classList.remove('focused');
        }, { once: true });
    }
    
    /**
     * Show field error message
     */
    function showFieldError(field, message) {
        const fieldGroup = field.closest('.form-group');
        const errorElement = document.createElement('div');
        errorElement.className = 'field-error';
        errorElement.textContent = message;
        
        const formText = fieldGroup.querySelector('.form-text');
        if (formText) {
            formText.parentNode.insertBefore(errorElement, formText.nextSibling);
        } else {
            fieldGroup.appendChild(errorElement);
        }
    }
    
    /**
     * Add visual enhancements
     */
    function addVisualEnhancements() {
        // Add form submission validation
        const form = document.querySelector('.modern-form');
        if (form) {
            form.addEventListener('submit', function(e) {
                if (!validateForm()) {
                    e.preventDefault();
                    showFormError('Molimo ispravite greške prije spremanja');
                }
            });
        }
        
        // Add section animations
        const sections = document.querySelectorAll('.form-section');
        sections.forEach((section, index) => {
            section.style.animationDelay = `${index * 0.1}s`;
            section.style.animation = 'fadeInUp 0.5s ease forwards';
        });
        
        // Add group selector enhancement
        const groupSelect = document.getElementById('grupa');
        if (groupSelect) {
            groupSelect.addEventListener('change', function() {
                updateFormBasedOnGroup(this.value);
            });
        }
    }
    
    /**
     * Update form based on selected group
     */
    function updateFormBasedOnGroup(group) {
        const sections = document.querySelectorAll('.form-section');
        
        // Add visual feedback based on group selection
        sections.forEach(section => {
            section.classList.remove('group-mnp', 'group-p2p', 'group-tomato');
            
            if (group.includes('MNP')) {
                section.classList.add('group-mnp');
            } else if (group.includes('P2P')) {
                section.classList.add('group-p2p');
            } else if (group.includes('Tomato')) {
                section.classList.add('group-tomato');
            }
        });
    }
    
    /**
     * Validate entire form
     */
    function validateForm() {
        const requiredFields = document.querySelectorAll('input[required], select[required]');
        let isValid = true;
        
        requiredFields.forEach(field => {
            if (field.type === 'number') {
                validateBonusField(field);
            }
            
            const fieldGroup = field.closest('.form-group');
            if (fieldGroup.classList.contains('has-error') || field.value === '') {
                isValid = false;
            }
        });
        
        return isValid;
    }
    
    /**
     * Show form-level error message
     */
    function showFormError(message) {
        const existingError = document.querySelector('.form-error');
        if (existingError) {
            existingError.remove();
        }
        
        const errorElement = document.createElement('div');
        errorElement.className = 'form-error alert alert-danger';
        errorElement.innerHTML = `
            <i class="fas fa-exclamation-triangle"></i>
            <span>${message}</span>
        `;
        errorElement.style.cssText = `
            background: #fee;
            color: #c53030;
            padding: 1rem;
            border-radius: 6px;
            border: 1px solid #feb2b2;
            margin-bottom: 1rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        `;
        
        const form = document.querySelector('.modern-form');
        form.parentNode.insertBefore(errorElement, form);
        
        // Scroll to error
        errorElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
        
        // Auto-remove after 5 seconds
        setTimeout(() => {
            if (errorElement.parentNode) {
                errorElement.remove();
            }
        }, 5000);
    }
    
    /**
     * Add keyboard shortcuts
     */
    document.addEventListener('keydown', function(e) {
        // Ctrl+S to save
        if (e.ctrlKey && e.key === 's') {
            e.preventDefault();
            const submitButton = document.querySelector('button[type="submit"]');
            if (submitButton) {
                submitButton.click();
            }
        }
        
        // Escape to cancel
        if (e.key === 'Escape') {
            const cancelButton = document.getElementById('cancelButton');
            if (cancelButton) {
                cancelButton.click();
            }
        }
    });
    
    console.log('Bonus edit form initialized');
});