/**
 * listbonus.js - JavaScript functionality for bonus list view
 * AŽURIRANO za novu strukturu tablica
 */
document.addEventListener('DOMContentLoaded', function() {
    // Only run list functionality if we're on the list page
    if (!document.getElementById('dataTable')) {
        return;
    }
    
    // Apply titles to cells for tooltip display on hover
    addCellTitles();
    
    // Find the table
    const dataTable = document.getElementById('dataTable');
    
    if (!dataTable) {
        console.warn('Bonus table not found');
        return;
    }
    
    // Add bonus-specific enhancements
    addHoverEffects();
    initFilters();
    initDeleteModal();
    initExportPanel();
    addBonusAnalytics();
    initTabSwitching();
    
    /**
     * Add title attributes to all cells
     */
    function addCellTitles() {
        const dataCells = document.querySelectorAll('.data-table td');
        dataCells.forEach(cell => {
            if (cell.textContent.trim()) {
                cell.setAttribute('title', cell.textContent.trim());
            }
        });
    }
    
    /**
     * Add hover effects to rows
     */
    function addHoverEffects() {
        const rows = dataTable.querySelectorAll('tbody tr');
        
        rows.forEach(row => {
            row.addEventListener('mouseenter', function() {
                this.style.backgroundColor = 'var(--hover)';
                this.style.transform = 'scale(1.01)';
                this.style.transition = 'all 0.2s ease';
            });
            
            row.addEventListener('mouseleave', function() {
                this.style.backgroundColor = '';
                this.style.transform = '';
            });
        });
    }
    
    /**
     * Initialize filter functionality
     */
    function initFilters() {
        const grupaFilterContainer = document.getElementById('grupaFilter');
        
        if (!grupaFilterContainer) {
            console.warn('Filter container not found');
            return;
        }
        
        // Get all unique groups from the table
        const groups = new Set();
        const groupCells = dataTable.querySelectorAll('tbody tr td:nth-child(3) .badge');
        
        groupCells.forEach(cell => {
            const groupValue = cell.textContent.trim();
            if (groupValue) {
                groups.add(groupValue);
            }
        });
        
        // Clear existing content
        grupaFilterContainer.innerHTML = '';
        
        // Create filter checkboxes for each group
        groups.forEach(group => {
            const label = document.createElement('label');
            label.className = 'filter-checkbox';
            
            const checkbox = document.createElement('input');
            checkbox.type = 'checkbox';
            checkbox.setAttribute('data-filter', 'grupa');
            checkbox.setAttribute('data-value', group);
            checkbox.checked = true;
            
            label.appendChild(checkbox);
            label.appendChild(document.createTextNode(' ' + group));
            
            grupaFilterContainer.appendChild(label);
        });
        
        // Add event listeners
        const filterCheckboxes = document.querySelectorAll('.filter-checkbox input[type="checkbox"]');
        filterCheckboxes.forEach(checkbox => {
            checkbox.addEventListener('change', applyFilters);
        });
        
        // Add search functionality
        const searchInput = document.getElementById('dataSearch');
        if (searchInput) {
            searchInput.addEventListener('input', applyFilters);
        }
    }
    
    /**
     * Apply filters to the table
     */
    function applyFilters() {
        if (!dataTable) return;
        
        const rows = dataTable.querySelectorAll('tbody tr');
        const activeFilters = {};
        const filterCheckboxes = document.querySelectorAll('.filter-checkbox input:checked');
        const searchInput = document.getElementById('dataSearch');
        const searchValue = searchInput ? searchInput.value.toLowerCase() : '';
        
        // Build active filters map
        filterCheckboxes.forEach(checkbox => {
            const filterType = checkbox.getAttribute('data-filter');
            const filterValue = checkbox.getAttribute('data-value');
            
            if (!activeFilters[filterType]) {
                activeFilters[filterType] = [];
            }
            
            activeFilters[filterType].push(filterValue);
        });
        
        // Apply filtering
        let visibleCount = 0;
        
        rows.forEach(row => {
            let shouldShow = true;
            
            // Apply search filter
            if (searchValue) {
                const rowText = row.textContent.toLowerCase();
                if (!rowText.includes(searchValue)) {
                    shouldShow = false;
                }
            }
            
            // Apply group filters
            if (shouldShow && activeFilters.grupa) {
                const groupCell = row.querySelector('td:nth-child(3) .badge');
                if (groupCell) {
                    const groupValue = groupCell.textContent.trim();
                    if (!activeFilters.grupa.includes(groupValue)) {
                        shouldShow = false;
                    }
                }
            }
            
            row.style.display = shouldShow ? '' : 'none';
            
            if (shouldShow) {
                visibleCount++;
            }
        });
        
        console.log('Filter applied, visible rows:', visibleCount);
        updateAnalytics();
    }
    
    /**
     * Add analytics for bonus data
     */
    function addBonusAnalytics() {
        const analyticsContainer = document.createElement('div');
        analyticsContainer.className = 'bonus-analytics';
        analyticsContainer.innerHTML = `
            <div class="analytics-card">
                <h4><i class="fas fa-chart-bar"></i> Analitika bonus shema</h4>
                <div class="analytics-grid">
                    <div class="analytics-item">
                        <span class="analytics-label">Ukupno shema:</span>
                        <span class="analytics-value" id="totalSchemes">-</span>
                    </div>
                    <div class="analytics-item">
                        <span class="analytics-label">Prosječna osnovna naknada:</span>
                        <span class="analytics-value" id="avgBaseAmount">-</span>
                    </div>
                    <div class="analytics-item">
                        <span class="analytics-label">Najviša naknada:</span>
                        <span class="analytics-value" id="maxAmount">-</span>
                    </div>
                    <div class="analytics-item">
                        <span class="analytics-label">Broj grupa:</span>
                        <span class="analytics-value" id="groupCount">-</span>
                    </div>
                </div>
            </div>
        `;
        
        // Insert analytics before the table
        const tableCard = document.querySelector('.card:has(.data-table)');
        if (tableCard) {
            tableCard.parentNode.insertBefore(analyticsContainer, tableCard);
        }
        
        updateAnalytics();
    }
    
    /**
     * Update analytics based on visible rows
     */
    function updateAnalytics() {
        const visibleRows = Array.from(dataTable.querySelectorAll('tbody tr')).filter(row => 
            row.style.display !== 'none'
        );
        
        if (visibleRows.length === 0) {
            updateAnalyticsElement('totalSchemes', '0');
            updateAnalyticsElement('avgBaseAmount', '-');
            updateAnalyticsElement('maxAmount', '-');
            updateAnalyticsElement('groupCount', '0');
            return;
        }
        
        // Calculate analytics
        updateAnalyticsElement('totalSchemes', visibleRows.length.toString());
        
        const baseAmounts = [];
        const groups = new Set();
        
        visibleRows.forEach(row => {
            const baseAmountCell = row.querySelector('td:nth-child(4)');
            const groupCell = row.querySelector('td:nth-child(3) .badge');
            
            if (baseAmountCell) {
                const baseAmount = parseFloat(baseAmountCell.textContent);
                if (!isNaN(baseAmount)) baseAmounts.push(baseAmount);
            }
            
            if (groupCell) {
                const group = groupCell.textContent.trim();
                if (group) groups.add(group);
            }
        });
        
        // Update values
        const avgAmount = baseAmounts.length > 0 ? 
            (baseAmounts.reduce((a, b) => a + b, 0) / baseAmounts.length).toFixed(2) : '-';
        updateAnalyticsElement('avgBaseAmount', avgAmount);
        
        const maxAmount = baseAmounts.length > 0 ? 
            Math.max(...baseAmounts).toFixed(2) : '-';
        updateAnalyticsElement('maxAmount', maxAmount);
        
        updateAnalyticsElement('groupCount', groups.size.toString());
    }
    
    /**
     * Helper function to safely update analytics elements
     */
    function updateAnalyticsElement(id, value) {
        const element = document.getElementById(id);
        if (element) {
            element.textContent = value;
        }
    }
    
    /**
     * Initialize delete modal functionality
     */
    function initDeleteModal() {
        const deleteButtons = document.querySelectorAll('.action-btn.delete');
        const deleteModal = document.getElementById('deleteModal');
        const deleteId = document.getElementById('deleteId');
        const deleteType = document.getElementById('deleteType');
        const deleteName = document.getElementById('deleteName');
        const modalClose = document.querySelector('.modal-close');
        const modalCancel = document.querySelector('.modal-cancel');
        
        if (!deleteModal) {
            console.warn('Delete modal not found');
            return;
        }
        
        deleteButtons.forEach(button => {
            button.addEventListener('click', function() {
                const id = this.getAttribute('data-id');
                const type = this.getAttribute('data-type');
                const name = this.getAttribute('data-name');
                
                if (deleteId) deleteId.value = id || '';
                if (deleteType) deleteType.value = type || '';
                if (deleteName) deleteName.textContent = name || 'Unknown';
                
                deleteModal.style.display = 'block';
            });
        });
        
        // Close modal events
        if (modalClose) {
            modalClose.addEventListener('click', function() {
                deleteModal.style.display = 'none';
            });
        }
        
        if (modalCancel) {
            modalCancel.addEventListener('click', function() {
                deleteModal.style.display = 'none';
            });
        }
        
        // Close modal when clicking outside
        window.addEventListener('click', function(event) {
            if (event.target === deleteModal) {
                deleteModal.style.display = 'none';
            }
        });
    }
    
    /**
     * Initialize export panel functionality
     */
    function initExportPanel() {
        const exportToggle = document.getElementById('exportToggle');
        const exportPanel = document.getElementById('exportPanel');
        const closeExport = document.getElementById('closeExport');
        
        if (!exportPanel || !exportToggle) {
            console.warn('Export panel elements not found');
            return;
        }
        
        exportToggle.addEventListener('click', function() {
            exportPanel.classList.toggle('visible');
        });
        
        if (closeExport) {
            closeExport.addEventListener('click', function() {
                exportPanel.classList.remove('visible');
            });
        }
    }
    
    /**
     * Add tab switching functionality - AŽURIRANO za novu strukturu
     */
    function initTabSwitching() {
        const tabs = document.querySelectorAll('.nav-tab');
        
        console.log('Found tabs:', tabs.length);
        
        tabs.forEach((tab, index) => {
            console.log(`Tab ${index}:`, tab.href);
            
            tab.addEventListener('click', function(e) {
                e.preventDefault();
                
                console.log('Tab clicked:', tab.href);
                
                // Get the target type from URL
                const url = new URL(tab.href);
                const targetType = url.searchParams.get('type');
                
                console.log('Target type:', targetType);
                console.log('Current URL:', window.location.href);
                
                if (targetType) {
                    // Add loading state to clicked tab
                    const originalContent = tab.innerHTML;
                    tab.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Loading...';
                    tab.classList.add('loading');
                    
                    // Show loading state on table
                    const tableContainer = document.querySelector('.table-responsive');
                    if (tableContainer) {
                        tableContainer.style.opacity = '0.5';
                        tableContainer.style.pointerEvents = 'none';
                    }
                    
                    // Navigate to new type with proper URL
                    const currentUrl = new URL(window.location);
                    currentUrl.searchParams.set('type', targetType);
                    
                    const finalUrl = currentUrl.toString();
                    console.log('Navigating to:', finalUrl);
                    
                    // Use window.location.href for proper page reload with new data
                    window.location.href = finalUrl;
                } else {
                    console.log('No target type found, using direct navigation');
                    // Fallback to direct navigation
                    window.location.href = tab.href;
                }
            });
        });
    }
    
    /**
     * Add keyboard shortcuts for list
     */
    document.addEventListener('keydown', function(e) {
        // Ctrl+F to focus search
        if (e.ctrlKey && e.key === 'f') {
            e.preventDefault();
            const searchInput = document.getElementById('dataSearch');
            if (searchInput) {
                searchInput.focus();
                searchInput.select();
            }
        }
        
        // Escape to clear search
        if (e.key === 'Escape') {
            const searchInput = document.getElementById('dataSearch');
            if (searchInput && searchInput.value) {
                searchInput.value = '';
                applyFilters();
            }
        }
    });
    
    // Listen for theme changes
    document.addEventListener('themechange', function(event) {
        console.log('Theme changed in bonus list page:', event.detail.theme);
        
        // Update analytics colors based on theme
        const analyticsItems = document.querySelectorAll('.analytics-item');
        analyticsItems.forEach(item => {
            item.style.transition = 'all 0.3s ease';
        });
    });
    
    console.log('Bonus list functionality initialized');
});