/**
 * Enhanced Osoba List with Feedback System - CSRF FIXED VERSION
 * NOTE: This script depends on table.js and should be loaded AFTER it
 */
document.addEventListener('DOMContentLoaded', function() {
    console.log('List page DOM loaded, initializing...');
    
    // Find the table (already initialized by table.js)
    const dataTable = document.getElementById('dataTable');
    
    if (!dataTable) {
        console.warn('Person table not found');
        return;
    }
    
    // Initialize enhanced functionality
    initializeFeedbackInteractions();
    initializeFilters();
    initializeDeleteModal();
    initializeExportPanel();
    initializeModals();
    addHoverEffects();
    updateBadgeColors();
    
    console.log('Enhanced Osoba List initialized');
});

/**
 * Initialize feedback interactions - FIXED VERSION
 */
function initializeFeedbackInteractions() {
    console.log('Initializing feedback interactions...');
    
    // Quick feedback buttons
    const feedbackButtons = document.querySelectorAll('.action-btn.feedback');
    console.log('Found feedback buttons:', feedbackButtons.length);
    
    feedbackButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            console.log('Quick feedback button clicked!');
            const personName = this.getAttribute('data-person-name');
            console.log('Person name:', personName);
            openQuickFeedbackModal(personName);
        });
    });
    
    // View feedbacks buttons
    const viewButtons = document.querySelectorAll('.action-btn.view-feedbacks, .view-all-btn');
    console.log('Found view feedback buttons:', viewButtons.length);
    
    viewButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            console.log('View feedbacks button clicked!');
            const personName = this.getAttribute('data-person-name');
            console.log('Person name:', personName);
            openViewFeedbacksModal(personName);
        });
    });
    
    // Add feedback buttons in no-feedback cells
    const addButtons = document.querySelectorAll('.add-feedback-btn');
    console.log('Found add feedback buttons:', addButtons.length);
    
    addButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            console.log('Add feedback button clicked!');
            const personName = this.getAttribute('data-person-name');
            console.log('Person name:', personName);
            openQuickFeedbackModal(personName);
        });
    });
    
    // Feedback expand buttons
    const expandButtons = document.querySelectorAll('.feedback-expand-btn');
    console.log('Found expand buttons:', expandButtons.length);
    
    expandButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            console.log('Expand feedback button clicked!');
            const personName = this.getAttribute('data-person-name');
            console.log('Person name:', personName);
            openViewFeedbacksModal(personName);
        });
    });
    
    console.log('Feedback interactions initialized');
}

/**
 * Open quick feedback modal - IMPROVED VERSION
 */
function openQuickFeedbackModal(personName) {
    console.log('Opening quick feedback modal for:', personName);
    
    const modal = document.getElementById('quickFeedbackModal');
    console.log('Quick feedback modal element:', modal);
    
    if (!modal) {
        console.error('Quick feedback modal not found!');
        alert('Greška: Modal za dodavanje feedback-a nije pronađen.');
        return;
    }
    
    if (!personName) {
        console.error('Person name not provided!');
        alert('Greška: Ime osobe nije pronađeno.');
        return;
    }
    
    // Set person name
    const personNameElement = document.getElementById('quickFeedbackPersonName');
    const personInputElement = document.getElementById('quickFeedbackPersonInput');
    
    if (personNameElement) personNameElement.textContent = personName;
    if (personInputElement) personInputElement.value = personName;
    
    // Clear form
    const form = document.getElementById('quickFeedbackForm');
    if (form) {
        form.reset();
        console.log('Quick feedback form reset');
    }
    
    const counter = document.getElementById('quickFeedbackCounter');
    if (counter) {
        counter.textContent = '0/2000';
        counter.classList.remove('warning', 'danger');
    }
    
    const preview = document.getElementById('quickRatingPreview');
    if (preview) {
        updateStarRating(0, preview);
    }
    
    console.log('Showing quick feedback modal...');
    showModal(modal);
}

/**
 * Open view feedbacks modal - IMPROVED VERSION
 */
function openViewFeedbacksModal(personName) {
    console.log('Opening view feedbacks modal for:', personName);
    
    const modal = document.getElementById('viewFeedbacksModal');
    console.log('View feedbacks modal element:', modal);
    
    if (!modal) {
        console.error('View feedbacks modal not found!');
        alert('Greška: Modal za pregled feedback-a nije pronađen.');
        return;
    }
    
    if (!personName) {
        console.error('Person name not provided!');
        alert('Greška: Ime osobe nije pronađeno.');
        return;
    }
    
    const personNameElement = document.getElementById('viewFeedbacksPersonName');
    if (personNameElement) {
        personNameElement.textContent = personName;
    }
    
    console.log('Showing view feedbacks modal...');
    showModal(modal);
    
    // Load feedbacks
    loadFeedbacksTimeline(personName);
}

/**
 * Load feedbacks timeline - CSRF FIXED VERSION
 */
function loadFeedbacksTimeline(personName) {
    console.log('Loading feedbacks timeline for:', personName);
    
    const container = document.getElementById('feedbacksTimelineContainer');
    if (!container) {
        console.error('Feedbacks timeline container not found!');
        return;
    }
    
    // Show loading
    container.innerHTML = '<div class="loading-state"><i class="fas fa-spinner fa-spin"></i> Učitavanje feedbackova...</div>';
    
    // Use fetchWithCsrf instead of fetch
    window.fetchWithCsrf(`/saleosoba/feedbacks?person_name=${encodeURIComponent(personName)}`, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
            'Content-Type': 'application/json'
        }
    })
    .then(response => {
        console.log('Fetch response status:', response.status);
        return response.json();
    })
    .then(data => {
        console.log('Feedbacks data received:', data);
        if (data.feedbacks && data.feedbacks.length > 0) {
            renderFeedbacksTimeline(data.feedbacks, container, data.can_manage);
        } else {
            container.innerHTML = '<div class="empty-state"><i class="fas fa-comment-slash"></i><p>Nema feedbackova za ovu osobu</p></div>';
        }
    })
    .catch(error => {
        console.error('Error loading feedbacks:', error);
        container.innerHTML = '<div class="error-state"><i class="fas fa-exclamation-triangle"></i><p>Greška pri učitavanju feedbackova</p></div>';
    });
}

/**
 * Render feedbacks timeline
 */
function renderFeedbacksTimeline(feedbacks, container, canManage) {
    console.log('Rendering feedbacks timeline:', feedbacks.length, 'feedbacks');
    
    const timeline = document.createElement('div');
    timeline.className = 'feedbacks-timeline';
    
    feedbacks.forEach(feedback => {
        const timelineItem = document.createElement('div');
        timelineItem.className = 'timeline-item';
        timelineItem.setAttribute('data-feedback-id', feedback.id);
        
        const date = new Date(feedback.datum);
        const formattedDate = date.toLocaleDateString('hr-HR', {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
        
        let ratingHtml = '';
        if (feedback.ocjena) {
            const rating = parseFloat(feedback.ocjena);
            const ratingClass = rating >= 4 ? 'excellent' : (rating >= 3 ? 'good' : 'poor');
            const stars = Array.from({length: 5}, (_, i) => 
                `<i class="fas fa-star ${i < rating ? 'star-filled' : 'star-empty'}"></i>`
            ).join('');
            
            ratingHtml = `
                <div class="timeline-rating">
                    <span class="timeline-rating-value rating-${ratingClass}">${rating.toFixed(2)}/5.00</span>
                    <div class="timeline-stars">${stars}</div>
                </div>
            `;
        }
        
        let actionsHtml = '';
        if (canManage) {
            actionsHtml = `
                <div class="timeline-actions">
                    <button type="button" class="timeline-edit-btn" data-feedback-id="${feedback.id}">
                        <i class="fas fa-edit"></i> Uredi
                    </button>
                    <button type="button" class="timeline-delete-btn" data-feedback-id="${feedback.id}">
                        <i class="fas fa-trash"></i> Obriši
                    </button>
                </div>
            `;
        }
        
        timelineItem.innerHTML = `
            <div class="timeline-content">
                <div class="timeline-header">
                    <div class="timeline-date">${formattedDate}</div>
                    ${actionsHtml}
                </div>
                ${feedback.feedback ? `<div class="timeline-feedback">${escapeHtml(feedback.feedback).replace(/\n/g, '<br>')}</div>` : ''}
                ${ratingHtml}
                <div class="timeline-meta">
                    <i class="fas fa-user"></i>
                    <span>${escapeHtml(feedback.created_by)}</span>
                </div>
            </div>
        `;
        
        timeline.appendChild(timelineItem);
    });
    
    container.innerHTML = '';
    container.appendChild(timeline);
    
    // Add event listeners to timeline action buttons
    container.querySelectorAll('.timeline-edit-btn').forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            const feedbackId = this.getAttribute('data-feedback-id');
            editTimelineFeedback(feedbackId);
        });
    });
    
    container.querySelectorAll('.timeline-delete-btn').forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            const feedbackId = this.getAttribute('data-feedback-id');
            deleteTimelineFeedback(feedbackId);
        });
    });
}

/**
 * Save quick feedback - CSRF FIXED VERSION
 */
function saveQuickFeedback() {
    console.log('Saving quick feedback...');
    
    const form = document.getElementById('quickFeedbackForm');
    if (!form) {
        console.error('Quick feedback form not found!');
        return;
    }
    
    const formData = new FormData(form);
    
    // Validate form
    const feedback = formData.get('feedback');
    const ocjena = formData.get('ocjena');
    
    console.log('Quick feedback data:', { feedback, ocjena });
    
    if (!feedback && !ocjena) {
        showNotification('Morate unijeti feedback ili ocjenu', 'error');
        return;
    }
    
    // Show loading
    const saveBtn = document.getElementById('saveQuickFeedbackBtn');
    if (!saveBtn) {
        console.error('Save quick feedback button not found!');
        return;
    }
    
    const originalText = saveBtn.innerHTML;
    saveBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Spremanje...';
    saveBtn.disabled = true;
    
    // Use fetchWithCsrf instead of fetch
    window.fetchWithCsrf('/saleosoba/add-feedback', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        console.log('Quick feedback save response:', data);
        if (data.success) {
            showNotification('Feedback je uspješno dodan', 'success');
            closeModal(document.getElementById('quickFeedbackModal'));
            // Refresh the page to show updated data
            setTimeout(() => {
                location.reload();
            }, 1000);
        } else {
            showNotification(data.error || 'Greška pri dodavanju feedback-a', 'error');
        }
    })
    .catch(error => {
        console.error('Error saving quick feedback:', error);
        showNotification('Greška pri dodavanju feedback-a', 'error');
    })
    .finally(() => {
        saveBtn.innerHTML = originalText;
        saveBtn.disabled = false;
    });
}

/**
 * Edit timeline feedback - CSRF FIXED VERSION
 */
function editTimelineFeedback(feedbackId) {
    console.log('Editing timeline feedback:', feedbackId);
    
    // Use fetchWithCsrf instead of fetch
    window.fetchWithCsrf(`/saleosoba/feedback/${feedbackId}`, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
            'Content-Type': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        console.log('Edit feedback data:', data);
        if (data.feedback) {
            openEditFeedbackModal(data.feedback);
        } else {
            showNotification('Greška pri učitavanju feedback-a', 'error');
        }
    })
    .catch(error => {
        console.error('Error loading feedback:', error);
        showNotification('Greška pri učitavanju feedback-a', 'error');
    });
}

/**
 * Delete timeline feedback - CSRF FIXED VERSION
 */
function deleteTimelineFeedback(feedbackId) {
    console.log('Deleting timeline feedback:', feedbackId);
    
    if (!confirm('Jeste li sigurni da želite obrisati ovaj feedback?')) {
        return;
    }
    
    const formData = new FormData();
    formData.append('feedback_id', feedbackId);
    
    // Use fetchWithCsrf instead of fetch
    window.fetchWithCsrf('/saleosoba/delete-feedback', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        console.log('Delete feedback response:', data);
        if (data.success) {
            showNotification('Feedback je uspješno obrisan', 'success');
            // Remove from timeline
            const timelineItem = document.querySelector(`[data-feedback-id="${feedbackId}"]`);
            if (timelineItem) {
                timelineItem.remove();
            }
        } else {
            showNotification(data.error || 'Greška pri brisanju feedback-a', 'error');
        }
    })
    .catch(error => {
        console.error('Error deleting feedback:', error);
        showNotification('Greška pri brisanju feedback-a', 'error');
    });
}

/**
 * Open edit feedback modal
 */
function openEditFeedbackModal(feedbackData) {
    console.log('Opening edit feedback modal:', feedbackData);
    
    const modal = document.getElementById('editFeedbackModal');
    if (!modal) {
        console.error('Edit feedback modal not found!');
        return;
    }
    
    // Populate form
    const feedbackIdInput = document.getElementById('editFeedbackId');
    const feedbackTextarea = document.getElementById('editFeedback');
    const ocjenaInput = document.getElementById('editOcjena');
    
    if (feedbackIdInput) feedbackIdInput.value = feedbackData.id;
    if (feedbackTextarea) feedbackTextarea.value = feedbackData.feedback || '';
    if (ocjenaInput) ocjenaInput.value = feedbackData.ocjena || '';
    
    // Update character counter
    const counter = document.getElementById('editFeedbackCounter');
    if (counter && feedbackTextarea) {
        updateCharacterCounter(feedbackTextarea, counter, 2000);
    }
    
    // Update rating preview
    const preview = document.getElementById('editRatingPreview');
    if (preview) {
        updateStarRating(feedbackData.ocjena || 0, preview);
    }
    
    showModal(modal);
}

/**
 * Update feedback from modal - CSRF FIXED VERSION
 */
function updateFeedback() {
    console.log('Updating feedback...');
    
    const form = document.getElementById('editFeedbackForm');
    if (!form) {
        console.error('Edit feedback form not found!');
        return;
    }
    
    const formData = new FormData(form);
    
    // Validate form
    const feedback = formData.get('feedback');
    const ocjena = formData.get('ocjena');
    
    if (!feedback && !ocjena) {
        showNotification('Morate unijeti feedback ili ocjenu', 'error');
        return;
    }
    
    // Show loading
    const updateBtn = document.getElementById('updateFeedbackBtn');
    if (!updateBtn) {
        console.error('Update feedback button not found!');
        return;
    }
    
    const originalText = updateBtn.innerHTML;
    updateBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Ažuriranje...';
    updateBtn.disabled = true;
    
    // Use fetchWithCsrf instead of fetch
    window.fetchWithCsrf('/saleosoba/update-feedback', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        console.log('Update feedback response:', data);
        if (data.success) {
            showNotification('Feedback je uspješno ažuriran', 'success');
            closeModal(document.getElementById('editFeedbackModal'));
            // Refresh the timeline
            const personName = document.getElementById('viewFeedbacksPersonName').textContent;
            if (personName) {
                loadFeedbacksTimeline(personName);
            }
        } else {
            showNotification(data.error || 'Greška pri ažuriranju feedback-a', 'error');
        }
    })
    .catch(error => {
        console.error('Error updating feedback:', error);
        showNotification('Greška pri ažuriranju feedback-a', 'error');
    })
    .finally(() => {
        updateBtn.innerHTML = originalText;
        updateBtn.disabled = false;
    });
}

/**
 * Initialize enhanced filters
 */
function initializeFilters() {
    // Initialize organization unit filters
    initOrgFilters();
    
    // Initialize feedback filters
    initFeedbackFilters();
    
    // Rating filters
    const ocjenaCheckboxes = document.querySelectorAll('input[data-filter="ocjena"]');
    ocjenaCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', applyFilters);
    });
    
    // Status filters
    const statusCheckboxes = document.querySelectorAll('input[data-filter="status"]');
    statusCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', applyFilters);
    });
}

/**
 * Initialize feedback filters
 */
function initFeedbackFilters() {
    const feedbackCheckboxes = document.querySelectorAll('input[data-filter="feedback"]');
    feedbackCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', applyFilters);
    });
}

/**
 * Initialize organization unit filters
 */
function initOrgFilters() {
    const orgFilterContainer = document.getElementById('orgFilter');
    if (!orgFilterContainer) return;
    
    // Get all unique org units from the table
    const orgUnits = new Set();
    const orgBadges = document.querySelectorAll('.org-badge');
    
    orgBadges.forEach(badge => {
        const orgText = badge.textContent.trim();
        if (orgText) {
            orgUnits.add(orgText);
        }
    });
    
    // Clear existing content
    orgFilterContainer.innerHTML = '';
    
    // Create filter checkboxes for each org unit
    orgUnits.forEach(orgUnit => {
        const label = document.createElement('label');
        label.className = 'filter-checkbox';
        
        const checkbox = document.createElement('input');
        checkbox.type = 'checkbox';
        checkbox.setAttribute('data-filter', 'org');
        checkbox.setAttribute('data-value', orgUnit);
        
        label.appendChild(checkbox);
        label.appendChild(document.createTextNode(' ' + orgUnit));
        
        orgFilterContainer.appendChild(label);
    });
    
    // Add event listeners to the new checkboxes
    const filterCheckboxes = orgFilterContainer.querySelectorAll('input[type="checkbox"]');
    filterCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', applyFilters);
    });
}

/**
 * Apply filters to the table
 */
function applyFilters() {
    const dataTable = document.getElementById('dataTable');
    if (!dataTable) return;
    
    const rows = dataTable.querySelectorAll('tbody tr');
    const activeFilters = {};
    const filterCheckboxes = document.querySelectorAll('.filter-checkbox input:checked');
    
    // Build active filters map
    filterCheckboxes.forEach(checkbox => {
        const filterType = checkbox.getAttribute('data-filter');
        const filterValue = checkbox.getAttribute('data-value');
        
        if (!activeFilters[filterType]) {
            activeFilters[filterType] = [];
        }
        
        activeFilters[filterType].push(filterValue);
    });
    
    // Show all rows if no filters are active
    if (Object.keys(activeFilters).length === 0) {
        rows.forEach(row => {
            row.style.display = '';
        });
        return;
    }
    
    // Apply filtering
    let visibleCount = 0;
    
    rows.forEach(row => {
        let shouldShow = true;
        
        // Check each filter type
        for (const filterType in activeFilters) {
            let matches = false;
            
            if (filterType === 'org') {
                const orgBadge = row.querySelector('.org-badge');
                if (orgBadge) {
                    const orgText = orgBadge.textContent.trim();
                    matches = activeFilters[filterType].some(value => orgText === value);
                }
            }
            else if (filterType === 'ocjena') {
                const ocjenaBadge = row.querySelector('.ocjena-badge');
                if (ocjenaBadge) {
                    const ocjenaText = ocjenaBadge.textContent.trim();
                    const ocjenaValue = parseFloat(ocjenaText) || 0;
                    
                    matches = activeFilters[filterType].some(value => {
                        const [min, max] = value.split('-').map(Number);
                        return ocjenaValue >= min && ocjenaValue <= max;
                    });
                }
            }
            else if (filterType === 'feedback') {
                const feedbackStatus = row.getAttribute('data-feedback');
                matches = activeFilters[filterType].some(value => feedbackStatus === value);
            }
            else if (filterType === 'status') {
                const rowStatus = row.getAttribute('data-status');
                matches = activeFilters[filterType].some(value => rowStatus === value);
            }
            
            if (!matches) {
                shouldShow = false;
                break;
            }
        }
        
        row.style.display = shouldShow ? '' : 'none';
        
        if (shouldShow) {
            visibleCount++;
        }
    });
    
    console.log('Filter applied, visible rows:', visibleCount);
}

/**
 * Initialize delete modal functionality
 */
function initializeDeleteModal() {
    const deleteButtons = document.querySelectorAll('.action-btn.delete');
    const deleteModal = document.getElementById('deleteModal');
    
    if (!deleteModal) return;
    
    const deleteId = document.getElementById('deleteId');
    const deleteName = document.getElementById('deleteName');
    
    deleteButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            const id = this.getAttribute('data-id');
            const name = this.getAttribute('data-name');
            
            if (deleteId) deleteId.value = id;
            if (deleteName) deleteName.textContent = name;
            showModal(deleteModal);
        });
    });
}

/**
 * Initialize export panel
 */
function initializeExportPanel() {
    const exportToggle = document.getElementById('exportToggle');
    const exportPanel = document.getElementById('exportPanel');
    const closeExport = document.getElementById('closeExport');
    
    if (!exportPanel || !exportToggle) return;
    
    exportToggle.addEventListener('click', function() {
        exportPanel.classList.toggle('visible');
    });
    
    if (closeExport) {
        closeExport.addEventListener('click', function() {
            exportPanel.classList.remove('visible');
        });
    }
}

/**
 * Initialize modal functionality - IMPROVED VERSION
 */
function initializeModals() {
    console.log('Initializing modals...');
    
    // Character counters for modal textareas
    initializeCharacterCounters();
    
    // Rating inputs
    initializeRatingInputs();
    
    // Modal close buttons
    document.querySelectorAll('.modal-close, .modal-cancel').forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            console.log('Modal close button clicked');
            const modal = this.closest('.modal');
            if (modal) {
                closeModal(modal);
            }
        });
    });
    
    // Close modal when clicking outside
    document.querySelectorAll('.modal').forEach(modal => {
        modal.addEventListener('click', function(e) {
            // Check if click is directly on modal background
            if (e.target === this) {
                console.log('Modal background clicked');
                closeModal(this);
            }
        });
        
        // Prevent closing modal when clicking on modal content
        const modalContent = modal.querySelector('.modal-content');
        if (modalContent) {
            modalContent.addEventListener('click', function(e) {
                e.stopPropagation();
            });
        }
    });
    
    // ESC key for closing modals
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            const openModal = document.querySelector('.modal.show');
            if (openModal) {
                closeModal(openModal);
            }
        }
    });
    
    // Save buttons
    const saveQuickFeedbackBtn = document.getElementById('saveQuickFeedbackBtn');
    if (saveQuickFeedbackBtn) {
        saveQuickFeedbackBtn.addEventListener('click', function(e) {
            e.preventDefault();
            console.log('Save quick feedback button clicked!');
            saveQuickFeedback();
        });
    }
    
    const updateFeedbackBtn = document.getElementById('updateFeedbackBtn');
    if (updateFeedbackBtn) {
        updateFeedbackBtn.addEventListener('click', function(e) {
            e.preventDefault();
            console.log('Update feedback button clicked!');
            updateFeedback();
        });
    }
    
    const closeFeedbacksViewBtn = document.getElementById('closeFeedbacksViewBtn');
    if (closeFeedbacksViewBtn) {
        closeFeedbacksViewBtn.addEventListener('click', function(e) {
            e.preventDefault();
            closeModal(document.getElementById('viewFeedbacksModal'));
        });
    }
    
    const modals = document.querySelectorAll('.modal');
    console.log(`Found ${modals.length} modals in list page`);
}

/**
 * Initialize character counters
 */
function initializeCharacterCounters() {
    const textareas = [
        { element: 'quickFeedback', counter: 'quickFeedbackCounter', max: 2000 },
        { element: 'editFeedback', counter: 'editFeedbackCounter', max: 2000 }
    ];
    
    textareas.forEach(textarea => {
        const element = document.getElementById(textarea.element);
        const counter = document.getElementById(textarea.counter);
        
        if (element && counter) {
            element.addEventListener('input', function() {
                updateCharacterCounter(this, counter, textarea.max);
            });
            console.log(`Character counter initialized for ${textarea.element}`);
        }
    });
}

/**
 * Update character counter
 */
function updateCharacterCounter(element, counter, maxLength) {
    const currentLength = element.value.length;
    
    counter.textContent = `${currentLength}/${maxLength}`;
    
    // Add warning/danger classes based on character count
    counter.classList.remove('warning', 'danger');
    
    if (currentLength > maxLength * 0.9) {
        counter.classList.add('danger');
    } else if (currentLength > maxLength * 0.8) {
        counter.classList.add('warning');
    }
    
    // Prevent typing if max length exceeded
    if (currentLength > maxLength) {
        element.value = element.value.substring(0, maxLength);
        counter.textContent = `${maxLength}/${maxLength}`;
        counter.classList.add('danger');
    }
}

/**
 * Initialize rating inputs
 */
function initializeRatingInputs() {
    const ratingInputs = [
        { input: 'quickOcjena', preview: 'quickRatingPreview' },
        { input: 'editOcjena', preview: 'editRatingPreview' }
    ];
    
    ratingInputs.forEach(rating => {
        const input = document.getElementById(rating.input);
        const preview = document.getElementById(rating.preview);
        
        if (input && preview) {
            // Update stars on input change
            input.addEventListener('input', function() {
                updateStarRating(this.value, preview);
            });
            
            // Make stars clickable
            const stars = preview.querySelectorAll('.stars i');
            stars.forEach((star, index) => {
                star.addEventListener('click', function() {
                    const rating = index + 1;
                    input.value = rating.toFixed(1);
                    updateStarRating(rating, preview);
                });
                
                star.addEventListener('mouseenter', function() {
                    const rating = index + 1;
                    updateStarRating(rating, preview, true);
                });
                
                star.addEventListener('mouseleave', function() {
                    updateStarRating(input.value, preview);
                });
            });
            console.log(`Rating input initialized for ${rating.input}`);
        }
    });
}

/**
 * Update star rating display
 */
function updateStarRating(rating, previewElement, isHover = false) {
    if (!previewElement) return;
    
    const stars = previewElement.querySelectorAll('.stars i');
    const ratingValue = parseFloat(rating) || 0;
    
    stars.forEach((star, index) => {
        star.classList.remove('star-filled', 'star-empty', 'star-hover');
        
        if (isHover) {
            if (index < ratingValue) {
                star.classList.add('star-hover');
            } else {
                star.classList.add('star-empty');
            }
        } else {
            if (index < ratingValue) {
                star.classList.add('star-filled');
            } else {
                star.classList.add('star-empty');
            }
        }
    });
}

/**
 * Add hover effects to rows and elements
 */
function addHoverEffects() {
    const rows = document.querySelectorAll('#dataTable tbody tr');
    
    rows.forEach(row => {
        const personIcon = row.querySelector('.person-icon');
        const ocjenaBadge = row.querySelector('.ocjena-badge');
        const orgBadge = row.querySelector('.org-badge');
        
        row.addEventListener('mouseenter', function() {
            if (personIcon) personIcon.classList.add('icon-highlight');
            if (ocjenaBadge) ocjenaBadge.classList.add('badge-highlight');
            if (orgBadge) orgBadge.classList.add('badge-highlight');
        });
        
        row.addEventListener('mouseleave', function() {
            if (personIcon) personIcon.classList.remove('icon-highlight');
            if (ocjenaBadge) ocjenaBadge.classList.remove('badge-highlight');
            if (orgBadge) orgBadge.classList.remove('badge-highlight');
        });
    });
}

/**
 * Update badge colors and data attributes
 */
function updateBadgeColors() {
    // Update ocjena badges
    const ocjenaBadges = document.querySelectorAll('.ocjena-badge');
    ocjenaBadges.forEach(badge => {
        const ocjenaText = badge.textContent.trim();
        const ocjenaValue = parseFloat(ocjenaText);
        badge.setAttribute('data-ocjena', ocjenaValue || 0);
    });
    
    // Update org badges
    const orgBadges = document.querySelectorAll('.org-badge');
    orgBadges.forEach(badge => {
        const orgText = badge.textContent.trim();
        badge.setAttribute('data-org', orgText);
    });
}

/**
 * Show modal - COMPLETELY FIXED VERSION
 */
function showModal(modal) {
    if (!modal) {
        console.error('Cannot show modal - modal element is null');
        return;
    }
    
    console.log('Showing modal:', modal.id);
    
    // Remove any existing styles that might interfere
    modal.style.removeProperty('height');
    modal.style.removeProperty('visibility');
    modal.style.removeProperty('opacity');
    
    // DODAJ OVO - postavi display na flex umjesto da ostane none
    modal.style.display = 'flex';
    
    // Add show class that uses flex
    modal.classList.add('show');
    
    // Force reflow
    modal.offsetHeight;
    
    // Prevent body scroll
    document.body.style.overflow = 'hidden';
    
    // Fix textarea height in modal
    const textareas = modal.querySelectorAll('textarea');
    textareas.forEach(textarea => {
        textarea.style.height = 'auto';
        textarea.style.height = Math.max(120, textarea.scrollHeight) + 'px';
    });
    
    // Focus first input
    const firstInput = modal.querySelector('input:not([type="hidden"]), textarea, select');
    if (firstInput) {
        setTimeout(() => {
            firstInput.focus();
            console.log('Focused first input:', firstInput);
        }, 200);
    }
    
    console.log('Modal should now be visible');
}

/**
 * Close modal - IMPROVED VERSION
 */
function closeModal(modal) {
    if (!modal) return;
    
    console.log('Closing modal:', modal.id);
    
    // Remove show class
    modal.classList.remove('show');
    
    // DODAJ OVO - vrati display na none
    modal.style.display = 'none';
    
    // Restore body scroll
    document.body.style.overflow = '';
    
    // Clear any form data if needed
    const form = modal.querySelector('form');
    if (form && (modal.id === 'quickFeedbackModal' || modal.id === 'addFeedbackModal')) {
        // Reset form only for add modals, not edit modals
        setTimeout(() => {
            form.reset();
            // Reset character counters
            const counters = modal.querySelectorAll('.char-counter');
            counters.forEach(counter => {
                counter.textContent = '0/2000';
                counter.classList.remove('warning', 'danger');
            });
        }, 300);
    }
}

/**
 * Show notification
 */
function showNotification(message, type = 'success') {
    console.log('Showing notification:', message, type);
    
    let notification = document.getElementById('notification');
    
    // Create notification if it doesn't exist
    if (!notification) {
        const notificationHtml = `
            <div id="notification" class="notification">
                <div class="notification-content">
                    <i class="notification-icon"></i>
                    <span class="notification-message"></span>
                </div>
            </div>
        `;
        document.body.insertAdjacentHTML('beforeend', notificationHtml);
        notification = document.getElementById('notification');
    }
    
    if (!notification) return;
    
    const icon = type === 'success' ? 'fas fa-check-circle' : 'fas fa-exclamation-triangle';
    
    notification.className = `notification ${type}`;
    
    const iconElement = notification.querySelector('.notification-icon');
    const messageElement = notification.querySelector('.notification-message');
    
    if (iconElement) iconElement.className = `notification-icon ${icon}`;
    if (messageElement) messageElement.textContent = message;
    
    notification.classList.add('show');
    
    setTimeout(() => {
        notification.classList.remove('show');
    }, 4000);
}

/**
 * Escape HTML to prevent XSS
 */
function escapeHtml(text) {
    if (!text) return '';
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return text.replace(/[&<>"']/g, function(m) { return map[m]; });
}

/**
 * Refresh table row after feedback update
 */
function refreshTableRow(personName, newFeedbackData) {
    const rows = document.querySelectorAll(`tr[data-person-name="${personName}"]`);
    
    rows.forEach(row => {
        // Update feedback cell
        const feedbackCell = row.querySelector('.feedback-cell');
        if (feedbackCell && newFeedbackData.latest_feedback) {
            const feedback = newFeedbackData.latest_feedback;
            const feedbackHtml = `
                <div class="feedback-preview">
                    <div class="feedback-text">
                        ${escapeHtml(feedback.feedback ? feedback.feedback.substring(0, 50) : '')}
                        ${feedback.feedback && feedback.feedback.length > 50 ? '...' : ''}
                    </div>
                    <div class="feedback-meta">
                        <small class="feedback-date">
                            <i class="fas fa-calendar"></i>
                            ${new Date(feedback.datum).toLocaleDateString('hr-HR')}
                        </small>
                        <small class="feedback-author">
                            <i class="fas fa-user"></i>
                            ${escapeHtml(feedback.created_by)}
                        </small>
                    </div>
                </div>
                <button type="button" class="feedback-expand-btn" 
                        data-person-name="${escapeHtml(personName)}">
                    <i class="fas fa-expand-alt"></i>
                </button>
            `;
            feedbackCell.innerHTML = feedbackHtml;
        }
        
        // Update rating cell
        const ratingCell = row.querySelector('.rating-cell');
        if (ratingCell && newFeedbackData.feedback_stats && newFeedbackData.feedback_stats.avg_ocjena) {
            const avgRating = newFeedbackData.feedback_stats.avg_ocjena;
            const ratingClass = avgRating >= 4 ? 'high-rating' : (avgRating >= 2.5 ? 'medium-rating' : 'low-rating');
            const stars = Array.from({length: 5}, (_, i) => 
                `<i class="fas fa-star ${i < avgRating ? 'star-filled' : 'star-empty'}"></i>`
            ).join('');
            
            const ratingHtml = `
                <span class="ocjena-badge ${ratingClass}">
                    ${avgRating.toFixed(2)}
                </span>
                <div class="star-rating-mini">
                    ${stars}
                </div>
            `;
            ratingCell.innerHTML = ratingHtml;
        }
        
        // Update feedback count
        const countCell = row.querySelector('.feedback-count-cell');
        if (countCell && newFeedbackData.feedback_stats) {
            const totalFeedbacks = newFeedbackData.feedback_stats.total_feedbacks || 0;
            const countHtml = totalFeedbacks > 0 ? `
                <span class="feedback-count-badge">
                    <i class="fas fa-comments"></i>
                    ${totalFeedbacks}
                </span>
                ${totalFeedbacks > 1 ? `
                    <button type="button" class="view-all-btn" 
                            data-person-name="${escapeHtml(personName)}">
                        <i class="fas fa-eye"></i>
                    </button>
                ` : ''}
            ` : '<span class="no-feedback-count">0</span>';
            
            countCell.innerHTML = countHtml;
        }
    });
    
    // Re-initialize event listeners for updated elements
    initializeFeedbackInteractions();
}

// Global functions for timeline actions
window.editTimelineFeedback = editTimelineFeedback;
window.deleteTimelineFeedback = deleteTimelineFeedback;

// Listen for theme changes
document.addEventListener('themechange', function(event) {
    console.log('Theme changed in osoba list page:', event.detail.theme);
    updateBadgeColors();
});