// Real Data Sales Admin Dashboard JavaScript
document.addEventListener('DOMContentLoaded', function() {
  // Initialize all dashboard functionality
  initializeCounterAnimations();
  initializeInteractiveElements();
  initializeRealTimeUpdates();
  
  // Set up periodic refresh for real-time data
  if (typeof refreshAdminInterval === 'undefined') {
    let refreshAdminInterval = setInterval(refreshRealTimeData, 300000); // Every 5 minutes
  }
});

/**
 * Initialize counter animations for KPI values
 */
function initializeCounterAnimations() {
  // Animate dashboard cards
  const cardValues = document.querySelectorAll('.card-value');
  cardValues.forEach(valueElement => {
    animateCounter(valueElement, 1200);
  });
  
  // Animate KPI values
  const kpiValues = document.querySelectorAll('.kpi-value');
  kpiValues.forEach(valueElement => {
    animateCounter(valueElement, 1000);
  });
  
  // Animate highlight values that are numbers
  const highlightValues = document.querySelectorAll('.highlight-value');
  highlightValues.forEach(valueElement => {
    const text = valueElement.textContent.trim();
    if (/^\d+$/.test(text)) { // Only animate pure numbers
      animateCounter(valueElement, 800);
    }
  });
}

/**
 * Animate a counter element from 0 to its final value
 */
function animateCounter(element, duration) {
  const text = element.textContent.trim();
  let finalValue, suffix = '';
  
  // Handle percentages
  if (text.includes('%')) {
    finalValue = parseFloat(text.replace('%', ''));
    suffix = '%';
  }
  // Handle pure numbers
  else if (/^\d+$/.test(text)) {
    finalValue = parseInt(text);
  }
  // Skip non-numeric values
  else {
    return;
  }
  
  if (isNaN(finalValue)) return;
  
  let startTimestamp = null;
  const startValue = 0;
  
  const step = (timestamp) => {
    if (!startTimestamp) startTimestamp = timestamp;
    const progress = Math.min((timestamp - startTimestamp) / duration, 1);
    
    const currentValue = Math.floor(progress * (finalValue - startValue) + startValue);
    
    if (suffix === '%') {
      element.textContent = currentValue.toFixed(1) + suffix;
    } else {
      element.textContent = currentValue + suffix;
    }
    
    if (progress < 1) {
      window.requestAnimationFrame(step);
    } else {
      // Ensure final value is exact
      element.textContent = (suffix === '%' ? finalValue.toFixed(1) : finalValue) + suffix;
    }
  };
  
  window.requestAnimationFrame(step);
}

/**
 * Initialize interactive elements
 */
function initializeInteractiveElements() {
  // Add hover effects to performance rows
  const performanceRows = document.querySelectorAll('.performance-row');
  performanceRows.forEach(row => {
    row.addEventListener('mouseenter', function() {
      this.style.transform = 'translateX(5px)';
    });
    
    row.addEventListener('mouseleave', function() {
      this.style.transform = 'translateX(0)';
    });
  });
  
  // Add click handlers to agent items for potential drill-down
  const agentItems = document.querySelectorAll('.agent-item');
  agentItems.forEach(item => {
    item.addEventListener('click', function() {
      const agentName = this.querySelector('.agent-name').textContent;
      console.log('Clicked on agent:', agentName);
      // Potential navigation to agent detail view
      // window.location.href = `/agentbonus/detail?agent=${encodeURIComponent(agentName)}`;
    });
  });
  
  // Add click handlers to KPI items for potential drill-down
  const kpiItems = document.querySelectorAll('.kpi-item');
  kpiItems.forEach(item => {
    item.addEventListener('click', function() {
      const label = this.querySelector('.kpi-label').textContent;
      console.log('Clicked on KPI:', label);
      // Potential navigation based on KPI type
    });
  });
  
  // Add click handlers to highlight items
  const highlightItems = document.querySelectorAll('.highlight-item');
  highlightItems.forEach(item => {
    item.addEventListener('click', function() {
      const label = this.querySelector('.highlight-label').textContent;
      console.log('Clicked on highlight:', label);
      // Potential navigation to detailed view
    });
  });
}

/**
 * Initialize real-time updates for dashboard data
 */
function initializeRealTimeUpdates() {
  // Add visual indicators for real-time data
  const dashboardCards = document.querySelectorAll('.dashboard-card');
  dashboardCards.forEach(card => {
    card.addEventListener('mouseenter', function() {
      this.style.transform = 'translateY(-8px)';
    });
    
    card.addEventListener('mouseleave', function() {
      this.style.transform = 'translateY(0)';
    });
  });
  
  // Initialize pulse animation for pending actions if count > 0
  const pendingKpi = document.querySelector('.kpi-item:nth-child(4) .kpi-value');
  if (pendingKpi && parseInt(pendingKpi.textContent) > 0) {
    pendingKpi.parentElement.parentElement.classList.add('urgent-attention');
  }
}

/**
 * Refresh real-time data via AJAX
 */
function refreshRealTimeData() {
  fetch('/sales/admin-stats', {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    }
  })
    .then(response => {
      if (!response.ok) {
        throw new Error('Network response was not ok');
      }
      return response.json();
    })
    .then(data => {
      updateDashboardValues(data);
      showUpdateIndicator();
    })
    .catch(error => {
      console.error('Error refreshing real-time data:', error);
      showErrorIndicator();
    });
}

/**
 * Update dashboard values with new data
 */
function updateDashboardValues(data) {
  // Update main dashboard cards
  updateElementValue('.dashboard-card:nth-child(1) .card-value', data.totalSales);
  updateElementValue('.dashboard-card:nth-child(2) .card-value', data.activeSales);
  updateElementValue('.dashboard-card:nth-child(3) .card-value', data.conversionRate, '%');
  updateElementValue('.dashboard-card:nth-child(4) .card-value', data.stornoRate, '%');
  
  // Update secondary KPIs
  updateElementValue('.kpi-item:nth-child(1) .kpi-value', data.todaySales);
  updateElementValue('.kpi-item:nth-child(2) .kpi-value', data.thisWeekSales);
  updateElementValue('.kpi-item:nth-child(3) .kpi-value', data.monthlyGrowth, '%');
  updateElementValue('.kpi-item:nth-child(4) .kpi-value', data.pendingActions);
  
  // Update pending actions urgency indicator
  const pendingElement = document.querySelector('.kpi-item:nth-child(4)');
  if (data.pendingActions > 0) {
    pendingElement.classList.add('urgent-attention');
  } else {
    pendingElement.classList.remove('urgent-attention');
  }
  
  // Update weekly change indicator
  if (data.thisWeekSales !== undefined && data.lastWeekSales !== undefined) {
    const weeklyChange = data.lastWeekSales > 0 ? 
      ((data.thisWeekSales - data.lastWeekSales) / data.lastWeekSales) * 100 : 0;
    
    const changeElement = document.querySelector('.change-indicator');
    if (changeElement) {
      const changeClass = weeklyChange >= 0 ? 'positive' : 'negative';
      changeElement.className = `change-indicator ${changeClass}`;
      changeElement.textContent = `${weeklyChange >= 0 ? '+' : ''}${weeklyChange.toFixed(1)}%`;
    }
  }
}

/**
 * Update a specific element's value with animation
 */
function updateElementValue(selector, newValue, suffix = '') {
  const element = document.querySelector(selector);
  if (!element || newValue === undefined) return;
  
  const currentValue = parseFloat(element.textContent.replace(/[^\d.-]/g, '')) || 0;
  const targetValue = parseFloat(newValue) || 0;
  
  // Only animate if value has changed
  if (currentValue !== targetValue) {
    // Add update flash effect
    element.classList.add('value-updated');
    
    // Animate to new value
    animateValueChange(element, currentValue, targetValue, suffix, 800);
    
    // Remove flash effect after animation
    setTimeout(() => {
      element.classList.remove('value-updated');
    }, 1000);
  }
}

/**
 * Animate value change from current to new value
 */
function animateValueChange(element, startValue, endValue, suffix, duration) {
  let startTimestamp = null;
  
  const step = (timestamp) => {
    if (!startTimestamp) startTimestamp = timestamp;
    const progress = Math.min((timestamp - startTimestamp) / duration, 1);
    
    const currentValue = startValue + (endValue - startValue) * progress;
    
    if (suffix === '%') {
      element.textContent = currentValue.toFixed(1) + suffix;
    } else {
      element.textContent = Math.round(currentValue) + suffix;
    }
    
    if (progress < 1) {
      window.requestAnimationFrame(step);
    } else {
      // Ensure final value is exact
      element.textContent = (suffix === '%' ? endValue.toFixed(1) : Math.round(endValue)) + suffix;
    }
  };
  
  window.requestAnimationFrame(step);
}

/**
 * Show update indicator
 */
function showUpdateIndicator() {
  // Create or update a small indicator showing last update time
  let indicator = document.querySelector('.update-indicator');
  if (!indicator) {
    indicator = document.createElement('div');
    indicator.className = 'update-indicator';
    document.querySelector('.content-header').appendChild(indicator);
  }
  
  const now = new Date();
  indicator.innerHTML = `
    <i class="fas fa-sync-alt"></i>
    <span>Ažurirano: ${now.toLocaleTimeString('hr-HR')}</span>
  `;
  
  // Add pulse animation
  indicator.classList.add('updated');
  setTimeout(() => {
    indicator.classList.remove('updated');
  }, 1000);
}

/**
 * Show error indicator
 */
function showErrorIndicator() {
  let indicator = document.querySelector('.update-indicator');
  if (!indicator) {
    indicator = document.createElement('div');
    indicator.className = 'update-indicator error';
    document.querySelector('.content-header').appendChild(indicator);
  }
  
  indicator.innerHTML = `
    <i class="fas fa-exclamation-triangle"></i>
    <span>Greška pri ažuriranju</span>
  `;
  
  indicator.classList.add('error');
  setTimeout(() => {
    indicator.classList.remove('error');
  }, 3000);
}

/**
 * Manual refresh function that can be called by user
 */
function manualRefresh() {
  showLoadingIndicator();
  refreshRealTimeData();
}

/**
 * Show loading indicator
 */
function showLoadingIndicator() {
  let indicator = document.querySelector('.update-indicator');
  if (!indicator) {
    indicator = document.createElement('div');
    indicator.className = 'update-indicator';
    document.querySelector('.content-header').appendChild(indicator);
  }
  
  indicator.innerHTML = `
    <i class="fas fa-spinner fa-spin"></i>
    <span>Ažuriranje...</span>
  `;
}

// Add CSS styles for update indicators and animations
const additionalStyles = `
<style>
.update-indicator {
  position: absolute;
  top: 1rem;
  right: 1rem;
  display: flex;
  align-items: center;
  gap: 0.5rem;
  padding: 0.5rem 1rem;
  background-color: rgba(var(--accent-primary-rgb), 0.1);
  color: var(--accent-primary);
  border-radius: 20px;
  font-size: 0.8rem;
  transition: all 0.3s ease;
  border: 1px solid rgba(var(--accent-primary-rgb), 0.2);
}

.update-indicator.updated {
  background-color: rgba(76, 175, 80, 0.1);
  color: #4caf50;
  border-color: rgba(76, 175, 80, 0.2);
  transform: scale(1.05);
}

.update-indicator.error {
  background-color: rgba(244, 67, 54, 0.1);
  color: #f44336;
  border-color: rgba(244, 67, 54, 0.2);
}

.value-updated {
  background-color: rgba(var(--accent-primary-rgb), 0.2) !important;
  transform: scale(1.05);
  transition: all 0.3s ease;
}

.urgent-attention {
  animation: pulse-urgent 2s infinite;
}

@keyframes pulse-urgent {
  0% { transform: scale(1); }
  50% { transform: scale(1.02); box-shadow: 0 0 0 4px rgba(244, 67, 54, 0.1); }
  100% { transform: scale(1); }
}

.content-header {
  position: relative;
}
</style>
`;

// Inject additional styles
document.head.insertAdjacentHTML('beforeend', additionalStyles);