// teambonusfinancije.js
document.addEventListener('DOMContentLoaded', function() {
  // Card value animation for EUR amounts
  const cardValues = document.querySelectorAll('.card-value');
  cardValues.forEach(valueElement => {
    let originalText = valueElement.textContent;
    
    // Extract numeric value
    let numericValue = parseFloat(originalText.replace(/[^\d.-]/g, ''));
    if (!isNaN(numericValue)) {
      if (originalText.includes('€')) {
        // EUR value animation
        animateEuroValue(valueElement, 0, numericValue, 1200);
      } else if (originalText.includes('%')) {
        // Percentage animation
        animateValue(valueElement, 0, numericValue, 1000, numericValue, 1, false, true);
      } else {
        // Simple integer animation
        animateValue(valueElement, 0, numericValue, 1000);
      }
    }
  });
  
  // Initialize summary counts animation
  const summaryValues = document.querySelectorAll('.summary-value');
  summaryValues.forEach(valueElement => {
    let originalText = valueElement.textContent;
    let numericValue = parseFloat(originalText.replace(/[^\d.-]/g, ''));
    
    if (!isNaN(numericValue)) {
      if (originalText.includes('€')) {
        animateEuroValue(valueElement, 0, numericValue, 800);
      } else if (originalText.includes('%')) {
        animateValue(valueElement, 0, numericValue, 800, numericValue, 1, false, true);
      } else {
        animateValue(valueElement, 0, numericValue, 800);
      }
    }
  });
  
  // Refresh stats periodically (every 5 minutes)
  if (typeof refreshFinancijeInterval === 'undefined') {
    let refreshFinancijeInterval = setInterval(refreshFinancijeStats, 300000);
  }

  // Add hover effects to service items
  const serviceItems = document.querySelectorAll('.service-item');
  serviceItems.forEach(item => {
    item.addEventListener('mouseenter', function() {
      this.style.transform = 'translateX(5px)';
    });
    
    item.addEventListener('mouseleave', function() {
      this.style.transform = 'translateX(0)';
    });
  });

  // Add hover effects to prijenos cards
  const prijenosCards = document.querySelectorAll('.prijenos-card');
  prijenosCards.forEach(card => {
    card.addEventListener('mouseenter', function() {
      this.style.transform = 'translateY(-5px)';
    });
    
    card.addEventListener('mouseleave', function() {
      this.style.transform = 'translateY(0)';
    });
  });

  // Period filter auto-submit on change
  const monthSelect = document.getElementById('month');
  const yearSelect = document.getElementById('year');
  
  if (monthSelect && yearSelect) {
    monthSelect.addEventListener('change', function() {
      this.form.submit();
    });
    
    yearSelect.addEventListener('change', function() {
      this.form.submit();
    });
  }
});

// Animate EUR value from start to end
function animateEuroValue(element, start, end, duration) {
  let startTimestamp = null;
  
  const step = (timestamp) => {
    if (!startTimestamp) startTimestamp = timestamp;
    const progress = Math.min((timestamp - startTimestamp) / duration, 1);
    const value = progress * (end - start) + start;
    
    // Format as currency
    const formattedValue = value.toLocaleString('hr-HR', {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2
    });
    
    element.textContent = formattedValue + ' €';
    
    if (progress < 1) {
      window.requestAnimationFrame(step);
    }
  };
  
  window.requestAnimationFrame(step);
}

// Animate counter value from start to end with optional decimal formatting
function animateValue(element, start, end, duration, finalValue, decimalPlaces, hasComma, hasPercent) {
  let startTimestamp = null;
  
  const step = (timestamp) => {
    if (!startTimestamp) startTimestamp = timestamp;
    const progress = Math.min((timestamp - startTimestamp) / duration, 1);
    const value = Math.floor(progress * (end - start) + start);
    
    if (decimalPlaces !== undefined) {
      // For values with decimal places
      const decimalSeparator = hasComma ? ',' : '.';
      let formattedValue;
      
      if (progress >= 1) {
        // On animation completion, use the exact final value
        formattedValue = typeof finalValue === 'number' ? 
          finalValue.toFixed(decimalPlaces).replace('.', decimalSeparator) : 
          value.toString();
      } else {
        formattedValue = value.toString();
      }
      
      // Add percent symbol if needed
      if (hasPercent) {
        formattedValue += '%';
      }
      
      element.textContent = formattedValue;
    } else {
      // For simple integers
      element.textContent = value;
    }
    
    if (progress < 1) {
      window.requestAnimationFrame(step);
    }
  };
  
  window.requestAnimationFrame(step);
}

// Function to refresh team bonus financije stats via AJAX
function refreshFinancijeStats() {
  const urlParams = new URLSearchParams(window.location.search);
  const month = urlParams.get('month') || new Date().getMonth() + 1;
  const year = urlParams.get('year') || new Date().getFullYear();
  
  fetch('/teambonusfinancije/api/stats', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
    },
    body: JSON.stringify({
      month: month,
      year: year
    })
  })
    .then(response => response.json())
    .then(data => {
      // Update total earnings
      updateEuroValue('.total-earnings .card-value', data.totalEarnings);
      
      // Update individual bonus type earnings
      if (data.earnings) {
        updateEuroValue('.tomato-earnings .card-value', data.earnings.tomato?.total_earnings || 0);
        updateEuroValue('.p2p-earnings .card-value', data.earnings.p2p?.total_earnings || 0);
      }
      
      // Update growth percentage
      if (data.comparison && data.comparison.growth_percent !== undefined) {
        const growthElement = document.querySelector('.growth-percent .card-value');
        if (growthElement) {
          const growth = data.comparison.growth_percent;
          growthElement.textContent = `${growth >= 0 ? '+' : ''}${growth.toFixed(1)}%`;
          growthElement.className = growthElement.className.replace(/(positive|negative)/, '');
          growthElement.classList.add(growth >= 0 ? 'positive' : 'negative');
        }
      }
      
      console.log('Team bonus financije stats refreshed successfully');
    })
    .catch(error => {
      console.error('Error refreshing team bonus financije stats:', error);
    });
}

// Helper to update EUR values
function updateEuroValue(selector, value) {
  const element = document.querySelector(selector);
  if (element) {
    const formattedValue = value.toLocaleString('hr-HR', {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2
    });
    element.textContent = formattedValue + ' €';
  }
}

// Helper to update percentage values
function updatePercentValue(selector, value) {
  const element = document.querySelector(selector);
  if (element) {
    element.textContent = value.toFixed(1) + '%';
  }
}

// Export functions for chart access
window.TeamBonusFinancije = {
  refreshStats: refreshFinancijeStats,
  updateEuroValue: updateEuroValue,
  updatePercentValue: updatePercentValue
};