/**
 * Team Bonus Financije Charts - REDESIGNED & CLEAN - AŽURIRANO za novu strukturu
 */
document.addEventListener('DOMContentLoaded', function() {
  const chartsData = document.getElementById('teambonusfinancije-charts-data');
  
  if (!chartsData) {
    console.error('Team bonus financije charts data element not found!');
    return;
  }
  
  try {
    const earningsData = JSON.parse(chartsData.getAttribute('data-earnings') || '{}');
    const monthlyTrend = JSON.parse(chartsData.getAttribute('data-monthly-trend') || '[]');
    const comparisonData = JSON.parse(chartsData.getAttribute('data-comparison') || '{}');
    
    // Create charts
    createMonthlyEarningsChart(monthlyTrend);
    createBonusTypeDistributionChart(earningsData);
    
  } catch (error) {
    console.error('Error initializing team bonus financije charts:', error);
  }
});

/**
 * Create monthly earnings trend chart with better styling
 */
function createMonthlyEarningsChart(monthlyTrend) {
  const canvas = document.getElementById('monthlyEarningsChart');
  if (!canvas) return;
  
  if (!monthlyTrend || monthlyTrend.length === 0) {
    canvas.parentElement.innerHTML = '<div class="empty-state"><i class="fas fa-chart-line"></i><p>Nema podataka o mjesečnim trendovima</p></div>';
    return;
  }
  
  const monthNames = ['Sij', 'Velj', 'Ožu', 'Tra', 'Svi', 'Lip', 'Srp', 'Kol', 'Ruj', 'Lis', 'Stu', 'Pro'];
  const labels = monthlyTrend.map(item => `${monthNames[item.month - 1]} ${item.year}`);
  const values = monthlyTrend.map(item => parseFloat(item.total_earnings || 0));
  
  const ctx = canvas.getContext('2d');
  new Chart(ctx, {
    type: 'line',
    data: {
      labels: labels,
      datasets: [{
        label: 'Ukupne zarade',
        data: values,
        backgroundColor: 'rgba(40, 167, 69, 0.15)',
        borderColor: '#28a745',
        borderWidth: 3,
        fill: true,
        tension: 0.4,
        pointBackgroundColor: '#28a745',
        pointBorderColor: '#ffffff',
        pointBorderWidth: 3,
        pointRadius: 6,
        pointHoverRadius: 8,
        pointHoverBackgroundColor: '#20c997',
        pointHoverBorderColor: '#ffffff',
        pointHoverBorderWidth: 3
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      interaction: {
        intersect: false,
        mode: 'index'
      },
      scales: {
        y: {
          beginAtZero: true,
          grid: {
            color: 'rgba(255, 255, 255, 0.1)',
            drawBorder: false
          },
          ticks: {
            color: '#6c757d',
            font: {
              size: 11
            },
            callback: function(value) {
              return value.toLocaleString('hr-HR', {
                minimumFractionDigits: 0,
                maximumFractionDigits: 0
              }) + ' €';
            }
          }
        },
        x: {
          grid: {
            display: false
          },
          ticks: {
            color: '#6c757d',
            font: {
              size: 11
            },
            maxRotation: 0
          }
        }
      },
      plugins: {
        legend: {
          display: false
        },
        tooltip: {
          backgroundColor: 'rgba(0, 0, 0, 0.8)',
          titleColor: '#ffffff',
          bodyColor: '#ffffff',
          borderColor: '#28a745',
          borderWidth: 1,
          cornerRadius: 8,
          padding: 12,
          callbacks: {
            label: function(context) {
              const value = context.raw.toLocaleString('hr-HR', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
              });
              return `Zarade: ${value} €`;
            }
          }
        }
      }
    }
  });
}

/**
 * Create bonus type distribution chart (pie chart) - AŽURIRANO za novu strukturu
 */
function createBonusTypeDistributionChart(earningsData) {
  const canvas = document.getElementById('bonusTypeChart');
  if (!canvas) return;
  
  if (!earningsData || Object.keys(earningsData).length === 0) {
    canvas.parentElement.innerHTML = '<div class="empty-state"><i class="fas fa-chart-pie"></i><p>Nema podataka o distribuciji zarada</p></div>';
    return;
  }
  
  // AŽURIRANO - samo 3 tipa bonus-a
  const bonusTypes = {
    'prijenos': 'Prijenos',
    'p2p': 'P2P',
    'tomato': 'Tomato'
  };
  
  const colors = ['#007bff', '#6f42c1', '#28a745'];
  const labels = [];
  const values = [];
  const backgroundColors = [];
  
  Object.keys(bonusTypes).forEach((key, index) => {
    if (earningsData[key] && earningsData[key].total_earnings > 0) {
      labels.push(bonusTypes[key]);
      values.push(parseFloat(earningsData[key].total_earnings));
      backgroundColors.push(colors[index]);
    }
  });
  
  if (values.length === 0) {
    canvas.parentElement.innerHTML = '<div class="empty-state"><i class="fas fa-chart-pie"></i><p>Nema zarada za prikaz</p></div>';
    return;
  }
  
  const ctx = canvas.getContext('2d');
  new Chart(ctx, {
    type: 'doughnut',
    data: {
      labels: labels,
      datasets: [{
        data: values,
        backgroundColor: backgroundColors,
        borderWidth: 3,
        borderColor: '#ffffff',
        hoverBorderWidth: 4,
        hoverBorderColor: '#ffffff'
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      cutout: '60%',
      plugins: {
        legend: {
          position: 'bottom',
          labels: {
            boxWidth: 15,
            boxHeight: 15,
            font: {
              size: 12,
              weight: '600'
            },
            color: '#495057',
            padding: 15,
            usePointStyle: true,
            pointStyle: 'circle',
            generateLabels: function(chart) {
              const data = chart.data;
              if (data.labels.length && data.datasets.length) {
                return data.labels.map((label, i) => {
                  const value = data.datasets[0].data[i];
                  const total = data.datasets[0].data.reduce((acc, val) => acc + val, 0);
                  const percentage = ((value / total) * 100).toFixed(1);
                  const formattedValue = value.toLocaleString('hr-HR', {
                    minimumFractionDigits: 0,
                    maximumFractionDigits: 0
                  });
                  return {
                    text: `${label}: ${formattedValue} € (${percentage}%)`,
                    fillStyle: data.datasets[0].backgroundColor[i],
                    hidden: false,
                    index: i
                  };
                });
              }
              return [];
            }
          }
        },
        tooltip: {
          backgroundColor: 'rgba(0, 0, 0, 0.8)',
          titleColor: '#ffffff',
          bodyColor: '#ffffff',
          borderColor: '#495057',
          borderWidth: 1,
          cornerRadius: 8,
          padding: 12,
          callbacks: {
            label: function(context) {
              const value = context.raw;
              const total = context.dataset.data.reduce((acc, val) => acc + val, 0);
              const percentage = ((value / total) * 100).toFixed(1);
              const formattedValue = value.toLocaleString('hr-HR', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
              });
              return `${context.label}: ${formattedValue} € (${percentage}%)`;
            }
          }
        }
      }
    }
  });
}