/**
 * Vrsta Schema List JavaScript
 * Specific functionality for vrsta schema list view
 * NOTE: This script depends on table.js and should be loaded AFTER it
 */
document.addEventListener('DOMContentLoaded', function() {
    // Apply titles to cells for tooltip display on hover
    addCellTitles();
    
    /**
     * Add title attributes to all cells for better display of truncated content
     */
    function addCellTitles() {
      const dataCells = document.querySelectorAll('.data-table td');
      dataCells.forEach(cell => {
        if (cell.textContent.trim()) {
          cell.setAttribute('title', cell.textContent.trim());
        }
      });
    }
    // Find the table (already initialized by table.js)
    const dataTable = document.getElementById('dataTable');
    
    if (!dataTable) {
      console.warn('Vrsta table not found');
      return;
    }
    
    // Add vrsta-specific enhancements
    addHoverEffects();
    initFilters(); // Add filters initialization
    initDeleteModal(); // Initialize delete confirmation modal
    initExportPanel(); // Initialize export panel
    
    /**
     * Add hover effects to rows and elements
     */
    function addHoverEffects() {
      const rows = dataTable.querySelectorAll('tbody tr');
      
      rows.forEach(row => {
        row.addEventListener('mouseenter', function() {
          // Any hover effects can be added here
        });
        
        row.addEventListener('mouseleave', function() {
          // Any hover effects removal can be added here
        });
      });
    }
    
    /**
     * Initialize filter functionality for tarifa and grupa
     */
    function initFilters() {
      const tarifaFilterContainer = document.getElementById('tarifaFilter');
      const grupaFilterContainer = document.getElementById('grupaFilter');
      
      if (!tarifaFilterContainer || !grupaFilterContainer) {
        console.warn('Filter containers not found');
        return;
      }
      
      // Get all unique tarifa values from the table
      const tarifaValues = new Set();
      const tarifaCells = dataTable.querySelectorAll('tbody tr td:nth-child(2)');
      
      tarifaCells.forEach(cell => {
        const tarifaValue = cell.textContent.trim();
        if (tarifaValue) {
          tarifaValues.add(tarifaValue);
        }
      });
      
      // Get all unique grupa values from the table
      const grupaValues = new Set();
      const grupaCells = dataTable.querySelectorAll('tbody tr td:nth-child(3) .group-badge');
      
      grupaCells.forEach(badge => {
        const grupaValue = badge.textContent.trim();
        if (grupaValue) {
          grupaValues.add(grupaValue);
        }
      });
      
      // Clear existing content
      tarifaFilterContainer.innerHTML = '';
      grupaFilterContainer.innerHTML = '';
      
      // Create filter checkboxes for each tarifa value
      tarifaValues.forEach(tarifaValue => {
        const label = document.createElement('label');
        label.className = 'filter-checkbox';
        
        const checkbox = document.createElement('input');
        checkbox.type = 'checkbox';
        checkbox.setAttribute('data-filter', 'tarifa');
        checkbox.setAttribute('data-value', tarifaValue);
        checkbox.checked = true; // Default to checked
        
        label.appendChild(checkbox);
        label.appendChild(document.createTextNode(' ' + tarifaValue));
        
        tarifaFilterContainer.appendChild(label);
      });
      
      // Create filter checkboxes for each grupa value
      grupaValues.forEach(grupaValue => {
        const label = document.createElement('label');
        label.className = 'filter-checkbox';
        
        const checkbox = document.createElement('input');
        checkbox.type = 'checkbox';
        checkbox.setAttribute('data-filter', 'grupa');
        checkbox.setAttribute('data-value', grupaValue);
        checkbox.checked = true; // Default to checked
        
        label.appendChild(checkbox);
        label.appendChild(document.createTextNode(' ' + grupaValue));
        
        grupaFilterContainer.appendChild(label);
      });
      
      // Add event listeners to the new checkboxes
      const filterCheckboxes = document.querySelectorAll('.filter-checkbox input[type="checkbox"]');
      filterCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', applyFilters);
      });
      
      // Add search functionality
      const searchInput = document.getElementById('dataSearch');
      if (searchInput) {
        searchInput.addEventListener('input', applyFilters);
      }
    }
    
    /**
     * Apply filters to the table
     */
    function applyFilters() {
      if (!dataTable) return;
      
      const rows = dataTable.querySelectorAll('tbody tr');
      const activeFilters = {};
      const filterCheckboxes = document.querySelectorAll('.filter-checkbox input:checked');
      const searchInput = document.getElementById('dataSearch');
      const searchValue = searchInput ? searchInput.value.toLowerCase() : '';
      
      console.log('Applying filters, checked filters:', filterCheckboxes.length);
      
      // Build active filters map
      filterCheckboxes.forEach(checkbox => {
        const filterType = checkbox.getAttribute('data-filter');
        const filterValue = checkbox.getAttribute('data-value');
        
        if (!activeFilters[filterType]) {
          activeFilters[filterType] = [];
        }
        
        activeFilters[filterType].push(filterValue);
      });
      
      console.log('Active filters:', activeFilters);
      
      // Apply filtering
      let visibleCount = 0;
      
      rows.forEach(row => {
        let shouldShow = true;
        
        // Apply search filter
        if (searchValue) {
          const rowText = row.textContent.toLowerCase();
          if (!rowText.includes(searchValue)) {
            shouldShow = false;
          }
        }
        
        // Apply category filters if row is still visible
        if (shouldShow) {
          // Check each filter type
          for (const filterType in activeFilters) {
            let matches = false;
            
            if (filterType === 'tarifa') {
              // Get the tarifa from the row
              const tarifaCell = row.querySelector('td:nth-child(2)');
              if (tarifaCell) {
                const tarifaValue = tarifaCell.textContent.trim();
                
                matches = activeFilters[filterType].some(value => 
                  tarifaValue === value
                );
              }
            } else if (filterType === 'grupa') {
              // Get the grupa from the row
              const grupaBadge = row.querySelector('td:nth-child(3) .group-badge');
              if (grupaBadge) {
                const grupaValue = grupaBadge.textContent.trim();
                
                matches = activeFilters[filterType].some(value => 
                  grupaValue === value
                );
              }
            }
            
            if (!matches) {
              shouldShow = false;
              break;
            }
          }
        }
        
        row.style.display = shouldShow ? '' : 'none';
        
        if (shouldShow) {
          visibleCount++;
        }
      });
      
      console.log('Filter applied, visible rows:', visibleCount);
    }
    
    /**
     * Initialize delete modal functionality
     */
    function initDeleteModal() {
      const deleteButtons = document.querySelectorAll('.action-btn.delete');
      const deleteModal = document.getElementById('deleteModal');
      const deleteForm = document.getElementById('deleteForm');
      const deleteId = document.getElementById('deleteId');
      const deleteName = document.getElementById('deleteName');
      const modalClose = document.querySelector('.modal-close');
      const modalCancel = document.querySelector('.modal-cancel');
      
      if (!deleteModal) {
        console.warn('Delete modal not found');
        return;
      }
      
      deleteButtons.forEach(button => {
        button.addEventListener('click', function() {
          const id = this.getAttribute('data-id');
          const name = this.getAttribute('data-name');
          
          deleteId.value = id;
          deleteName.textContent = name;
          deleteModal.style.display = 'block';
        });
      });
      
      // Close modal when clicking on X or Cancel
      modalClose.addEventListener('click', function() {
        deleteModal.style.display = 'none';
      });
      
      modalCancel.addEventListener('click', function() {
        deleteModal.style.display = 'none';
      });
      
      // Close modal when clicking outside
      window.addEventListener('click', function(event) {
        if (event.target == deleteModal) {
          deleteModal.style.display = 'none';
        }
      });
    }
    
    /**
     * Initialize export panel functionality
     */
    function initExportPanel() {
      const exportToggle = document.getElementById('exportToggle');
      const exportPanel = document.getElementById('exportPanel');
      const closeExport = document.getElementById('closeExport');
      
      if (!exportPanel || !exportToggle) {
        console.warn('Export panel elements not found');
        return;
      }
      
      exportToggle.addEventListener('click', function() {
        exportPanel.classList.toggle('visible');
      });
      
      closeExport.addEventListener('click', function() {
        exportPanel.classList.remove('visible');
      });
    }
    
    // Listen for theme changes
    document.addEventListener('themechange', function(event) {
      console.log('Theme changed in vrsta list page:', event.detail.theme);
    });
  });