<?php
namespace tasks\Controllers;

use Controllers\BaseController;
use Models\DatabaseTable;
use tasks\Models\DatabaseTableTasks;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Validation\Validator;

class Comments extends BaseController 
{
    private DatabaseTable $korisniciTable;
    private DatabaseTableTasks $zadatciTable;
    private DatabaseTableTasks $vrsteTable;
    private DatabaseTableTasks $zadatakVrsteTable;
    private DatabaseTableTasks $osobeTable;
    private DatabaseTableTasks $zadatakOsobeTable;
    private DatabaseTableTasks $zadatakCommentsTable;
    private Validator $validator;

    public function __construct(
        DatabaseTable $korisniciTable,
        DatabaseTableTasks $zadatciTable,
        DatabaseTableTasks $vrsteTable,
        DatabaseTableTasks $zadatakVrsteTable,
        DatabaseTableTasks $osobeTable,
        DatabaseTableTasks $zadatakOsobeTable,
        DatabaseTableTasks $zadatakCommentsTable,
        Validator $validator,
        Logger $logger
    ) {
        parent::__construct($logger);
        $this->korisniciTable = $korisniciTable;
        $this->zadatciTable = $zadatciTable;
        $this->vrsteTable = $vrsteTable;
        $this->zadatakVrsteTable = $zadatakVrsteTable;
        $this->osobeTable = $osobeTable;
        $this->zadatakOsobeTable = $zadatakOsobeTable;
        $this->zadatakCommentsTable = $zadatakCommentsTable;
        $this->validator = $validator;
        
        $this->addValidationRules();
    }

    private function addValidationRules(): void 
    {
        $this->validator->addRule(
            'comment_length',
            function($value) {
                $length = strlen(trim($value));
                return $length > 0 && $length <= 256;
            },
            'Comment must be between 1 and 256 characters'
        );
    }

    public function addComments(Request $request): Response 
    {
        // Validate using the validator
        $rules = [
            'comment_name' => ['required'],
            'comment_content' => ['required', 'comment_length']
        ];

        $validation = $request->validate($rules);
        if ($validation->fails()) {
            $errors = $validation->errors();
            $errorMessage = reset($errors)[0] ?? 'Invalid input';
            return new Response(json_encode(['error' => $errorMessage]));
        }

        try {
            $commentData = [
                'zadatakId' => $request->getPost('zadatakId'),
                'parent_comment_id' => $request->getPost('comment_id'),
                'comment' => $this->sanitizeInput($request->getPost('comment_content')),
                'comment_sender_name' => $this->sanitizeInput($request->getPost('comment_name')),
                'date' => $request->getPost('date')
            ];

            $this->zadatakCommentsTable->addComment($commentData);
            return new Response(json_encode(['error' => '<label class="text-success">Comment Added</label>']));
        } catch (\Exception $e) {
            $this->logger->error('Error adding comment', [
                'error' => $e->getMessage(),
                'data' => $commentData ?? null
            ]);
            return new Response(json_encode(['error' => '<label class="text-danger">Error adding comment</label>']));
        }
    }

    public function fetchComments(Request $request): Response 
    {
        $zadatakId = $request->getPost('myData');
        
        if (empty($zadatakId)) {
            return new Response('');
        }

        try {
            // Dohvati samo komentare najviše razine (parent_comment_id = 0)
            $comments = $this->zadatakCommentsTable->findTaskComments((int)$zadatakId);
            if (empty($comments)) {
                return new Response('');
            }

            // Za svaki komentar, izračunaj ukupan broj odgovora rekurzivno
            foreach ($comments as &$comment) {
                $comment['reply_count'] = $this->zadatakCommentsTable->countAllReplies($comment['comment_id']);
            }
            
            $output = $this->renderComments($comments);
            return new Response($output);
        } catch (\Exception $e) {
            $this->logger->error('Error fetching comments', [
                'error' => $e->getMessage(),
                'zadatakId' => $zadatakId
            ]);
            return new Response('');
        }
    }
    
    public function fetchReplies(Request $request): Response 
    {
        $commentId = $request->getPost('commentId');
        
        if (empty($commentId)) {
            return new Response(json_encode(['error' => 'Comment ID is required']));
        }

        try {
            // Dohvati direktne odgovore na ovaj komentar
            $replies = $this->zadatakCommentsTable->findCommentReplies((int)$commentId);
            
            if (empty($replies)) {
                return new Response(json_encode(['replies' => '']));
            }
            
            // Za svaki odgovor, izračunaj ukupan broj njegovih odgovora (rekurzivno)
            foreach ($replies as &$reply) {
                $reply['reply_count'] = $this->zadatakCommentsTable->countAllReplies($reply['comment_id']);
            }
            
            $output = $this->renderReplies($replies);
            return new Response(json_encode(['replies' => $output]));
        } catch (\Exception $e) {
            $this->logger->error('Error fetching replies', [
                'error' => $e->getMessage(),
                'commentId' => $commentId
            ]);
            return new Response(json_encode(['error' => 'Error fetching replies']));
        }
    }

    private function sanitizeInput(string $input): string 
    {
        return trim(htmlspecialchars($input, ENT_QUOTES, 'UTF-8'));
    }

    private function renderComments(array $comments): string 
    {
        $output = '';
        foreach ($comments as $comment) {
            $output .= $this->renderSingleComment($comment);
        }
        return $output;
    }
    
    private function renderReplies(array $replies): string 
    {
        $output = '';
        foreach ($replies as $reply) {
            $output .= $this->renderSingleReply($reply);
        }
        return $output;
    }

    // Izmjena metode renderSingleComment u Comments.php klasi
    private function renderSingleComment(array $comment): string 
    {
        $logo = asset('default/images/logo.png');
        $date = date("d-m-Y H:i", strtotime($comment['date']));
        
        // Koristimo countAllReplies umjesto obične count metode da bismo uključili sve razine odgovora
        $replyCount = $this->zadatakCommentsTable->countAllReplies($comment['comment_id']);
        
        // Samo prikaži badge za odgovore ako postoje odgovori
        $replyCountHtml = '';
        if ($replyCount > 0) {
            $replyCountHtml = "<span class=\"reply-count\" data-count=\"{$replyCount}\"><i class=\"fas fa-comment-dots\"></i> {$replyCount}</span>";
        }
            
        return <<<HTML
            <div class="comment-thread" id="thread-{$comment['comment_id']}">
                <div class="comment" data-comment-id="{$comment['comment_id']}">
                    <div class="comment-header">
                        <div class="comment-avatar">
                            <img class="rounded-circle" src="{$logo}" alt="User avatar" />
                        </div>
                        <div class="comment-meta">
                            <h5 class="comment-author">{$comment['comment_sender_name']}</h5>
                            <p class="comment-date"><i class="far fa-clock"></i> {$date}</p>
                        </div>
                        <div class="comment-actions">
                            <button class="btn btn-sm btn-outline-primary comment-reply-btn" data-comment-id="{$comment['comment_id']}">
                                <i class="fas fa-reply"></i>
                            </button>
                        </div>
                    </div>
                    <div class="comment-content">
                        {$comment['comment']}
                    </div>
                    <div class="comment-footer">
                        {$replyCountHtml}
                    </div>
                </div>
                <div class="comment-replies" id="replies-{$comment['comment_id']}"></div>
            </div>
        HTML;
    }

    // Također izmjena metode renderSingleReply da koristi istu logiku
    private function renderSingleReply(array $reply): string 
    {
        $logo = asset('default/images/logo.png');
        $date = date("d-m-Y H:i", strtotime($reply['date']));
        
        // Koristimo countAllReplies umjesto obične count metode
        $replyCount = $this->zadatakCommentsTable->countAllReplies($reply['comment_id']);
        
        // Samo prikaži badge za odgovore ako postoje odgovori
        $replyCountHtml = '';
        if ($replyCount > 0) {
            $replyCountHtml = "<span class=\"reply-count\" data-count=\"{$replyCount}\"><i class=\"fas fa-comment-dots\"></i> {$replyCount}</span>";
        }
            
        return <<<HTML
            <div class="comment reply-comment" data-comment-id="{$reply['comment_id']}">
                <div class="comment-header">
                    <div class="comment-avatar">
                        <img class="rounded-circle" src="{$logo}" alt="User avatar" />
                    </div>
                    <div class="comment-meta">
                        <h5 class="comment-author">{$reply['comment_sender_name']}</h5>
                        <p class="comment-date"><i class="far fa-clock"></i> {$date}</p>
                    </div>
                    <div class="comment-actions">
                        <button class="btn btn-sm btn-outline-primary comment-reply-btn" data-comment-id="{$reply['comment_id']}">
                            <i class="fas fa-reply"></i>
                        </button>
                    </div>
                </div>
                <div class="comment-content">
                    {$reply['comment']}
                </div>
                <div class="comment-footer">
                    {$replyCountHtml}
                </div>
                <div class="comment-replies" id="replies-{$reply['comment_id']}"></div>
            </div>
        HTML;
    }
}