<?php
namespace tasks\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Controllers\BaseController;
use Models\DatabaseTable;
use tasks\Models\DatabaseTableTasks;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use DateTime;
use DateTimeZone;

class Mytasks extends BaseController 
{
    private DatabaseTable $korisniciTable;
    private DatabaseTableTasks $zadatciTable;
    private DatabaseTableTasks $vrsteTable;
    private DatabaseTableTasks $zadatakVrsteTable;
    private DatabaseTableTasks $osobeTable;
    private DatabaseTableTasks $zadatakOsobeTable;
    private DatabaseTableTasks $zadatakCommentsTable;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableTasks $zadatciTable,
        DatabaseTableTasks $vrsteTable,
        DatabaseTableTasks $zadatakVrsteTable,
        DatabaseTableTasks $osobeTable,
        DatabaseTableTasks $zadatakOsobeTable,
        DatabaseTableTasks $zadatakCommentsTable,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->zadatciTable = $zadatciTable;
        $this->vrsteTable = $vrsteTable;
        $this->zadatakVrsteTable = $zadatakVrsteTable;
        $this->osobeTable = $osobeTable;
        $this->zadatakOsobeTable = $zadatakOsobeTable;
        $this->zadatakCommentsTable = $zadatakCommentsTable;
    }

    public function mytasks(Request $request): Response 
    {
        $korisnik = $this->authentication->getUser();
        $osobe = $this->osobeTable->find('vidljivost', '1');

        return $this->response()->render(
            'tasks/resources/views/mytasks/mytasks_home.html.php',
            [
                'korisnik' => $korisnik,
                'osobe' => $osobe
            ],
            'Moji zadaci'
        );
    }

    public function fetchmytasks(Request $request): Response 
    {
        $menuItem = $request->getPost('menuItem') ?? "allMenuItem";
        $tip = $request->getPost('tip') ?? '';
        $currentLimit = (int)($request->getPost('currentLimit') ?? 5);

        // Dohvati zadatke prema filterima
        $zadaci = $this->zadatciTable->findZadatciByFilter($menuItem, $tip, $currentLimit);
        
        // Dohvati ukupan broj zadataka za paginaciju
        $totalCount = $this->zadatciTable->countZadatciByFilter($menuItem, $tip);

        // Generiraj brojač ticketa
        $output = $this->generateTicketCounter($menuItem, $tip, $totalCount);

        // Generiraj HTML za svaki zadatak
        foreach ($zadaci as $zadatak) {
            $output .= $this->generateZadatakHtml($zadatak);
        }

        return new Response($output);
    }

    private function generateTicketCounter(string $menuItem, string $tip, int $totalCount): string 
    {
        return "<div class='text-center' id='brojticketa' data-id1='{$menuItem}' data-id2='{$tip}'>
                    <label>{$totalCount}</label>
                </div>";
    }

    private function generateZadatakHtml($zadatak): string 
    {
        $date = new DateTime($zadatak->ticket_date, new DateTimeZone(\baseKRIZAN\Config\Config::get('defaulttimezone')));
        $timeDiff = $this->formatTimeDiff($date);
        
        // Dohvati i formatiraj osobe i vrste
        $osobe = $this->zadatciTable->findOsobeByZadatak($zadatak->id);
        $vrste = $this->zadatciTable->findVrsteByZadatak($zadatak->id);
        
        $hyperlinkedOsobe = $this->generateHyperlinks($osobe, 'osoba');
        $hyperlinkedVrste = $this->generateHyperlinks($vrste, 'vrsta');

        return $this->renderZadatakTemplate(
            $zadatak,
            $date,
            $timeDiff,
            $hyperlinkedOsobe,
            $hyperlinkedVrste
        );
    }

    private function formatTimeDiff(DateTime $date): string 
    {
        $now = new DateTime(\baseKRIZAN\Config\Config::get('defaulttimezone'));
        $interval = $now->diff($date);
        
        if ($interval->d > 0) {
            return $date->format('j F');
        }
        if ($interval->h > 0) {
            return $interval->format('%hh');
        }
        return $interval->format('%im');
    }

    private function generateHyperlinks(array $items, string $type): string 
    {
        if (empty($items)) {
            return '';
        }

        $links = array_map(function($item) use ($type) {
            return sprintf(
                '<span class="hashtag" id="%s" id2="%s">%s</span>',
                htmlspecialchars($type),
                htmlspecialchars($item['id']),
                htmlspecialchars($item['name'])
            );
        }, $items);

        return implode(', ', $links);
    }

    private function renderZadatakTemplate(
        $zadatak,
        DateTime $date,
        string $timeDiff,
        string $hyperlinkedOsobe,
        string $hyperlinkedVrste
    ): string {
        $logo = asset('default/images/logo.png');
        $tvrtka = htmlspecialchars(\baseKRIZAN\Config\Config::get('company'), ENT_QUOTES, 'UTF-8');
        $editUrl = route('task/edit/' . $zadatak->id);
        
        return <<<HTML
            <div class="tweet">
                <img class="tweet__author-logo" src="{$logo}" />
                <div class="tweet__main_center">
                    <div class="tweet__header">
                        <div class="tweet__author-name">{$tvrtka}</div>
                        <div class="tweet__author-slug">@fieldOPS</div>&nbsp;·
                        <div class="tweet__publish-time" title="{$date->format('d.m.Y H:i:s')}">
                            {$timeDiff}
                        </div>
                    </div>
                    <div class="tweet__content">
                        {$zadatak->ticket_number}<br>
                        {$zadatak->ticket_address}
                    </div>
                    <div class="tweet__content">
                        <i class="fa-solid fa-person"></i> <i class="fa-solid fa-people-group"></i>
                        {$hyperlinkedVrste}
                        <i class="fa-solid fa-people-group"></i> <i class="fa-solid fa-person"></i><br>
                        <i class="fa-solid fa-person-digging"></i> <i class="fa-solid fa-people-carry-box"></i>
                        {$hyperlinkedOsobe}
                        <i class="fa-solid fa-person-digging"></i> <i class="fa-solid fa-people-carry-box"></i>
                    </div>
                    <div class="tweet__content">
                        <i class="fa-regular fa-pen-to-square"></i><i class="fa-regular fa-pen-to-square"></i>
                        <a href="{$editUrl}">#edit</a>
                        <i class="fa-regular fa-pen-to-square"></i><i class="fa-regular fa-pen-to-square"></i>
                        <i class="fa-regular fa-image"></i><i class="fa-regular fa-image"></i>
                        <a href="{$zadatak->ticket_sharelink}">#info</a>
                        <i class="fa-regular fa-image"></i><i class="fa-regular fa-image"></i>
                    </div>
                </div>
            </div>
        HTML;
    }
}