<?php

namespace tasks\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Controllers\BaseController;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use tasks\Models\DatabaseTableTasks;
use baseKRIZAN\Validation\Validator;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;

class Osoba extends BaseController 
{
    private DatabaseTable $korisniciTable;
    private DatabaseTableTasks $osobeTable;
    private Validator $validator;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableTasks $osobeTable,
        Validator $validator,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->osobeTable = $osobeTable;
        $this->validator = $validator;
    }

    public function edit(Request $request): Response 
    {
        try {
            $osoba = null;
            $editToken = null;

            if ($id = $request->getParam('id')) {
                $osoba = $this->osobeTable->findById($id);
                if ($osoba) {
                    $editToken = $this->createEditToken($osoba->id, 'osoba');
                }
            }

            return $this->response()->render(
                'tasks/resources/views/osoba/edit.html.php',
                [
                    'osoba' => $osoba,
                    'edit_token' => $editToken
                ],
                'Edit Osoba'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading person edit form', [
                'message' => $e->getMessage(),
                'id' => $id ?? null
            ]);
            return $this->response()->renderError('Failed to load person edit form');
        }
    }

    public function saveEdit(Request $request): Response 
    {
        try {
            // 1. DOHVAĆANJE OSNOVNIH PODATAKA
            $editToken = $request->getPost('edit_token');
            $osoba = $request->getPost('osoba');
            $osobaId = null;
            $validToken = null;
            
            // 2. VALIDACIJA TOKENA (AKO POSTOJI) - ovo NE troši token
            if ($editToken) {
                $tokenResult = $this->validateEditTokenForValidation($request, 'osoba');
                if (isset($tokenResult['error'])) {
                    return $this->renderOsobaForm(
                        $osoba, 
                        $editToken, 
                        $tokenResult
                    );
                }
                // Token je valjan
                $osobaId = $tokenResult['id'];
                $validToken = $tokenResult['token'];
                $osoba['id'] = $osobaId;
            }
            
            // 3. VALIDACIJA PODATAKA FORME
            $rules = [
                'osoba[name]' => ['required', 'alpha_spaces', ['max', 255]],
                'osoba[radnomjesto]' => ['required', 'plain_text', ['max', 255]],
                'osoba[tvrtka]' => ['required', 'plain_text', ['max', 255]],
                'osoba[area]' => ['required', 'plain_text', ['max', 255]],
                'osoba[orgjedinica]' => ['required', 'plain_text', ['max', 255]],
                'osoba[vidljivost]' => ['required', 'numeric']
            ];

            $validation = $request->validate($rules);
            if ($validation->fails()) {
                // Stvori novi token za ponovni pokušaj
                $newEditToken = null;
                if ($osobaId) {
                    if ($validToken) {
                        $newEditToken = $this->refreshEditToken($validToken, 'osoba');
                    } else {
                        $newEditToken = $this->createEditToken($osobaId, 'osoba');
                    }
                }

                return $this->renderOsobaForm(
                    $osoba, 
                    $newEditToken, 
                    $validation->errors()
                );
            }

            // 4. USPJEŠNA VALIDACIJA - SADA troši token (jednokratno)
            if ($validToken) {
                $this->consumeEditToken($validToken);
            }

            // 5. SPREMANJE PODATAKA
            $this->osobeTable->save($osoba);
            
            // 6. PREUSMJERAVANJE
            return Response::redirect('taskfte/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error saving person', [
                'message' => $e->getMessage(),
                'osoba' => $osoba ?? null
            ]);
            return $this->response()->renderError('Failed to save person');
        }
    }
    
    public function list(): Response 
    {
        try {
            $osobe = $this->osobeTable->findAll();
            $this->sessionManager->set('valid_osobe_ids', array_map(fn($osoba) => $osoba->id, $osobe));

            return $this->response()->render(
                'tasks/resources/views/osoba/list.html.php',
                [
                    'user' => $this->authentication->getUser(),
                    'osobe' => $osobe
                ],
                'Sve Osobe'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading persons list', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load persons list');
        }
    }

    public function delete(Request $request): Response 
    {
        try {
            $osobaId = $request->getPost('id');
            
            // Sigurnosna provjera da li je ID na listi valjanih ID-ova
            $validOsobeIds = $this->sessionManager->get('valid_osobe_ids', []);
            if (!in_array($osobaId, $validOsobeIds ?? [])) {
                $this->logger->security('Unauthorized person deletion attempt', [
                    'id' => $osobaId,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Unauthorized action');
            }

            // Brisanje osobe
            $this->osobeTable->delete($osobaId);
            $this->removeFromSession($osobaId);

            return Response::redirect('taskfte/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error deleting person', [
                'message' => $e->getMessage(),
                'id' => $osobaId ?? null
            ]);
            return $this->response()->renderError('Failed to delete person');
        }
    }

    private function removeFromSession(int $osobaId): void 
    {
        $validIds = $this->sessionManager->get('valid_osobe_ids', []);
        $validIds = array_filter($validIds, fn($id) => $id != $osobaId);
        $this->sessionManager->set('valid_osobe_ids', $validIds);
    }

    /**
     * Helper method to render the osoba form with errors
     */
    private function renderOsobaForm(array $osoba, ?string $editToken, array $errors): Response
    {
        return $this->response()->render(
            'tasks/resources/views/osoba/edit.html.php',
            [
                'osoba' => (object)$osoba,
                'edit_token' => $editToken,
                'errors' => $errors
            ],
            'Edit Osoba'
        );
    }
}