<?php

namespace tasks\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Controllers\BaseController;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Validation\Validator;
use baseKRIZAN\Database\DatabaseConnection;
use tasks\Models\DatabaseTableTasks;
use Models\DatatablesSSP;
use Models\Entity\Korisnik;

class Tasks extends BaseController 
{
    private const MAX_EXPORT_ROWS = 1000;
    private const YES = 'yes';
    private const NO = 'no';

    protected DatabaseTable $korisniciTable;
    protected DatabaseTableTasks $zadatciTable;
    protected DatabaseTableTasks $vrsteTable;
    protected DatabaseTableTasks $zadatakVrsteTable;
    protected DatabaseTableTasks $osobeTable;
    protected DatabaseTableTasks $zadatakOsobeTable;
    protected DatabaseTableTasks $zadatakCommentsTable;
    protected Validator $validator;
    protected DatabaseConnection $dbConnection;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication, 
        DatabaseTableTasks $zadatciTable,
        DatabaseTableTasks $vrsteTable,
        DatabaseTableTasks $zadatakVrsteTable,
        DatabaseTableTasks $osobeTable,
        DatabaseTableTasks $zadatakOsobeTable,
        DatabaseTableTasks $zadatakCommentsTable,
        Validator $validator,
        DatabaseConnection $dbConnection,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->zadatciTable = $zadatciTable;
        $this->vrsteTable = $vrsteTable;
        $this->zadatakVrsteTable = $zadatakVrsteTable;
        $this->osobeTable = $osobeTable;
        $this->zadatakOsobeTable = $zadatakOsobeTable;
        $this->zadatakCommentsTable = $zadatakCommentsTable;
        $this->validator = $validator;
        $this->dbConnection = $dbConnection;
    }

    /**
     * DataTables mapping konfiguracija za zadatke
     */
    private function getDataTablesMapping(): array
    {
        $tableConfig = [
            'table_name' => 'zadatak',
            'primary_key' => 'id',
            'fixed_columns' => 2,
            'route_name' => 'taskslist'
        ];

        $columns = [
            // ID i akcije - uvijek fiksno
            [
                'type' => 'number',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'id',
                'sqlcolumnname' => 'id',
                'width' => 75,
                'render' => 'actions'
            ],
            // Sharelink
            [
                'type' => 'number',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Sharelink',
                'sqlcolumnname' => 'ticket_sharelink',
                'width' => 50,
                'render' => 'sharelink'
            ],
            // Korisnik ID
            [
                'type' => 'number',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'snimio',
                'sqlcolumnname' => 'user_id',
                'width' => 50
            ],
            // Datum zadatka
            [
                'type' => 'datetime',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'datum',
                'sqlcolumnname' => 'ticket_date',
                'width' => 100
            ],
            // Tip
            [
                'type' => 'text',
                'searchbox' => self::NO,
                'dropdown' => self::YES,
                'calculateinfoot' => self::NO,
                'headername' => 'Tip',
                'sqlcolumnname' => 'ticket_type',
                'width' => 50
            ],
            // FSW VOID/WWMS/Broj naloga
            [
                'type' => 'number',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'FSW VOID/WWMS/Broj naloga',
                'sqlcolumnname' => 'ticket_number',
                'width' => 75
            ],
            // Povezani nalozi
            [
                'type' => 'number',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Povezani nalozi',
                'sqlcolumnname' => 'ticket_connumb',
                'width' => 75
            ],
            // Adresa
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Adresa',
                'sqlcolumnname' => 'ticket_address',
                'width' => 200
            ],
            // Kontakt
            [
                'type' => 'text',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Kontakt',
                'sqlcolumnname' => 'ticket_contact',
                'width' => 100
            ],
            // Prioritet
            [
                'type' => 'text',
                'searchbox' => self::NO,
                'dropdown' => self::YES,
                'calculateinfoot' => self::NO,
                'headername' => 'Prioritet',
                'sqlcolumnname' => 'ticket_priority',
                'width' => 75
            ],
            // Dogovor
            [
                'type' => 'datetime',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Dogovor',
                'sqlcolumnname' => 'ticket_arrdate',
                'width' => 100
            ],
            // Info
            [
                'type' => 'text',
                'searchbox' => self::YES,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Info',
                'sqlcolumnname' => 'ticket_comment',
                'width' => 250
            ],
            // Zadatak preuzet
            [
                'type' => 'datetime',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Zadatak preuzet',
                'sqlcolumnname' => 'ticket_taken',
                'width' => 100
            ],
            // Zadatak završen
            [
                'type' => 'datetime',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Zadatak završen',
                'sqlcolumnname' => 'ticket_resolved',
                'width' => 100
            ],
            // Trajanje zadatka
            [
                'type' => 'number',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Trajanje zadatka',
                'sqlcolumnname' => 'ticket_h',
                'width' => 50
            ],
            // Rad (€)
            [
                'type' => 'currency',
                'searchbox' => self::NO,
                'dropdown' => self::NO,
                'calculateinfoot' => self::NO,
                'headername' => 'Rad (€)',
                'sqlcolumnname' => 'ticket_work',
                'width' => 50,
                'render' => 'currency'
            ],
            // Status
            [
                'type' => 'text',
                'searchbox' => self::NO,
                'dropdown' => self::YES,
                'calculateinfoot' => self::NO,
                'headername' => 'Status',
                'sqlcolumnname' => 'ticket_vrsta',
                'width' => 100
            ],
            // Djelatnici
            [
                'type' => 'text',
                'searchbox' => self::NO,
                'dropdown' => self::YES,
                'calculateinfoot' => self::NO,
                'headername' => 'Djelatnici',
                'sqlcolumnname' => 'ticket_osoba',
                'width' => 150
            ]
        ];

        return array_merge($tableConfig, ['columns' => $columns]);
    }

    public function edit(Request $request): Response 
    {
        try {
            $korisnik = $this->authentication->getUser();
            $vrste = $this->vrsteTable->findAll();
            $osobe = $this->osobeTable->find('vidljivost', '1');
            
            $zadatak = null;
            $editToken = null;
            $selectedVrste = [];
            $selectedOsobe = [];
            
            $id = $request->getParam('id');
            if ($id) {
                $zadatak = $this->zadatciTable->findById($id);
                if ($zadatak) {
                    $editToken = $this->createEditToken($zadatak->id, 'zadatak');
                    
                    // Fetch selected vrste for this zadatak
                    $selectedVrsteData = $this->zadatciTable->findVrsteByZadatak($zadatak->id);
                    $selectedVrste = array_column($selectedVrsteData, 'id');

                    // Fetch selected osobe for this zadatak
                    $selectedOsobeData = $this->zadatciTable->findOsobeByZadatak($zadatak->id);
                    $selectedOsobe = array_column($selectedOsobeData, 'id');
                }
            }

            return $this->response()->render(
                'tasks/resources/views/tasks/edit.html.php',
                [
                    'zadatak' => $zadatak,
                    'edit_token' => $editToken,
                    'user' => $korisnik,
                    'vrste' => $vrste,
                    'osobe' => $osobe,
                    'selected_vrste' => $selectedVrste,
                    'selected_osobe' => $selectedOsobe
                ],
                'Uredi zadatak'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading task edit form', [
                'message' => $e->getMessage(),
                'id' => $id ?? null
            ]);
            return $this->response()->renderError('Failed to load task edit form');
        }
    }

    public function saveEdit(Request $request): Response 
    {
        try {
            // 1. DOHVAĆANJE OSNOVNIH PODATAKA
            $editToken = $request->getPost('edit_token');
            $zadatak = $request->getPost('zadatak');
            $taskId = null;
            $validToken = null;
            
            // 2. VALIDACIJA TOKENA (AKO POSTOJI)
            if ($editToken) {
                $tokenResult = $this->validateEditTokenForValidation($request, 'zadatak');
                if (isset($tokenResult['error'])) {
                    $this->logger->security('Edit token validation failed during save', [
                        'error' => $tokenResult['error'],
                        'user_id' => $this->authentication->getUser()->id ?? null,
                        'ip' => $request->getIp()
                    ]);
                    
                    return Response::redirect('tasks/edit' . (isset($zadatak['id']) ? '?id=' . $zadatak['id'] : ''));
                }
                
                $taskId = $tokenResult['id'];
                $validToken = $tokenResult['token'];
                $zadatak['id'] = $taskId;
            } else {
                $taskId = $zadatak['id'] ?? null;
            }
            
            // 3. VALIDACIJA PODATAKA FORME
            $rules = [
                'zadatak[ticket_type]' => ['required', 'alpha_num'],
                'zadatak[ticket_number]' => ['required', 'numeric'],
                'zadatak[ticket_connumb]' => ['optional', 'numeric'],
                'zadatak[ticket_address]' => ['required', 'plain_text'],
                'zadatak[ticket_contact]' => ['optional', 'plain_text'],
                'zadatak[ticket_priority]' => ['optional', 'alpha_spaces'],
                'zadatak[ticket_arrdate]' => ['optional', 'datetime'],
                'zadatak[ticket_comment]' => ['optional', 'comment'],
                'zadatak[ticket_sharelink]' => ['optional', 'url'],
                'zadatak[ticket_taken]' => ['optional', 'datetime'],
                'zadatak[ticket_resolved]' => ['optional', 'datetime'],
                'zadatak[ticket_work]' => ['optional', 'currency'],
                'vrsta' => ['required', 'array'],
                'osoba' => ['required', 'array']
            ];

            $validation = $request->validate($rules);
            if ($validation->fails()) {
                $selectedVrste = $request->getPost('vrsta') ?? [];
                $selectedOsobe = $request->getPost('osoba') ?? [];

                $newEditToken = null;
                if ($taskId) {
                    if ($validToken) {
                        $newEditToken = $this->refreshEditToken($validToken, 'zadatak');
                    } else {
                        $newEditToken = $this->createEditToken($taskId, 'zadatak');
                    }
                }

                $this->logger->debug('Form validation failed, returning to edit form', [
                    'task_id' => $taskId,
                    'errors_count' => count($validation->errors()),
                    'new_token_created' => !empty($newEditToken)
                ]);

                return $this->renderTaskForm(
                    [
                        'zadatak' => array_merge(
                            $zadatak, 
                            [
                                'selected_vrste' => $selectedVrste,
                                'selected_osobe' => $selectedOsobe
                            ]
                        ),
                        'edit_token' => $newEditToken
                    ],
                    $validation->errors()
                );
            }

            // 4. USPJEŠNA VALIDACIJA - troši token
            if ($validToken) {
                $this->consumeEditToken($validToken);
            }

            // 5. PRIPREMA PODATAKA ZA SPREMANJE
            $korisnik = $this->authentication->getUser();
            $zadatak['user_id'] = $korisnik->id;
            $zadatak['ticket_date'] = new \DateTime(\baseKRIZAN\Config\Config::get('defaulttimezone'));

            // 6. PROCESIRANJE SPECIFIČNIH POLJA
            $this->processDateFields($zadatak);
            $this->processAutomaticFields($zadatak);
            $this->processVrstaAndOsoba($zadatak, $request);
            
            // 7. SPREMANJE PODATAKA
            $zadatakEntity = $this->zadatciTable->save($zadatak);
            $this->saveVrsteAndOsobe($zadatakEntity, $request);

            $this->logger->info('Task saved successfully', [
                'task_id' => $zadatakEntity->id,
                'user_id' => $korisnik->id
            ]);

            // 8. PREUSMJERAVANJE
            return Response::redirect('tasks/list');
            
        } catch (\Throwable $e) {
            $this->logger->error('Error saving task', [
                'message' => $e->getMessage(),
                'zadatak' => $zadatak ?? null,
                'user_id' => $this->authentication->getUser()->id ?? null
            ]);
            return $this->response()->renderError('Failed to save task');
        }
    }

    public function list(): Response 
    {
        try {
            return $this->response()->render(
                'tasks/resources/views/tasks/list.html.php',
                [
                    'title' => 'Svi zadaci'
                ],
                'Svi zadaci'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading tasks list', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load tasks list');
        }
    }

    public function fetchtasks(): Response 
    {
        try {
            $requestData = $_POST;
            
            // 1. KORISTI INTERNI MAPPING
            $tasksMapping = $this->getDataTablesMapping();
            
            // 2. PRIPREMI KOLUMNE ZA DatatablesSSP
            $columns = [];
            foreach ($tasksMapping['columns'] as $index => $column) {
                $columns[] = [
                    'db' => $column['sqlcolumnname'],
                    'dt' => $index
                ];
            }

            // 3. POZOVI DatatablesSSP DIREKTNO (tasks ne trebaju WHERE uvjete za uloge)
            $data = DatatablesSSP::complexWithUniques(
                $requestData,
                $this->dbConnection,
                $tasksMapping['table_name'],
                $tasksMapping['primary_key'],
                $columns,
                $tasksMapping['columns'],
                null // Nema WHERE uvjeta za tasks
            );

            // 4. DODAJ KONFIGURACIJU NA PRVI ZAHTJEV
            if (isset($requestData['init']) && $requestData['init'] === 'true') {
                $data['config'] = [
                    'columns' => $tasksMapping['columns'],
                    'fixedColumns' => $tasksMapping['fixed_columns'],
                    'routeName' => $tasksMapping['route_name']
                ];
                
                if (isset($tasksMapping['rowStyling'])) {
                    $data['config']['rowStyling'] = $tasksMapping['rowStyling'];
                }
            }
            
            echo json_encode($data);
            exit;
        } catch (\Throwable $e) {
            $this->logger->error('Error fetching tasks data', [
                'message' => $e->getMessage()
            ]);
            return Response::json(['error' => 'Failed to fetch tasks data'], 500);
        }
    }

    public function export(Request $request): Response 
    {
        try {
            $tasks = $this->zadatciTable->findAll();
            
            $columns = [
                'id', 'user_id', 'ticket_date', 'ticket_type', 
                'ticket_number', 'ticket_connumb', 'ticket_address', 
                'ticket_contact', 'ticket_priority', 'ticket_arrdate', 
                'ticket_comment', 'ticket_taken', 'ticket_resolved', 
                'ticket_h', 'ticket_day', 'ticket_month',  
                'ticket_year', 'ticket_work', 'ticket_sharelink', 
                'ticket_vrsta', 'ticket_osoba', 'ticket_brojdj', 
            ];
            
            $exportData = $this->exportTasks($tasks, $columns, "svi zadaci");
            
            return Response::download(
                $exportData['content'],
                $exportData['filename'],
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Export failed', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id', []),
                'ip' => $request->getIp()
            ]);
            return $this->response()->renderError('Failed to export users');
        }
    }

    public function delete(Request $request): Response 
    {
        try {
            $korisnik = $this->authentication->getUser();
            $zadatakId = $request->getPost('id');
            $zadatak = $this->zadatciTable->findById($zadatakId);
            
            if (!$zadatak) {
                $this->logger->security('Attempt to delete non-existent task', [
                    'id' => $zadatakId,
                    'user_id' => $korisnik->id,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Record not found');
            }

            if ($zadatak->user_id != $korisnik->id && 
                !$korisnik->hasPermission(\Models\Entity\Korisnik::KORISNIK_MASTER)) {
                $this->logger->security('Unauthorized task deletion attempt', [
                    'task_id' => $zadatakId,
                    'user_id' => $korisnik->id,
                    'ip' => $request->getIp()
                ]);
                return $this->renderTaskForm(
                    [],
                    ['error' => ['Nemate potrebna prava za ovu akciju']]
                );
            }

            $this->zadatciTable->delete($zadatak->id);

            return Response::redirect('tasks/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error deleting task', [
                'message' => $e->getMessage(),
                'id' => $zadatakId ?? null
            ]);
            return $this->response()->renderError('Failed to delete task');
        }
    }
    
    public function admin(): Response 
    {
        try {
            $statistics = [
                'taskTypesCount' => $this->vrsteTable->getTaskTypesCount(),
                'usersCount' => $this->osobeTable->getUsersCount(),
                'completedTasksCount' => $this->zadatciTable->getCompletedTasksCount(),
                'inProgressTasksCount' => $this->zadatciTable->getInProgressTasksCount(),
                'tasksByStatus' => $this->zadatciTable->getTasksByStatus(),
                'tasksByType' => $this->zadatciTable->getTasksByType(),
                'topPerformers' => $this->zadatciTable->getTopPerformers(),
                'monthlyTaskCounts' => $this->zadatciTable->getMonthlyTaskCounts(),
                'totalRevenue' => $this->zadatciTable->getTotalRevenue(),
                'averageResolutionTime' => $this->zadatciTable->getAverageResolutionTime(),
                'tasksByPriority' => $this->zadatciTable->getTasksByPriority(),
                'recentTasks' => $this->zadatciTable->getRecentTasks()
            ];
            
            return $this->response()->render(
                'tasks/resources/views/tasks/admin.html.php',
                [
                    'title' => 'Admin',
                    'statistics' => $statistics
                ],
                'Admin'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading admin dashboard', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load admin dashboard');
        }
    }
    
    public function getAdminStats(): Response
    {
        try {
            $statistics = [
                'taskTypesCount' => $this->vrsteTable->getTaskTypesCount(),
                'usersCount' => $this->osobeTable->getUsersCount(),
                'completedTasksCount' => $this->zadatciTable->getCompletedTasksCount(),
                'inProgressTasksCount' => $this->zadatciTable->getInProgressTasksCount(),
                'tasksByStatus' => $this->zadatciTable->getTasksByStatus(),
                'tasksByType' => $this->zadatciTable->getTasksByType(),
                'topPerformers' => $this->zadatciTable->getTopPerformers(),
                'monthlyTaskCounts' => $this->zadatciTable->getMonthlyTaskCounts(),
                'totalRevenue' => $this->zadatciTable->getTotalRevenue(),
                'averageResolutionTime' => $this->zadatciTable->getAverageResolutionTime(),
                'tasksByPriority' => $this->zadatciTable->getTasksByPriority()
            ];
            
            return Response::json($statistics);
        } catch (\Throwable $e) {
            $this->logger->error('Error fetching admin stats', [
                'message' => $e->getMessage()
            ]);
            return Response::json(['error' => 'Failed to fetch admin stats'], 500);
        }
    }

    // HELPER METHODS
    private function processDateFields(array &$zadatak): void 
    {
        require_once APP_ROOT . '/moduli/tasks/Helpers/convertDateToEnglishFormat.php';
        
        $dateFields = ['ticket_arrdate', 'ticket_taken', 'ticket_resolved'];
        foreach ($dateFields as $field) {
            if (isset($zadatak[$field])) {
                $zadatak[$field] = convertDateTimeToEnglishFormat($zadatak[$field]);
            }
        }
    }

    private function processAutomaticFields(array &$zadatak): void 
    {
        if (!empty($zadatak['ticket_taken']) && !empty($zadatak['ticket_resolved'])) {
            $d1 = new \DateTime($zadatak['ticket_taken'], new \DateTimeZone(\baseKRIZAN\Config\Config::get('defaulttimezone')));
            $d2 = new \DateTime($zadatak['ticket_resolved'], new \DateTimeZone(\baseKRIZAN\Config\Config::get('defaulttimezone')));
    
            $interval = $d1->diff($d2);
            $zadatak['ticket_h'] = ($interval->days * 24) + $interval->h + (($interval->i) / 60);
            $zadatak['ticket_day'] = $d2->format('d');
            $zadatak['ticket_month'] = $d2->format('m');
            $zadatak['ticket_year'] = $d2->format('Y');
        } else {
            $zadatak['ticket_h'] = 0;
            $zadatak['ticket_day'] = null;
            $zadatak['ticket_month'] = null;
            $zadatak['ticket_year'] = null;
        }
    }

    private function processVrstaAndOsoba(array &$zadatak, Request $request): void 
    {
        if (!empty($request->getPost('vrsta')) && !empty($request->getPost('osoba'))) {
            $zadatak['ticket_osoba'] = $this->processEntries($this->osobeTable, $request->getPost('osoba'));
            $zadatak['ticket_brojdj'] = count($request->getPost('osoba'));
            $zadatak['ticket_vrsta'] = $this->processEntries($this->vrsteTable, $request->getPost('vrsta'));
        }
    }

    private function processEntries(DatabaseTableTasks $table, array $entries): string 
    {
        return implode(", ", array_map(
            fn($id) => $table->findById($id)->name,
            $entries
        ));
    }

    private function saveVrsteAndOsobe($zadatakEntity, Request $request): void 
    {
        $zadatakEntity->clearVrste();
        $zadatakEntity->clearOsobe();

        foreach ($request->getPost('vrsta') as $vrstaId) {
            $zadatakEntity->addVrsta($vrstaId);
        }

        foreach ($request->getPost('osoba') as $osobaId) {
            $zadatakEntity->addOsoba($osobaId);
        }
    }

    private function renderTaskForm(array $data, array $errors): Response 
    {
        $zadatak = null;
        $selectedVrste = [];
        $selectedOsobe = [];

        if (isset($data['zadatak'])) {
            $zadatak = is_array($data['zadatak']) 
                ? (object)$data['zadatak'] 
                : $data['zadatak'];

            if (isset($zadatak->ticket_work) && is_string($zadatak->ticket_work)) {
                $zadatak->ticket_work = floatval(str_replace(',', '.', $zadatak->ticket_work));
            }

            $selectedVrste = $data['zadatak']['selected_vrste'] ?? [];
            $selectedOsobe = $data['zadatak']['selected_osobe'] ?? [];
        }

        return $this->response()->render(
            'tasks/resources/views/tasks/edit.html.php',
            [
                'errors' => $errors,
                'zadatak' => $zadatak,
                'edit_token' => $data['edit_token'] ?? null,
                'user' => $this->authentication->getUser(),
                'vrste' => $this->vrsteTable->findAll(),
                'osobe' => $this->osobeTable->find('vidljivost', '1'),
                'selected_vrste' => $selectedVrste,
                'selected_osobe' => $selectedOsobe
            ],
            'Uredi zadatak'
        );
    }

    private function exportTasks(array $tasks, array $columns, string $filename): array 
    {
        try {
            $currentUser = $this->authentication->getUser();
            if (!$currentUser || $currentUser->user_permissions < Korisnik::KORISNIK_MASTER) {
                throw new \RuntimeException('Unauthorized to export users');
            }
    
            if (count($tasks) > self::MAX_EXPORT_ROWS) {
                throw new \RuntimeException(
                    sprintf('Export limit exceeded: %d rows', self::MAX_EXPORT_ROWS)
                );
            }
    
            require_once APP_ROOT . '/app/Helpers/excelexport.php';
            $savefileas = sprintf(
                "%s - %s tjedan %s",
                $filename,
                date('d-m-Y'),
                date("W", strtotime(date('d-m-Y')))
            );
    
            return excelexport('tasks', $columns, $savefileas, $tasks);
        } catch (\Throwable $e) {
            $this->logger->error('Export failed', [
                'message' => $e->getMessage(),
                'rows_count' => count($tasks)
            ]);
            throw $e;
        }
    }
}